# -*- coding: utf-8 -*-

import urllib

from collections import namedtuple, defaultdict

from mpfs.common import errors
from mpfs.common.util import from_json
from mpfs.config import settings
from mpfs.core.services.common_service import Service


YATEAM_FORCE_ENABLED_UIDS = settings.yateam['force_enable_uids']


StaffUserRecord = namedtuple('StaffUserRecord', [
    'yateam_uid',
    'yateam_login',
    'uid',
    'login',
    'chief_yateam_uid',
    'is_dismissed',
    'affiliation',
    'is_homeworker',
])


class StaffService(Service):
    name = 'staff'

    @staticmethod
    def __to_record(staff_record):
        return StaffUserRecord(
            yateam_uid=staff_record['internal_uid'],
            yateam_login=staff_record['internal_login'],
            chief_yateam_uid=staff_record['chief_uid'],
            uid=staff_record.get('external_uid'),
            login=staff_record.get('external_login'),
            is_dismissed=staff_record['is_dismissed'],
            affiliation=staff_record['affiliation'],
            is_homeworker=staff_record['is_homeworker'],
        )

    def __get_user_infos(self, yateam_uid_or_uids=None):
        url = self.base_url + 'login_links/export/'
        if yateam_uid_or_uids:
            query = {'internal_uid': yateam_uid_or_uids}
            url = url + '?' + urllib.urlencode(query, doseq=True)
        response = self.open_url(url)
        result = self.__get_users_info_with_forced_yateam(yateam_uid_or_uids)
        login_to_record_dict = from_json(response)
        for v in login_to_record_dict.itervalues():
            result[v['internal_uid']] = StaffService.__to_record(v)
        return result.values()

    def __get_users_info_with_forced_yateam(self, yateam_uid_or_uids=None):
        if yateam_uid_or_uids is None:
            yateam_uid_or_uids = YATEAM_FORCE_ENABLED_UIDS
        elif not isinstance(yateam_uid_or_uids, list):
            yateam_uid_or_uids = [yateam_uid_or_uids]

        records = {}
        for uid in yateam_uid_or_uids:
            if uid not in YATEAM_FORCE_ENABLED_UIDS:
                continue
            record = defaultdict(str)
            record['internal_uid'] = '0'
            record['external_uid'] = uid
            records[uid] = StaffService.__to_record(record)
        return records

    def get_all_user_infos(self):
        return self.__get_user_infos()

    def get_user_infos(self, yateam_uids):
        return self.__get_user_infos(yateam_uids)

    def get_user_info(self, yateam_uid):
        response = self.__get_user_infos(yateam_uid)
        if len(response) == 0:
            raise errors.StaffUserNotFound(yateam_uid)
        if len(response) > 1:
            raise errors.StaffMoreUsersThanExpected(yateam_uid)
        return response[0]
