# -*- coding: utf-8 -*-
import collections
import urllib
import urlparse

from collections import defaultdict
from copy import copy
from inspect import isclass
from types import ModuleType

import mpfs.engine.process
from mpfs.common import errors
from mpfs.common.util import from_json, to_json, normalize_unicode
from mpfs.core.services.common_service import Service
from mpfs.platform.utils import model_name_for_serializer


__all__ = ['TankerService', 'TankerHelper']


error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('tanker')


class TankerService(Service):
    """
    Сервис для общения с API танкера

    Документация по API: http://doc.yandex-team.ru/Tanker/api-reference/concepts/about.xml
    """
    name = 'tanker'
    # api_error = errors.TankerNoResponse
    # log = service_log
    base_url = None  # should be set in config
    timeout = None  # should be set in config

    _keyset_cache = defaultdict(dict)
    """Кэш кейсэтов.
    Структура: {<project_id>: {<keyset_id>: {<lang>: {<key>: <value>, ...}, ...}, ...}, ...}
    """

    @classmethod
    def clear_cache(cls):
        cls._keyset_cache = defaultdict(dict)

    def get_keyset(self, project_id, keyset_id):
        """
        Возвращает dict содержащий кейсет

        Описание структуры возвращаемых данных:
        http://doc.yandex-team.ru/Tanker/api-reference/concepts/formats.xml#format-json
        """
        keyset = self._keyset_cache[project_id].get(keyset_id, None)
        if not keyset:
            try:
                keyset = self._load_keyset(project_id, keyset_id)
            except Exception, e:
                error_log.error('Tanker error loading keyset "%s" for project "%s": %s' % (keyset_id, project_id, e))
                keyset = {}
        return keyset

    def upload_keyset(self, project_id, keyset_id, **keysets):
        """
        Загружает кейсэты в танкер

        :param project_id:
        :param keyset_id:
        :param keysets: {<language, e.g. 'ru'>: {<key>: <value>, ...}, ...}
        :return:
        """
        keys = defaultdict(lambda: defaultdict(defaultdict))
        for lang, keyset in keysets.iteritems():
            for key, value in keyset.iteritems():
                keys[key]['info'] = {
                    'context': '',
                    'is_plural': False,
                    'references': '',
                }
                keys[key]['translations'][lang] = {
                    'form': value,
                    'status': 'approved',
                    'translator_comment': '',
                    'change_date': '',
                    'author': 'mpfs',
                }

        tjson = {
            'export_info': {
                'request': {
                    'project_id': project_id,
                    'keyset_id': keyset_id,
                },
                'name': project_id,
                'branch': 'master',
            },
            'keysets': {
                keyset_id: {
                    'keys': keys,
                }
            }
        }
        import json
        print json.dumps(tjson, indent=4)
        # self._request(
        #     'POST', '/keysets/merge/',
        #     query={'project-id': project_id, 'keyset-id': keyset_id},
        #     data=to_json(tjson),
        #     headers={'AUTHORIZATION': ''}
        # )


    def _request(self, method, url, query=None, data=None, headers=None):
        """Выполняет запрос к сервису"""
        headers = headers or {}
        absolute_url = urlparse.urljoin(self.base_url, url)
        if isinstance(query, dict):
            qs = urllib.urlencode(query)
            absolute_url = '%s?%s' % (absolute_url, qs)
        post = {}
        pure_data = None
        if isinstance(data, dict):
            post = data
        else:
            pure_data = data
        return self.open_url(absolute_url, method=method, post=post, pure_data=pure_data, headers=headers)

    def _load_keyset(self, project_id, keyset_id):
        """Скичивает кейсет из танкера и помещает его в кэш"""
        resp = self._request('GET', '/keysets/json/', query={'project-id': project_id, 'keyset-id': keyset_id})
        keyset = from_json(resp)
        self._keyset_cache[project_id][keyset_id] = keyset
        return keyset


##############################################################################
#  BEGIN of Key Builders
##############################################################################


class I18nKeyBuilder(object):
    """
    Конструктор ключей танкера для различных объектов
    """
    obj_type = None
    supported_keys = []

    @classmethod
    def get_obj_type(cls):
        return cls.obj_type

    @classmethod
    def build(cls, obj, key):
        raise NotImplemented()

    @classmethod
    def is_acceptable(cls, obj):
        return isinstance(obj, cls.get_obj_type())

    @classmethod
    def get(cls, obj, key):
        if not cls.is_acceptable(obj):
            raise ValueError('obj must be instance of the following type: %s' % cls.get_obj_type().__name__)
        if key not in cls.supported_keys:
            raise ValueError('key value can be one of the following: %s' % ', '.join(cls.supported_keys))
        return cls.build(obj, key)


class I18nApiModuleKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для модулей API"""
    obj_type = ModuleType
    supported_keys = ['title']

    @classmethod
    def build(cls, obj, key):
        name_chunks = obj.__name__.split('.')
        version = name_chunks[-3]
        name = name_chunks[-2]
        return 'api-%s-%s-%s' % (name, version, key)


class I18nApiResourceKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для модулей API"""
    supported_keys = ['title']

    @classmethod
    def get_obj_type(cls):
        from mpfs.platform.resources import BaseResource
        return BaseResource

    @classmethod
    def build(cls, obj, key):
        return 'resource-%s-%s' % (obj.path, key)


class I18nHandlerKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для хэндлеров"""
    supported_keys = ['title', 'description']

    @classmethod
    def get_obj_type(cls):
        from mpfs.platform.handlers import BasePlatformHandler
        return BasePlatformHandler

    @classmethod
    def build(cls, obj, key):
        return 'handler-%s-%s-%s' % (obj.method, obj.path, key)


class I18nHandlerQueryParameterKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для querystring параметров хэндлеров"""
    supported_keys = ['title']

    @classmethod
    def get_obj_type(cls):
        from mpfs.platform.fields import BaseField
        return BaseField

    @classmethod
    def is_acceptable(cls, obj):
        is_acceptable = super(I18nHandlerQueryParameterKeyBuilder, cls).is_acceptable(obj)
        if is_acceptable:
            from mpfs.platform.fields import QueryDict
            return isinstance(obj.parent, QueryDict) and obj.parent.parent
        return is_acceptable

    @classmethod
    def build(cls, obj, key):
        http_method = obj.parent.parent.method
        path = obj.parent.parent.path
        name = obj.name
        return 'query-param-%s-%s-%s-%s' % (http_method, path, name, key)


class I18nSerializerFieldKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для полей сериализаторов"""
    supported_keys = ['title']

    @classmethod
    def get_obj_type(cls):
        from mpfs.platform.fields import BaseField
        return BaseField

    @classmethod
    def is_acceptable(cls, obj):
        is_acceptable = super(I18nSerializerFieldKeyBuilder, cls).is_acceptable(obj)
        if is_acceptable:
            from mpfs.platform.serializers import BaseSerializer
            return isclass(obj.parent) and issubclass(obj.parent, BaseSerializer)
        return is_acceptable

    @classmethod
    def build(cls, obj, key):
        model_name = model_name_for_serializer(obj.parent)
        return 'model-attr-%s-%s-%s' % (model_name, obj.name, key)


class I18nPermissionKeyBuilder(I18nKeyBuilder):
    """Создаёт ключи для разрешений"""
    supported_keys = ['title']

    @classmethod
    def get_obj_type(cls):
        from mpfs.platform.permissions import ClientHasScopesPermission
        return ClientHasScopesPermission

    @classmethod
    def build(cls, obj, key):
        return 'permission-%s-%s' % ('-'.join(obj.scopes), key)


##############################################################################
#  END of Key Builders
##############################################################################


class TankerHelper(collections.Mapping):
    """
    Кейсэт танкера для указанного языка

    Представляет собой удобный прокси для получения локализованного значения ключей без необходимости 100500 раз
    передавать явно язык, project_id и keyset_id.

    Пример:
    >>> I18N_PROJECT_ID = 'some_project_id'
    >>> I18N_KEYSET_ID = 'some_keyset_id'
    >>> keyset = TankerHelper(I18N_PROJECT_ID, I18N_KEYSET_ID, 'en')
    >>> keyset['translated_key']
    Translated
    >>> keyset.lang = 'ru'
    >>> keyset['translated_key']
    Переведено
    >>> keyset['unexistent_key'] == None
    True
    """
    KEY_BUILDERS = [I18nApiModuleKeyBuilder, I18nHandlerKeyBuilder, I18nHandlerQueryParameterKeyBuilder,
                    I18nSerializerFieldKeyBuilder, I18nPermissionKeyBuilder]

    def __init__(self, project_id, keyset_id, lang):
        self.project_id = project_id
        self.keyset_id = keyset_id
        self.lang = lang
        # self.service = TankerService()
        self._set_items = {}

    def get_keyset(self):
        # https://jira.yandex-team.ru/browse/CHEMODAN-16956
        # return self.service.get_keyset(self.project_id, self.keyset_id).get(self.lang, {})
        return {}

    def build_key_for_object(self, obj, key):
        for kb in self.KEY_BUILDERS:
            if kb.is_acceptable(obj):
                return kb.get(obj, key)

    def get_key_for_object(self, obj, key):
        return self.get(self.build_key_for_object(obj, key), None)

    def save(self):
        updated = {}
        for key, val in self._set_items.iteritems():
            tanker_val = self.get_keyset().get(key, None)
            if tanker_val:
                tanker_val = normalize_unicode(tanker_val)
            if val is not None:
                val = normalize_unicode(val)
            if val != tanker_val:
                updated[key] = val
        # self.service.upload_keyset(self.project_id, self.keyset_id, **{self.lang: updated})

    def __getitem__(self, key):
        return self._set_items.get(key, self.get_keyset().get(key, None))

    def __setitem__(self, key, value):
        self._set_items[key] = value

    def __len__(self):
        return len(set(self.get_keyset().keys()) | set(self._set_items.keys()))

    def __iter__(self):
        keyset = copy(self.get_keyset())
        keyset.update(self._set_items)
        return iter(keyset)
