# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис API Телемоста

"""
import urllib

from urllib2 import HTTPError

import re

from requests import RequestException

import mpfs.engine.process
import mpfs.common.errors as errors
from mpfs.common.util import from_json, filter_uid_by_percentage
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.core.services.passport_service import passport
from mpfs.core.services.telemost_conference_manager_service import tcm
from mpfs.platform.utils.uid_formatters import is_yateam_uid
from mpfs.config import settings


default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('telemost-api')
use_separate_telemost_cluster_for_ya_team = settings.feature_toggles['use_separate_telemost_cluster_for_ya_team']
SEPARATE_TELEMOST_CLUSTER_FOR_YT_TEAM = settings.feature_toggles['separate_telemost_cluster_for_ya_team']
SEPARATE_TELEMOST_CLUSTERS = settings.feature_toggles['separate_telemost_clusters']
SEPARATE_TELEMOST_CLUSTERS_CONF_JOIN_SHORT_URL_RE = re.compile(SEPARATE_TELEMOST_CLUSTERS['conf_join_short_url_re'])


# helpers for quick implementation of Telemost-YT
def is_separate_yt_telemost_enabled_for(uid):
    return (uid in SEPARATE_TELEMOST_CLUSTER_FOR_YT_TEAM['enable_for_uids'] or
            filter_uid_by_percentage(uid, SEPARATE_TELEMOST_CLUSTER_FOR_YT_TEAM['percentage']))


def _has_staff(uid):
    if uid is None:
        return False
    try:
        user_info = passport.userinfo(uid)
        return user_info.get('has_staff')
    except Exception:
        return False


def _parse_conference_uri(uri):
    match = SEPARATE_TELEMOST_CLUSTERS_CONF_JOIN_SHORT_URL_RE.match(uri)
    conf_key = None
    if match:
        conf_key = match.group('conf_key')
        conf_key = urllib.unquote(conf_key)
    return conf_key
# end of helpers


class TelemostApiService(RequestsPoweredServiceBase):

    def __init__(self):
        super(TelemostApiService, self).__init__()
        self.send_cloud_request_id = True

    def create_conference(self, uid=None, staff_only=None, is_permanent=None, external_meeting=None, calendar_event_id=None):
        parameters = {}
        if staff_only is not None:
            parameters['staff_only'] = staff_only
        if is_permanent is not None:
            parameters['is_permanent'] = is_permanent
        if uid:
            parameters['uid'] = uid
        if external_meeting is not None:
            parameters['external_meeting'] = external_meeting
        if calendar_event_id is not None:
            parameters['calendar_event_id'] = calendar_event_id

        response = self.request('POST', '/v1/conferences', parameters)
        if response.status_code == 201:
            return from_json(response.content)
        raise errors.TelemostApiBadResponse

    def get_conference(self, conference_uri, uid=None):
        parameters = {}
        if uid:
            parameters['uid'] = uid
        path = '/v1/conferences/%s/connection' % urllib.quote(conference_uri, '')
        response = self.request('GET', path, parameters)
        status_code = response.status_code
        if status_code == 200:
            return from_json(response.content)
        raise errors.TelemostApiBadResponse()

    def _exception_from_response(self, response):
        status_code = response.status_code
        json = from_json(response.content)
        code = status_code
        if isinstance(json, dict):
            code = json.get('error', {}).get('name', str(status_code))
        if status_code == 404 and json.get('error', {}).get('name') == 'NoSuchUserInConference':
            return errors.TelemostApiNoSuchUserInConference(code)
        if status_code == 404:
            return errors.TelemostApiNotFound(code)
        if status_code == 403 and json.get('error', {}).get('name') == 'ConferenceLinkNotCome':
            return errors.TelemostApiNotCome(code, from_json(json['error']['message']))
        if status_code == 403 and json.get('error', {}).get('name') == 'StreamNotStarted':
            return errors.TelemostApiStreamNotStarted(code)
        if status_code == 403 and json.get('error', {}).get('name') == 'NoSuchBroadcastCreated':
            return errors.TelemostApiNoSuchBroadcastCreated(code)
        if status_code == 403 and json.get('error', {}).get('name') == 'InvalidTranslatorToken':
            return errors.TelemostApiInvalidTranslatorToken(code)
        if status_code == 403 and json.get('error', {}).get('name') == 'ForbiddenAccessToCreateBroadcast':
            return errors.TelemostApiForbiddenAccessToCreateBroadcast(code)
        if status_code == 403 and json.get('error', {}).get('name') == 'ForbiddenAccessToStartBroadcast':
            return errors.TelemostApiForbiddenAccessToStartBroadcast(code)
        if status_code == 403:
            return errors.TelemostApiForbidden(code)
        if status_code == 400:
            return errors.TelemostApiBadRequest(code)
        if status_code == 410 and json.get('error', {}).get('name') == 'BroadcastLinkExpired':
            return errors.TelemostApiBroadcastLinkExpired(code)
        if status_code == 410:
            return errors.TelemostApiGone(code)
        if status_code == 409:
            return errors.TelemostApiConflict(code)
        if status_code == 429 and json.get('error', {}).get('name') == 'TelemostOverload':
            return errors.TelemostOverload(code)
        return errors.TelemostApiBadResponse(code)


class TelemostYaTeamApiService(TelemostApiService):
    def __init__(self):
        super(TelemostYaTeamApiService, self).__init__()


class TemplatedTelemostService(TelemostApiService):
    def __init__(self, telemost_api_host):
        super(TemplatedTelemostService, self).__init__()
        self.base_url = self.base_url % telemost_api_host


telemost_api = TelemostApiService()
telemost_ya_team_api = TelemostYaTeamApiService()


class FanOutToAllTelemostService(object):
    def __init__(self, services):
        self.services = services

    # изначально чтобы включать/выключать фичи в биллинге. не-идемпотентные запросы отправлять запрещается
    # будет сделано 2 ретрая в каждый хост благодаря
    # mpfs/lib/mpfs/core/services/common_service.py:42 (SYSTEM_HTTP_RETRIES)
    def request(self, method, url, *args, **kwargs):
        collected_errors = []
        responses = []
        for service in self.services:
            try:
                responses.append(service.request(method, url, *args, **kwargs))
            except Exception as e:
                error_log.exception('Failed to fan out request %s:%s to service %s. Error is %s'
                                    % (method, url, service, e))
                collected_errors.append(e)
        if len(collected_errors) > 0:
            raise collected_errors[0]

        return responses[0]


# Костыль сделанный вспешке для телемоста
class MultiTelemostRequestsPoweredServiceBase(object):
    def __init__(self, services):
        self.services = services

    def request(self, method, url, *args, **kwargs):
        last_error = None
        for service in self.services:
            try:
                return service.request(method, url, *args, **kwargs)
            except (errors.TelemostApiNotFound, errors.TelemostApiBadRequest, errors.TelemostApiGone) as e:
                last_error = e
            except errors.TelemostApiBadResponse as e:
                # retry requests to Telemost-YT only for unknown errors from specific handlers
                if (isinstance(url, basestring) and
                    any(url.startswith(prefix)
                        for prefix in SEPARATE_TELEMOST_CLUSTER_FOR_YT_TEAM['route_unknown_errors_for'])):
                    last_error = e
                else:
                    raise

        # не получили нормального ответа ни от одного кластера
        raise last_error

    def get_conference(self, *args, **kwargs):
        last_error = None
        for service in self.services:
            try:
                return service.get_conference(*args, **kwargs)
            except errors.TelemostApiError as e:
                last_error = e

        raise last_error


class TelemostViaTCMService(object):
    FALLBACK_SERVICES = (telemost_api, telemost_ya_team_api)

    def __init__(self, services):
        self.services = services

    @staticmethod
    def _is_enabled_for(uid):
        if not SEPARATE_TELEMOST_CLUSTERS['enabled']:
            return False

        # Если 100%, то анонимы смогут заходить в конфу
        if SEPARATE_TELEMOST_CLUSTERS['percentage'] == 100:
            return True

        if uid in SEPARATE_TELEMOST_CLUSTERS['enabled_for']:
            return True

        if uid is None:
            return False

        return filter_uid_by_percentage(uid, SEPARATE_TELEMOST_CLUSTERS['percentage'])

    @classmethod
    def get_telemost_api(cls, uid, uri=None, room_id=None):
        if not cls._is_enabled_for(uid):
            return get_telemost_api(uid)

        short_url_id = None

        if uri:
            short_url_id = _parse_conference_uri(uri)
            if short_url_id is None:
                default_log.info(u'Failed to parse %s for TCM' % uri)

        try:
            info = tcm.get_conference_info(uid, short_url_id=short_url_id, conference_id=room_id)
        except (HTTPError, RequestException):
            default_log.info('Failed to get Telemost API host from TCM, using all services')
            fallback_services = [TemplatedTelemostService(host)
                                 for host in SEPARATE_TELEMOST_CLUSTERS['fallback_hosts']]
            service = MultiTelemostRequestsPoweredServiceBase(services=fallback_services)
        else:
            service = TemplatedTelemostService(info['balancer_host'])

        return service

    @classmethod
    def get_telemost_api_for_general_info(cls, uid):
        u"""Для получения Telemost API в ручках не связанных с конкретной коференцией"""
        if not cls._is_enabled_for(uid):
            return get_telemost_api(uid)

        is_staff = False
        is_yateam = False
        if uid is not None:
            is_yateam = is_yateam_uid(uid)
            is_staff = _has_staff(uid)

        try:
            info = tcm.get_user_info(uid, is_staff=is_staff, is_yandex_team=is_yateam)
        except (HTTPError, RequestException):
            default_log.info('Failed to get Telemost API host from TCM, using default service')
            service = TemplatedTelemostService(
                SEPARATE_TELEMOST_CLUSTERS['fallback_host_on_create']
            )
        else:
            service = TemplatedTelemostService(info['balancer_host'])

        return service

    @classmethod
    def get_telemost_api_for_create(cls, uid):
        if not cls._is_enabled_for(uid):
            return get_telemost_api_for_create(uid)

        is_staff = False
        is_yateam = False
        if uid is not None:
            is_yateam = is_yateam_uid(uid)
            is_staff = _has_staff(uid)

        try:
            info = tcm.get_new_conference_info(uid, is_staff=is_staff, is_yandex_team=is_yateam)
        except (HTTPError, RequestException):
            default_log.info('Failed to get Telemost API host from TCM, using default service')
            service = TemplatedTelemostService(
                SEPARATE_TELEMOST_CLUSTERS['fallback_host_on_create']
            )
        else:
            service = TemplatedTelemostService(info['balancer_host'])

        return service


if use_separate_telemost_cluster_for_ya_team:
    telemost_api_with_fallback = MultiTelemostRequestsPoweredServiceBase([telemost_api, telemost_ya_team_api])
else:
    telemost_api_with_fallback = telemost_api


def get_telemost_api(uid=None):
    if (use_separate_telemost_cluster_for_ya_team and
            is_separate_yt_telemost_enabled_for(uid)):
        return telemost_api_with_fallback

    return telemost_api


def get_telemost_api_for_create(uid=None):
    if (use_separate_telemost_cluster_for_ya_team and
            uid is not None and (is_yateam_uid(uid) or _has_staff(uid)) and
            is_separate_yt_telemost_enabled_for(uid)):
        return telemost_ya_team_api

    return telemost_api
