# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис корзины

"""

import time

import mpfs.engine.process

from mpfs.common.util.experiments.logic import change_experiment_context_with, experiment_manager
from mpfs.core.metastorage.control import trash, disk_info
from mpfs.core.services.disk_service import StorageSpaceLimited, Resources
from mpfs.core.address import Address
from mpfs.common import errors

log = mpfs.engine.process.get_default_log()


class Trash(StorageSpaceLimited):
    name = 'trash'
    control = trash
    space_counter_key = 'trash_size'
    space_counter_groups = ('disk', 'trash',)

    def __init__(self, *args, **kwargs):
        super(Trash, self).__init__(*args, **kwargs)
        from mpfs.core.filesystem.resources.trash import TrashFile, TrashFolder

        self.resources = Resources(dir=TrashFolder, file=TrashFile)

    def put(self, address, val, version=None, **kwargs):
        return self.control.put(address.uid, address.path, val, version, **kwargs)

    def drop(self, uid):
        self.control.remove(uid, '/trash')
        self.control.make_folder(uid, '/trash', {})
        return self.control.folder_content(uid, '/trash').value

    def make_folder(self, address, parent, data):
        """
        Создание папки по родителю
        """

        value = {
            'type': 'dir',
            'name': data['name'],
            'visible': 1,
            'ctime': data['ctime'],
            'mtime': data['mtime'],
            'utime': data['utime'],
            'meta': data.get('meta') or {},
        }
        self.update_value_with_trash_attributes(value, address, data['id'])

        return self.control.make_folder(address.uid, address.path, value)

    @classmethod
    def update_value_with_trash_attributes(cls, value, address, original_id):
        with change_experiment_context_with(uid=address.uid):
            if experiment_manager.is_feature_active('new_trash_drop_and_restore') and address.get_path_depth_lvl() > 2:
                return

            value['original_id'] = original_id
            value['meta']['append_time'] = int(time.time())
            value['meta']['original_parent_id'] = Address(address.uid + ':' + original_id).get_parent().path + '/'

    def get_folder_content(self, resource):
        result = []
        for element in self.control.folder_content(resource.uid, resource.path).value:
            result.append(element.data)
        return result

    def process_children(self, resource, resources):
        result = []
        folders = []
        files = []

        for item in resources:
            if not item.get('type'):
                log.warn('Bad resource \n%s' % '\n'.join('%s = %s' % (str(k), str(v)) for k, v in item.iteritems()))
                continue
            item['meta'] = item.get('meta', {})
            item['meta'].pop('wh_version', '')

            if item['type'] == 'dir':
                item_path = resource.address.get_child_folder(item['name'])
                item['id'] = item_path.path
                item['name'] = item.pop('static_name', '')
                resource.sorted_folders.append(item['id'])
                resource.child_folders[item['id']] = item
                resource.construct_child_folder(item['id'], item, resource.version)
                folders.append(item)

            if item['type'] == 'file':
                item_path = resource.address.get_child_file(item['name'])
                item['id'] = item_path.path
                if 'source' not in item:
                    item['source'] = self.name
                item['name'] = item.pop('static_name', '')
                resource.sorted_files.append(item['id'])
                resource.child_files[item['id']] = item
                resource.construct_child_file(item['id'], item, resource.version)
                files.append(item)
        result.extend(folders)
        result.extend(files)

        return result

    def get_resource(self, uid, address, version=None, **kwargs):
        element = None
        if address.storage_name == 'trash':
            element = super(Trash, self).get_resource(uid, address, version=version, **kwargs)
        if element:
            return element
        else:
            raise errors.ResourceNotFound(address.id)

    def used(self, uid):
        try:
            resp = disk_info.value(uid, 'trash_size')
            size = int(resp.value.data)
        except Exception:
            size = 0
        if size < 0:
            size = 0
        return size

    def is_empty(self, uid):
        if self.used(uid) > 0:
            return 0
        elif trash.count(uid) > 2:
            return 0
        else:
            return 1
