# -*- coding: utf-8 -*-
import time

import mpfs.engine.process

import mpfs.common.errors.billing as errors

from mpfs.common.static.tags.billing import *
from mpfs.common.util import from_json, to_json
from mpfs.core.billing.constants import DEFAULT_BUY_QTY
from mpfs.core.billing.inapp.app_store_parser import app_store_response_parser
from mpfs.core.services.common_service import RequestsPoweredServiceBase

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class BillingOrderInfo(dict):
    """
    Вспомогательный класс-обертка над ответом биллинга на запрос check_order
    """
    @property
    def has_btime(self):
        return SUBS_UNTIL in self

    @property
    def has_lbtime(self):
        return PAYMENT_TS_MSEC in self

    @property
    def btime(self):
        if self.has_btime:
            return int(float(self[SUBS_UNTIL]))

    @property
    def lbtime(self):
        if self.has_lbtime:
            return int(float(self[PAYMENT_TS_MSEC])) // 1000

    @property
    def billing_is_active(self):
        return not bool(self.get(FINISH_TS))


class PaymentInfo(object):
    _refundable_statuses = {CLEARED, AUTHORIZED, SUCCESS}

    def __init__(self, trust_payment_info):
        self._trust_payment_info = trust_payment_info

    @property
    def payment_url(self):
        return self._trust_payment_info.get(PAYMENT_URL)

    @property
    def payment_status(self):
        return self._trust_payment_info.get(PAYMENT_STATUS)

    @property
    def payment_status_code(self):
        return self._trust_payment_info.get(PAYMENT_STATUS_CODE)

    @property
    def is_payment_refundable(self):
        # Для того, чтобы сделать возврат, нужно, чтобы платеж был в одном из статусов:
        # payment_status: {"cleared", "authorized", "success"}. Иначе - вызов создания рефанда вернет ошибку.
        return self.payment_status in self._refundable_statuses


class RefundStatus(object):
    _posible_statuses = {SUCCESS, WAIT_FOR_NOTIFICATION, FAILED, ERROR}
    _error_statuses = {FAILED, ERROR}

    def __init__(self, trust_refund_status):
        self._data = trust_refund_status
        status = self._data.get(STATUS)
        if status not in self._posible_statuses:
            raise errors.BigBillingBadResult('Unknown refund status: %s' % status)
        self._status = status

    @property
    def status(self):
        return self._status

    @property
    def status_description(self):
        return self._data.get(STATUS_DESC)

    @property
    def fiscal_receipt_url(self):
        return self._data.get(FISCAL_RECEIPT_URL)

    @property
    def is_error(self):
        return self.status in self._error_statuses

    @property
    def can_retry(self):
        # Рефанд можно ретраить (создавать новый после неуспешной попытки) если в поле status: failed
        return self.status == FAILED


class TrustPaymentsService(RequestsPoweredServiceBase):
    """
    Сервис для общения с TRUST биллингом.

    https://wiki.yandex-team.ru/TRUST/Payments/
    Интерфейс почти повторяет API BalanceXMLRPC
    """
    new_to_old_convertation_map = {
        STARTED: WAIT_FOR_NOTIFICATION,
        AUTHORIZED: SUCCESS,
        CLEARED: SUCCESS,
        NOT_AUTHORIZED: CANCELLED,
        CANCELED: REFUND,
    }
    valid_statuses = [SUCCESS, WAIT_FOR_NOTIFICATION]

    def __init__(self):
        self.enabled_for_all = None
        self.enabled_for_uids = []
        self.service_token = None
        super(TrustPaymentsService, self).__init__()

    def _exception_from_response(self, response):
        return errors.BigBillingBadResult(from_json(response.content))

    def request(self, method, relative_url, params=None, data=None, headers=None, cookies=None, timeout=None,
                send_tvm_ticket=None, check_status=True, *args, **kwargs):
        body = None
        if data is not None:
            body = to_json(data, true_utf8=False)

        response = super(TrustPaymentsService, self).request(
            method, relative_url, params=params, data=body, headers=headers, cookies=cookies, timeout=timeout,
            send_tvm_ticket=send_tvm_ticket, *args, **kwargs)
        parsed_response = from_json(response.content)

        if check_status:
            status = parsed_response.get(STATUS)
            if not status or status not in self.valid_statuses:
                raise errors.BigBillingBadResult(parsed_response)
        return parsed_response

    def order_place(self, uid, number, billing_pid, region, tariff_name=None):
        """
        Создать заказ (подписку)

        https://wiki.yandex-team.ru/TRUST/Payments/API/Orders/
        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/
        """
        data = {
            'product_id': billing_pid,
            'order_id': number,
        }
        if tariff_name is not None:
            data['developer_payload'] = {
                'tariff_name': tariff_name
            }

        headers = self._build_headers(uid=uid, region=region)

        if billing_pid.endswith(SUBS):
            relative_url = '/trust-payments/v2/subscriptions'
        else:
            relative_url = '/trust-payments/v2/orders'

        return self.request('POST', relative_url, data=data, headers=headers)

    def list_payment_methods(self, uid, region):
        """
        Листинг возможных способов оплаты

        https://wiki.yandex-team.ru/TRUST/Payments/API/PaymentMethods/
        """
        headers = self._build_headers(uid=uid, region=region)
        relative_url = '/trust-payments/v2/payment-methods'

        resp = self.request('GET', relative_url, headers=headers)
        return [x['id'] for x in resp['bound_payment_methods']]

    def payment_make(self, uid, number, paymethod_id, callback, return_path, bb_tld=None, template_tag=None,
                     locale=RU_RU):
        """
        Запустить оплату заказа

        При успешном выполнении вернет url для оплаты
        https://wiki.yandex-team.ru/TRUST/Payments/API/Baskets
        """
        headers = self._build_headers(uid=uid)

        # Пытаемся понять, создавали ли мы уже оплату по заказу или нет
        from mpfs.core.billing.order import look_for_order  # циклический импорт
        order = look_for_order(number)
        last_payment_id = self.get_last_payment_id(uid, order)
        if last_payment_id is not None:
            payment_info = self.check_payment_by_id(uid, last_payment_id)
            if payment_info.payment_url:
                return {'url': payment_info.payment_url, 'params': None}

        data = {
            'orders': [{
                'order_id': number,
                'qty': DEFAULT_BUY_QTY,
            }],
            'paymethod_id': paymethod_id,
            'back_url': callback,
            'return_path': return_path,
            'lang': locale
        }
        if bb_tld is not None:
            data['domain_sfx'] = bb_tld
        if template_tag is not None:
            data['template_tag'] = template_tag

        relative_url = '/trust-payments/v2/payments'
        payment_id = self.request('POST', relative_url, headers=headers, data=data)['purchase_token']

        relative_url = '/trust-payments/v2/payments/%s/start' % payment_id
        response = self.request('POST', relative_url, headers=headers)
        return {'url': response['payment_url'], 'params': {'purchase_token': payment_id}, PAYMENT_ID: payment_id}

    def create_product(self, billing_product):
        """
        Создать продукт в ББ

        https://wiki.yandex-team.ru/TRUST/Payments/API/Products/
        """
        headers = self._build_headers()
        relative_url = '/trust-payments/v2/products/'
        return self.request('POST', relative_url, headers=headers, data=billing_product)

    def unsubscribe(self, uid, number):
        """
        Отказаться от автопродляемой подписки

        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/
        """
        data = {
            'finish_ts': time.time(),
        }
        headers = self._build_headers(uid=uid)
        relative_url = '/trust-payments/v2/subscriptions/%s' % number
        return self.request('PUT', relative_url, headers=headers, data=data)

    def check_order(self, uid, number, convert_to_old_format=True):
        """
        Проверить состояние заказа

        Для совместимости с существующим интерфейсом в качестве ответа собирается результат собирается из результата
        проверки заказа (подписки) и оплаты, соответствующей заказу. В случае подписки берется последняя оплата.
        https://wiki.yandex-team.ru/TRUST/Payments/API/Orders/
        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/
        """
        from mpfs.core.billing.order import look_for_order  # циклический импорт
        order = look_for_order(number)
        is_auto = order is None or order.auto  # если заказа нет, то пойдем в ручку подписок
        if is_auto:
            result = self.__check_order_for_auto(uid, order)
        else:
            result = self.__check_order_for_non_auto(uid, order)

        if convert_to_old_format:
            result = self.__convert_to_old_format(result)
        return result

    def get_last_payment_id(self, uid, order):
        """
        Вытащить payment_id последней оплаты

        Для неавтопродляемых заказов достаем из базы, для автопродляемых из /trust-payments/v2/subscriptions
        """
        if order is None:
            return None
        if order.auto:
            relative_url = '/trust-payments/v2/subscriptions/%s' % order.number
            headers = self._build_headers(uid=uid)
            order_info = self.request('GET', relative_url, headers=headers)
            if order_info.get('payments'):
                return order_info['payments'][-1]
        else:
            return order.payment_id

    def check_payment_by_id(self, uid, payment_id):
        relative_url = '/trust-payments/v2/payments/%s' % payment_id
        headers = self._build_headers(uid=uid)
        response = self.request('GET', relative_url, headers=headers)
        return PaymentInfo(response)

    def is_enabled(self, uid=None):
        if self.enabled_for_all:
            return True
        return uid in self.enabled_for_uids

    def _build_headers(self, headers=None, uid=None, region=None, user_ip=None):
        headers = headers or {}

        if uid is not None:
            headers['X-Uid'] = uid
        if region is not None:
            headers['X-Region-Id'] = region
        if user_ip is not None:
            headers['X-User-Ip'] = user_ip
        headers.update({
            'X-Service-Token': self.service_token,
            'Content-Type': 'application/json',
        })
        return headers

    def __check_order_for_auto(self, uid, order):
        # получаем payment_id не через get_last_payment_id так как он содержится в вызове
        # /trust-payments/v2/subscriptions/<id> чтобы не делать лишний поход в ББ
        relative_url = '/trust-payments/v2/subscriptions/%s' % order.number
        headers = self._build_headers(uid=uid)
        order_info = self.request('GET', relative_url, headers=headers)
        result = order_info
        if order_info.get(SUBS_UNTIL):
            result[SUBS_UNTIL_MSEC] = str(
                int(float(order_info[SUBS_UNTIL]) * 1000))  # для поддержки интерфейса BalanceXMLRPC

        if not order_info.get('payments'):
            default_log.info('Missing payments in /trust-payments/v2/subscriptions response uid=%s, order_number=%s ' %
                             (uid, order.number))
            return self.__handle_missing_payment(result)
        payment_id = order_info['payments'][-1]

        payment_info = self.check_payment_by_id(uid, payment_id)

        result[STATUS] = payment_info.payment_status
        result[STATUS_CODE] = payment_info.payment_status_code

        return result

    def __check_order_for_non_auto(self, uid, order):
        relative_url = '/trust-payments/v2/orders/%s' % order.number
        headers = self._build_headers(uid=uid)
        order_info = self.request('GET', relative_url, headers=headers)
        result = order_info

        payment_id = self.get_last_payment_id(uid, order)
        if not payment_id:
            default_log.info('Missing payment_id in non auto order uid=%s, order_number=%s ' % (uid, order.number))
            return self.__handle_missing_payment(result)

        payment_info = self.check_payment_by_id(uid, payment_id)

        result[STATUS] = payment_info.payment_status
        result[STATUS_CODE] = payment_info.payment_status_code

        return result

    @staticmethod
    def __handle_missing_payment(order_info):
        order_info[STATUS] = NO_PAYMENT
        err_msg = 'no payment has been started for this order'  # Имитируем поведение BalanceXMLRPC
        raise errors.BigBillingBadResult(extra_msg=err_msg)

    @classmethod
    def __convert_to_old_format(cls, order_payment_info):
        """
        Приводит к старому формату статусов

        https://wiki.yandex-team.ru/Balance/Simple/#mappingstatusovbalancesimple.checkbasketnagrafsostojanijjplatezhavnovomrestapi
        Некоторые статусы менять не надо, их оставляем как есть.
        """
        if order_payment_info.get(STATUS) in cls.new_to_old_convertation_map:
            order_payment_info[STATUS] = cls.new_to_old_convertation_map[order_payment_info[STATUS]]
        return order_payment_info

    def create_refund(self, uid, payment_id, order_number, reason):
        """
        Создать возврат средств

        chttps://wiki.yandex-team.ru/TRUST/Payments/API/Refunds/

        :param str uid: uid пользователя
        :param str payment_id: id корзины
        :param order_number: номер заказа
        :param str reason: человекочитаемая причина возврата
        :return str: refund_id
        """
        relative_url = '/trust-payments/v2/refunds'
        headers = self._build_headers(uid=uid, user_ip='127.0.0.1')

        # завязываемся на то, что у нас всегда создается корзина с одним заказом и количеством = 1
        data = {
            'purchase_token': payment_id,
            'reason_desc': reason,
            'orders': [{
                'order_id': order_number,
                'delta_qty': DEFAULT_BUY_QTY,
            }]
        }
        response = self.request('POST', relative_url, data=data, headers=headers)

        status = response.get(STATUS)
        refund_id = response.get(TRUST_REFUND_ID)

        if not status or status != SUCCESS or not refund_id:
            raise errors.BigBillingBadResult(response)

        return refund_id

    def start_refund(self, uid, refund_id):
        """
        Запустить возврат средств

        :param str uid: uid пользователя
        :param str refund_id: id возврата
        :return RefundStatus:
        """
        relative_url = '/trust-payments/v2/refunds/%s/start' % refund_id
        headers = self._build_headers(uid=uid, user_ip='127.0.0.1')
        response = self.request('POST', relative_url, headers=headers)
        return RefundStatus(response)

    def get_refund_status(self, uid, refund_id):
        """
        Получить статус возврата средств

        :param str uid: uid пользователя
        :param str refund_id: id возврата
        :return RefundStatus:
        """
        relative_url = '/trust-payments/v2/refunds/%s' % refund_id
        headers = self._build_headers(uid=uid, user_ip='127.0.0.1')
        response = self.request('GET', relative_url, headers=headers, check_status=False)
        return RefundStatus(response)

    def verify_inapp_receipt(self, receipt, store_id):
        relative_url = '/trust-payments/v2/inapp_check_receipt/'
        headers = self._build_headers()
        data = {
            'receipt': receipt,
            'store_id': store_id,
        }
        response = self.request('GET', relative_url, data=data, headers=headers)
        if response['result']['receipt_check_status'] != 'valid':
            raise errors.BillingInAppInvalidVerification(response)
        default_log.info('Receipt verification result %s' % response['result']['receipt_info'])
        return app_store_response_parser(response['result']['receipt_info'])

    def handle_inapp_receipt(self, uid, receipt, store_id, currency='RUB'):
        relative_url = '/trust-payments/v2/inapp_receipt/'
        headers = self._build_headers(uid=uid, user_ip='127.0.0.1')
        data = {
            'receipt': receipt,
            'store_id': store_id,
            'currency': currency,
        }
        try:
            response = self.request('PUT', relative_url, data=data, headers=headers)
        except errors.BigBillingBadResult as e:
            if (len(e.args) > 0 and
                    'status_desc' in e.args[0] and
                    'Error due to google api call 400' in e.args[0]['status_desc']):
                error_log.exception("Got 400 from google api")
                raise errors.BillingInAppInvalidVerification('Invalid google play token')
            else:
                raise
        else:
            if response['status'] != 'success':
                raise errors.BillingTrustInvalidVerification(response)
            return response['items']

    def get_inapp_subscription(self, subs_id):
        relative_url = '/trust-payments/v2/inapp_subscription/%s' % subs_id
        response = self.request('GET', relative_url, headers=self._build_headers())
        if response['status'] != 'success':
            raise
        response.pop('status', None)
        return response

    def resync_inapp_subscription(self, subs_id):
        relative_url = '/trust-payments/v2/inapp_subscription/%s/resync' % subs_id
        return self.request('POST', relative_url, headers=self._build_headers())


trust_payment_service = TrustPaymentsService()
