# -*- coding: utf-8 -*-

"""
Сервис TVM 2.0

https://wiki.yandex-team.ru/passport/tvm2/theory/
https://wiki.yandex-team.ru/passport/tvm2/migration/
https://wiki.yandex-team.ru/passport/blackbox/#22.05.2017getuserticket
https://wiki.yandex-team.ru/passport/tvm2/api/#polucheniepublichnyxkljuchejj
https://wiki.yandex-team.ru/passport/tvm2/library/
https://wiki.yandex-team.ru/passport/tvm2/quickstart/#0-opredeljaemsjasokruzhenijami
https://wiki.yandex-team.ru/disk/TVM/

"""
import time

import ticket_parser2
import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util import from_json
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.core.services.tvm_service import TVMBaseTicket
from ticket_parser2.api.v1 import BlackboxEnv, ServiceContext, UserContext
from ticket_parser2.api.v1.exceptions import TvmException


default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('tvm')

SERVICES_TVM_2_0_SECRET_KEY = settings.services['tvm_2_0']['secret_key']
SERVICES_TVM_2_0_CLIENT_ID = settings.services['tvm_2_0']['client_id']
SERVICES_TVM_2_0_KEYS_CHECK_INTERVAL = settings.services['tvm_2_0']['keys_check_interval']
SERVICES_TVM_2_0_SERVICE_TICKET_REFRESH_INTERVAL = settings.services['tvm_2_0']['service_ticket_refresh_interval']
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']
SERVICES_TVM_2_0_BLACKBOX_ENV = settings.services['tvm_2_0']['blackbox_env']
SERVICES_TVM_2_0_YATEAM_BLACKBOX_ENV = settings.services['tvm_2_0']['yateam_blackbox_env']


class TVM2Service(RequestsPoweredServiceBase):
    """
    Сервис для общения с TVM 2.0 и валидации тикетов.

    Для использования класса необходимо вызвать метод update_tvm_pub_key, чтобы проинициализировать контексты. этот
    метод надо вызывать повторно каждые 24 часа. Затем необходимо вызвать update_service_tickets, чтобы выписать
    сервисные тикеты для походов в другие сервисы (действуют чуть больше часа).

    Сервисные тикеты можно достать с помощью
    mpfs.engine.process.get_tvm_2_0_service_ticket_for_client(<client_id>)
    """
    _tvm_service_context = None
    _tvm_user_context = None
    _yateam_tvm_user_context = None

    def __init__(self):
        super(TVM2Service, self).__init__()
        self.last_user_context_public_keys_update_time = None
        self.last_service_context_public_keys_update_time = None
        self.service_pub_keys = None
        self.user_pub_keys = None

    def get_new_service_ticket(self, dst_client_id):
        """Получить новый сервисный TVM тикет

        :rtype: TVM2Ticket
        """
        if not SERVICES_TVM_2_0_ENABLED:
            return None

        ts = int(time.time())
        data = {
            'ts': ts,
            'src': SERVICES_TVM_2_0_CLIENT_ID,
            'dst': dst_client_id,
            'grant_type': 'client_credentials',
            'sign': self._tvm_service_context.sign(ts, dst_client_id)
        }
        response = self.request('POST', 'ticket', data=data)
        return TVM2Ticket.build_tvm_ticket(from_json(response.content)[str(dst_client_id)]['ticket'])

    def get_all_pub_keys(self):
        """Получить публичные ключи"""
        tvm_keys = self.request('GET', 'keys', {'lib_version': ticket_parser2.__version__}).content
        return tvm_keys

    def validate_service_ticket(self, ticket):
        if self.need_update_pub_keys():
            self.update_public_keys()
        try:
            validated_ticket = self._tvm_service_context.check(ticket.value())
        except TvmException as ex:
            error_log.error(
                'Failed to validate service ticket %s, last public keys update timestamp is %s used public keys %s '
                'TVM debug: message="%s" debug_info="%s"' %
                (ticket.value(), self.last_service_context_public_keys_update_time, self.service_pub_keys, ex.message, ex.debug_info),
                exc_info=True
            )
            raise
        return validated_ticket

    def validate_user_ticket(self, ticket):
        if self.need_update_pub_keys():
            self.update_public_keys()
        try:
            validated_ticket = self._tvm_user_context.check(ticket.value())
        except TvmException as ex:
            error_log.error(
                'Failed to validate user ticket %s, last public keys update timestamp %s used public keys %s'
                'TVM debug: message="%s" debug_info="%s"' %
                (ticket.value(), self.last_user_context_public_keys_update_time, self.user_pub_keys, ex.message, ex.debug_info),
                exc_info=True
            )
            raise
        return validated_ticket

    def validate_yateam_user_ticket(self, ticket):
        if self.need_update_pub_keys():
            self.update_public_keys()

        try:
            validated_ticket = self._yateam_tvm_user_context.check(ticket.value())
        except TvmException as ex:
            error_log.error(
                'Failed to validate yateam user ticket %s, last public keys update timestamp %s used public keys %s'
                'TVM debug: message="%s" debug_info="%s"' %
                (ticket.value(), self.last_user_context_public_keys_update_time, self.user_pub_keys, ex.message, ex.debug_info),
                exc_info=True
            )
            raise
        return validated_ticket

    def update_public_keys(self, silent_mode_on_errors=True):
        """
        Обновляет публичные ключи клиентов с TVM авторизацией
        """
        if not SERVICES_TVM_2_0_ENABLED:
            return

        try:
            tvm_keys = self.get_all_pub_keys()
        except Exception:
            if silent_mode_on_errors:
                error_log.error('Error occurred during fetching TVM public keys. Using old keys.', exc_info=True)
            else:
                error_log.error('Error occurred during fetching TVM public keys.', exc_info=True)
                raise

        cur_time = int(time.time())

        try:
            TVM2Service._tvm_service_context = ServiceContext(
                SERVICES_TVM_2_0_CLIENT_ID, SERVICES_TVM_2_0_SECRET_KEY, tvm_keys)
        except Exception:
            error_log.error('Something failed during creating TVM ServiceContext', exc_info=True)
            if not silent_mode_on_errors:
                raise
        else:
            self.last_service_context_public_keys_update_time = cur_time
            self.service_pub_keys = tvm_keys

        try:
            TVM2Service._tvm_user_context = UserContext(
                BlackboxEnv.__getattr__(SERVICES_TVM_2_0_BLACKBOX_ENV), tvm_keys)
            TVM2Service._yateam_tvm_user_context = UserContext(
                BlackboxEnv.__getattr__(SERVICES_TVM_2_0_YATEAM_BLACKBOX_ENV), tvm_keys)
        except Exception:
            error_log.error('Something failed during creating TVM UserContext', exc_info=True)
            if not silent_mode_on_errors:
                raise
        else:
            self.last_user_context_public_keys_update_time = cur_time
            self.user_pub_keys = tvm_keys

    def update_service_tickets(self):
        """
        Обновляет сервисные TVM тикеты для внешних сервисов
        """
        if not SERVICES_TVM_2_0_ENABLED:
            return
        for service_name, service_settings in settings.services.items():
            if 'tvm_2_0' not in service_settings:
                continue
            try:
                ticket = self.get_new_service_ticket(service_settings['tvm_2_0']['client_id'])
                mpfs.engine.process.set_tvm_2_0_service_ticket_for_client(
                    service_settings['tvm_2_0']['client_id'], ticket)
            except Exception:
                error_log.error('Couldn\'t update service ticket for service %s' % service_name, exc_info=True)

    def need_update_pub_keys(self):
        cur_time = int(time.time())
        need_update_service_context_pub_keys = ((cur_time - self.last_service_context_public_keys_update_time)
                                                > SERVICES_TVM_2_0_KEYS_CHECK_INTERVAL)
        need_update_user_context_pub_keys = ((cur_time - self.last_user_context_public_keys_update_time)
                                             > SERVICES_TVM_2_0_KEYS_CHECK_INTERVAL)
        return need_update_service_context_pub_keys or need_update_user_context_pub_keys


class TVM2Ticket(TVMBaseTicket):
    ticket_lifespan = SERVICES_TVM_2_0_SERVICE_TICKET_REFRESH_INTERVAL


tvm2 = TVM2Service()
