# -*- coding: utf-8 -*-

"""
Сервис TVM

https://wiki.yandex-team.ru/passport/auth-tokens/#serveriligruppaserverovdljaprilozhenijj
https://wiki.yandex-team.ru/passport/auth-tokens/using/#testovyetikety
https://wiki.yandex-team.ru/disk/TVM/
https://wiki.yandex-team.ru/product-security/tvm/
https://wiki.yandex-team.ru/passport/auth-tokens/using/libticketpa
https://wiki.yandex-team.ru/passport/auth-tokens/faq/#kakperedavatixranittiket

"""
import base64
import hashlib
import hmac
import time
import urllib

import ticket_parser

from enum import IntEnum

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.config import settings
from mpfs.core.signals import (
    add_timer,
    register_signal,
    Target,
)
from mpfs.core.services.common_service import Service, RequestsPoweredServiceBase

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('tvm')

SERVICES_TVM_SECRET_KEY = settings.services['tvm']['secret_key']
SERVICES_TVM_CLIENT_ID = settings.services['tvm']['client_id']
SERVICES_TVM_KEYS_CHECK_INTERVAL = settings.services['tvm']['keys_check_interval']
SERVICES_TVM_TICKET_LIFESPAN = settings.services['tvm']['ticket_lifetime']
SERVICES_TVM_ENABLED = settings.services['tvm']['enabled']


class TVMSignatureParametersBuilder(object):
    @staticmethod
    def build():
        """Возвращает dict с параметрами, необходимыми для получения TVM тикета.
           Содержит ts, ts_sign и client_id.
        """
        ts = str(int(time.time()))
        sign = TVMSignatureParametersBuilder._get_ts_sign(ts)
        data = {
            'ts_sign': sign,
            'ts': ts,
            'client_id': SERVICES_TVM_CLIENT_ID,
        }
        return data

    @staticmethod
    def _get_ts_sign(ts):
        """:param str ts: timestamp"""
        # base64 не умеет работать без паддинга, как этого требует TVM, поэтому помогаем ей в этом.
        padded_secret = SERVICES_TVM_SECRET_KEY + '=' * (4 - len(SERVICES_TVM_SECRET_KEY) % 4)
        decoded_secret = base64.urlsafe_b64decode(padded_secret)
        hash_ = hmac.new(key=decoded_secret, msg=ts, digestmod=hashlib.sha256).digest()
        # Удалять '=' в конце тоже необходимо, поскольку этого требует TVM.
        sign = base64.urlsafe_b64encode(hash_).strip('=')
        return sign


class TVMService(RequestsPoweredServiceBase):
    _tvm_context = ticket_parser.TAsymmetricPublicContext()

    def get_new_ticket(self):
        """Получить новый TVM тикет
        :return: Объект класса TVMTicket
        """
        if not SERVICES_TVM_ENABLED:
            return None

        data = TVMSignatureParametersBuilder.build()
        data['grant_type'] = 'client_credentials'

        response = self.request('POST', 'ticket', data=data)
        return TVMTicket.build_tvm_ticket(response.content, is_external=False)

    def get_new_pub_keys(self, client_ids):
        """Получить публичные ключи для перечисленных client_ids
        :return: Строка из ключей, разделенных '\n'
        """
        client_ids = (str(i) for i in client_ids)
        response = self.request('POST', 'ticket', data={'getkeys': 'yes', 'client_id': ','.join(client_ids)})
        return response.content

    @classmethod
    def validate_ticket(cls, ticket):
        try:
            result = TVMService._tvm_context.check_string(ticket.value())
        except ticket_parser.TicketParserException:
            return None
        return result

    def collect_tvm_pub_keys(self):
        """
        Обновляет публичные ключи клиентов с TVM авторизацией
        """
        if not SERVICES_TVM_ENABLED:
            return

        client_ids = mpfs.engine.process.get_tvm_clients()
        if not client_ids:
            default_log.info('No TVM clients have been registered')
            return
        try:
            new_tvm_context = ticket_parser.TAsymmetricPublicContext()
            response_keys = self.get_new_pub_keys(client_ids)
            if not response_keys:
                default_log.info('No public keys were received')
            else:
                new_tvm_context.add_formatted_keys(response_keys)
                default_log.info('Some TVM public keys were updated')
            TVMService._tvm_context = new_tvm_context
        except Exception as e:
            error_log.error('Error occurred during getting TVM public_keys')


class TVMBaseTicket(object):
    """
    Общий класс для TVM 1.0 и TVM 2.0 тикетов

    :param int ticket_lifespan: Время жизни тикета. По истечению этого времени тикет считается недействительным
    """
    ticket_lifespan = 0

    def __init__(self):
        self._ctime = int(time.time())
        self._ticket = None

    @classmethod
    def build_tvm_ticket(cls, ticket):
        tvm_ticket = cls()
        tvm_ticket._ticket = ticket
        return tvm_ticket

    def value(self):
        return self._ticket

    def is_expired(self):
        return (int(time.time()) - self._ctime) > self.ticket_lifespan


class TVMTicket(TVMBaseTicket):
    """
    Внутреннее представление TVM 1.0 тикета

    :param external: является ли тикет переданным в сервис или мы поулчили его сами. Для внешних тикетов
               не производится проверка времени жизни.
    """
    ticket_lifespan = SERVICES_TVM_TICKET_LIFESPAN / 2

    def __init__(self):
        super(TVMTicket, self).__init__()
        self._external = False

    @classmethod
    def build_tvm_ticket(cls, ticket, is_external=True):
        tvm_ticket = super(TVMTicket, cls).build_tvm_ticket(ticket)
        tvm_ticket._external = is_external
        return tvm_ticket

    def is_external(self):
        return self._external


class TVMEnv(IntEnum):
    """TVM окружения.

    Выделяющая окружения как:
      * внутреннее (yandex-team)
      * внешнее
    """
    YaTeam = 1
    External = 2


def setup_tvm_pub_keys_update():
    """
    Запустить периодическое обновление публичных ключей TVM.
    """
    if not SERVICES_TVM_ENABLED:
        return

    register_signal(
        mpfs.engine.process.Signal.TVM_KEYS,
        lambda signum: tvm.collect_tvm_pub_keys(),
        Target.ALL_WORKERS
    )
    add_timer(mpfs.engine.process.Signal.TVM_KEYS, SERVICES_TVM_KEYS_CHECK_INTERVAL)


tvm = TVMService()
