# -*- coding: utf-8 -*-
"""
Сервис UaaS

Параметры запроса:
headers:
User-Agent - без User-Agent не будет происходить разбиение по типам устройств
             (эксперимент не будет отдан, если у эксперимента есть ограничения
             по устройствам). Тип устройства указан практически для всех
             экспериментов.

https://wiki.yandex-team.ru/users/buryakov/usaas/
https://st.yandex-team.ru/CHEMODAN-41712
"""
import hashlib

from mpfs.common.util import from_json, to_json, Cached
from mpfs.common import errors
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.common.util.user_agent_parser import UserAgent
from mpfs.engine.process import get_service_log


service_log = get_service_log('uaas')


class UAAS(RequestsPoweredServiceBase, Cached):
    _experiments_by_uid = {}
    u"""Кэш для запросов в UaaS без user-agent'а (только по uid)"""
    uaas_request_useragents = {
        (UserAgent.IOS, 'phone'):
            'Mozilla/5.0 (iPhone; CPU iPhone OS 11_1 like Mac OS X) AppleWebKit/604.3.5 (KHTML, like Gecko) '
            'Version/11.0 Mobile/15B93 Safari/604.1',
        (UserAgent.ANDROID, 'phone'):
            'Mozilla/5.0 (Linux; Android 5.1.1; Nexus 6 Build/LYZ28E) AppleWebKit/537.36 (KHTML, like Gecko) '
            'Chrome/64.0.3265.0 Mobile Safari/537.36',
        (UserAgent.IOS, 'tablet'):
            'Mozilla/5.0 (iPad; CPU OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) '
            'Version/9.0 Mobile/13B143 Safari/601.1',
        (UserAgent.ANDROID, 'tablet'):
            'Mozilla/5.0 (Linux; Android 4.4; Nexus 5 Build/_BuildID_) AppleWebKit/537.36 (KHTML, like Gecko) '
            'Version/4.0 Chrome/30.0.0.0 Mobile Safari/537.36',
    }
    DESKTOP_UAAS_UA = 'Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.102 YaBrowser/18.11.1.80'
    EMPTY_DEVICE_IDS = {'unknown', 'null', 'nil', 'none', ''}

    def _get_headers_and_cookies_by_user_agent(self, user_agent):
        cookies = {}
        headers = {}
        try:
            os = user_agent.get_os_family()
        except ValueError:
            raise errors.UAASRequestInvalidClient()

        if os in UserAgent.DESKTOP:
            request_user_agent = self.DESKTOP_UAAS_UA
        else:
            request_user_agent = self.uaas_request_useragents.get((os, user_agent.device_type))

        if not request_user_agent:
            raise errors.UAASRequestInvalidClient()

        headers['User-Agent'] = request_user_agent

        if user_agent.id is not None and str(user_agent.id).lower() not in self.EMPTY_DEVICE_IDS:
            cookies['fuid01'] = self._make_fuid01(str(user_agent.id))

        return headers, cookies

    def get_disk_experiments(self, user_agent=None, uid=None):
        return self.get_experiments(user_agent=user_agent, uid=uid, acceptable_handlers={'DISK'}, relative_url='/disk/')

    def get_telemost_experiments(self, user_agent=None, uid=None, additional_headers=None):
        return self.get_experiments(user_agent=user_agent, uid=uid, acceptable_handlers={'TELEMOST'},
                                    relative_url='/telemost/', additional_headers=additional_headers)

    def get_experiments(self, user_agent=None, uid=None, acceptable_handlers=None, relative_url='/', additional_headers=None):
        """
        Получить параметры экспериментов для пользователя

        :param user_agent: экземпляр mpfs.common.util.user_agent_parser.UserAgent
        :param uid: uid пользователя
        :param list|set acceptable_handlers: список допустимых значений поля HANDLER, если не задано, то все значения
        :param str relative_url: path по которому запрашивать эксперименты
        :return: список параметров экспериментов
        """
        if user_agent is None and uid is None:
            raise errors.UAASRequestInvalidClient()

        if user_agent is None and uid is not None and uid in self._experiments_by_uid:
            return self._experiments_by_uid[uid]

        headers, cookies = {}, {}
        if user_agent is not None:
            headers, cookies = self._get_headers_and_cookies_by_user_agent(user_agent)

        params = {}
        if uid is not None:
            params['uuid'] = uid
        if additional_headers:
            headers.update(additional_headers)

        response = self.request('GET', relative_url=relative_url, headers=headers, params=params, cookies=cookies)
        if not response.headers.get('X-Yandex-ExpConfigVersion'):
            raise errors.UAASExpConfigVersionNotFoundError(
                'header X-Yandex-ExpConfigVersion is not presented'
            )
        exp_flags = response.headers.get('X-Yandex-ExpFlags')

        flags_list = []
        if exp_flags:
            flags_list = self._decode_experiments(exp_flags, acceptable_handlers)
        if user_agent is None and uid is not None:
            self._experiments_by_uid[uid] = flags_list

        service_log.info(to_json(flags_list))

        return flags_list

    @staticmethod
    def _make_fuid01(val):
        return hashlib.md5(val).hexdigest()[:16]

    @staticmethod
    def _decode_experiments(encoded_experiments, acceptable_handlers=None):
        if acceptable_handlers and not isinstance(acceptable_handlers, set):
            acceptable_handlers = set(acceptable_handlers)

        disk_experiments = []
        for experiments in encoded_experiments.split(','):
            experiments = experiments.decode('base64')
            experiments = from_json(experiments)

            if not isinstance(experiments, list):
                raise errors.UAASResponseError(
                    'expected encoded list in json, but received type `%s`' % type(experiments)
                )

            if acceptable_handlers and all(e.get('HANDLER') not in acceptable_handlers for e in experiments):
                continue  # отфильтровываем все ненужные эксперименты

            for experiment in experiments:
                if experiment.get('HANDLER') in acceptable_handlers:
                    disk_experiments.append(experiment)

        return disk_experiments

    @classmethod
    def reset(cls):
        cls._experiments_by_uid = {}


class UAASDiskExperiment(object):
    """
    Объект эксперемента

    Raw UAAS experiment exmaple:
    {
      "CONTEXT": {
        "DISK": {
          "testid": [
            "155462"
          ],
          "data": [
            {
              "activityTimeout": 5.184e+09,
              "closedBannerTimeout": 2.592e+09,
              "downloadAppTimeout": 172800000
            }
          ],
          "flags": [
            "disk_public_banner_PO_test"
          ]
        }
      },
      "HANDLER": "DISK"
    }
    """
    def __init__(self, raw_experiment):
        self.raw_experiment = raw_experiment
        try:
            self.disk_exp = raw_experiment['CONTEXT']['DISK']
        except Exception:
            self.disk_exp = None

    def get_test_id(self):
        if self.disk_exp:
            return self.disk_exp['testid'][0]

    def get_data(self):
        if self.disk_exp:
            return self.disk_exp.get('data')

    def has_flags(self):
        return bool(self.disk_exp and self.disk_exp.get('flags'))

    def get_flags(self):
        if not self.has_flags():
            return []
        return self.disk_exp['flags']

    def has_match_by_flag_prefix(self, flag_prefix):
        flags = self.get_flags()
        for flag in flags:
            if flag.startswith(flag_prefix):
                return True
        return False

    def __repr__(self):
        return "%s(%s, %s)" % (
            self.__class__.__name__,
            self.get_test_id(),
            self.get_flags()
        )


class NewUAAS(RequestsPoweredServiceBase, Cached):

    cache_enabled = False

    def get_experiments(self, user_agent=None, uid=None):
        """
        Получить дисковые эксперементы

        :return: список объектов UAASDiskExperiment
        """
        raw_experiment = self.get_disk_experiments(user_agent=user_agent, uid=uid)
        return [UAASDiskExperiment(exp) for exp in raw_experiment]

    def get_disk_experiments(self, user_agent=None, uid=None, additional_headers=None):
        """
        Получить параметры экспериментов для пользователя

        Лучше использовать метод get_experiments

        :param user_agent: экземпляр str
        :param uid: uid пользователя
        :return: список параметров экспериментов
        """
        headers = {'user-agent': user_agent}
        if additional_headers:
            headers.update(additional_headers)
        params = {'uuid': uid}

        if self.cache_enabled:
            try:
                response = self._instances[(uid, user_agent)]
            except KeyError:
                response = self.request('GET', '/', headers=headers, params=params)
                self._instances[(uid, user_agent)] = response
        else:
            response = self.request('GET', '/', headers=headers, params=params)
        flags_list = from_json(response.content)['uas_exp_flags']

        service_log.info(to_json(flags_list))

        return filter(lambda flag: flag['HANDLER'] == 'DISK', flags_list)


uaas = UAAS()
new_uaas = NewUAAS()
