
import re
import json
import os

import mpfs.engine.process

from time import mktime
from datetime import datetime
from mpfs.common import errors
from mpfs.core.services.common_service import StorageService, BaseFilter

service_log = mpfs.engine.process.get_service_log("yavideo")
error_log = mpfs.engine.process.get_error_log()


class YaVideo(StorageService):
    name = 'yavideo'
    api_error = errors.YaVideoError
    log = service_log

    def __init__(self, *args, **kwargs):
        StorageService.__init__(self, *args, **kwargs)

        self.available_filters = {
            'ctime' : BaseFilter,
            'visible' : BaseFilter,
            'name' : BaseFilter,
            'public' : BaseFilter,
        }

        self.available_bounds = {
            'sort' : 'sort',
            'order' : 'order',
            'amount' : 'amount',
            'offset' : 'offset',
        }

        # actually sort params are ignored on yandex.video backend, but we need this to correctly pass
        # amount and offset params
        self.available_sort_fields = {
            'ctime' : 'ctime',
            'mtime' : 'mtime',
            'utime' : 'utime',
            'etime' : 'etime',
            'size'  : 'size',
            'name_natural' : 'name',
            'name': 'name',
        }

    def process_response(self, params):
        action = params["action"]
        params["__result-type"] = "json"
        args='&'.join([ t[0] + '=' + str(t[1]) for t in params.items()])
        response = self.open_url(self.base_url + action + "?" + args)
        result = json.loads(response)
        if result.get('error'):
            err = result.get('error')
            error_log.error("Bad response from yavideo: %s : %s", err.get('message'), err.get('stack-trace'))
            raise errors.YaVideoError()
        return result["result"]

    def parse_film(self, f):
        previews = []

        for p in f['previews']:
            previews.append({'url': p['url'], 'name': p['size']})

        film = {}
        meta = {}
        name = f['title']

        meta['url'] = f['origUrl']
        meta['directoryid'] = f['directoryId']
        meta['streamId'] = f['streamId']

        if f['isAudio']:
            film['mimetype'] = 'audio/mpeg'

            if not name.endswith(".mp3"):
                name = name + ".mp3"
        else:
            film['mimetype'] = 'video/mp4'

            # previews for video only
            meta['sizes'] = previews
            meta['thumbnail'] = f['thumbUrl']

        if f.get('dar'):
            meta['dar'] = f['dar']

        film['id'] = f.get('id')
        film['type'] = 'file'
        film['meta'] = meta
        film['name'] = name
        film['ctime'] = self.time_parse(f['uploadTime'])
        film['size'] = f['origSize']

        return film

    def parse_album(self, a):
        album = {'id': a.get('id'),
                 'type': 'dir',
                 'meta': {},
                 'name': a['title'],
                 'ctime': self.time_parse(a['creationTime'])}
        return album

    def parse_response(self, resp):
        result = []
        for a in resp.get('albums') or []:
            result.append(self.parse_album(a))

        for f in resp.get('films') or []:
            result.append(self.parse_film(f))

        this = {}

        if resp.get('album'):
            this = self.parse_album(resp['album'])

        return {'this': this, 'list': result}

    def get_film(self, address):
        if address.is_storage:
            raise errors.NotFile()

        request_data = {}
        request_data['action'] = 'getMyFilmForDisk'
        request_data['__uid'] = address.uid
        request_data['filmN'] = address.name

        resp = self.process_response(request_data)
        return self.parse_film(resp)


    def folder_content(self, resource):
        super(YaVideo, self).folder_content(resource)

        result = []
        request_data = {}

        for k, v in self.request_processing_fields['bounds'].iteritems():
            request_data[k] = v
        if not 'amount' in request_data:
            request_data['amount'] = 10000

        request_data['limit'] = request_data['amount']
        request_data.pop('amount')

        if resource.address.is_storage:
            action = 'getMyAlbumsForDisk'
        else:
            request_data['albumN'] = resource.address.name
            action = 'getMyFilmsForDisk'

        request_data['action'] = action
        request_data['__uid'] = resource.uid

        resp = self.process_response(request_data)
        parsed = self.parse_response(resp)

        resource.update(parsed['this'])
        resource.id = resource.address.path

        for child in parsed['list']:
            child = child.get('this', child)
            child['id'] = os.path.join(resource.address.path, str(child['id']))
            child['visible'] = 1
            if child['type'] == 'dir':
                resource.sorted_folders.append(child['id'])
                resource.child_folders[child['id']] = child
                resource.construct_child_folder(child['id'], child)
            else:
                resource.sorted_files.append(child['id'])
                resource.child_files[child['id']] = child
                resource.construct_child_file(child['id'], child)
            result.append(child)
        return result

    def time_parse(self, val):
        RE_TIME = '(\d+)-(\d+)-(\d+).(\d+):(\d+):(\d+)(.\d+)'
        _y, _mnth, _d, _h, _min, _s, difference = re.search(RE_TIME, val).groups()
        return int(mktime(datetime(
            int(_y),
            int(_mnth),
            int(_d),
            int(_h),
            int(_min),
            int(_s)).timetuple()))

    def remove_film_or_album(self, uid, path):
        RE_PATH = '/yavideo/(\d+)/(\d+)?'
        m = re.search(RE_PATH, path)
        if not m:
            raise errors.YaVideoError

        request_data = {}
        albumN, filmN = m.groups()

        if not filmN :
            request_data['action'] = 'deleteMyAlbum'
            request_data['albumN'] = albumN
        else :
            request_data['action'] = 'deleteMyFilm'
            request_data['filmN'] = filmN

        request_data['__uid'] = uid
        self.process_response(request_data)

    def lock_check(self, uid, path, version=None):
        return False


