# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Заберун

http://wiki.yandex-team.ru/Pochta/ya.disk/downloader

"""
import time
import base64
import re
import urllib
import datetime

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util.crypt import CryptAgent
from mpfs.core.services.common_service import Service

error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('zaberun')

USE_TOKEN_V2 = settings.feature_toggles['zaberun_use_token_v2']


class Zaberun(Service):

    name = 'zaberun'
    log  = service_log
    crypt_agent = None

    def __init__(self, *args, **kwargs):
        super(Zaberun, self).__init__(*args, **kwargs)
        if self.stid_crypted:
            self.crypt_agent = CryptAgent(key=self.stid_secret)

    def generate_preview_url(self, uid, mid, filename, **kwargs):
        if not filename:
            filename = 'preview'
        size = kwargs.pop('size', None)
        crop = kwargs.pop('crop', 0)
        kwargs['parser'] = 'preview'

        url = self.generate_file_url(uid, mid, filename, **kwargs)

        if size or crop:
            url = url + '&size=%s&crop=%s' % (size, crop)
        return url

    def generate_file_url(self, uid, mid, filename, **kwargs):
        url_params = self.process_params(uid, mid, filename, **kwargs)

        # шифруем stid
        if self.stid_crypted:
            try:
                processed_mid = self.crypt_agent.encrypt(mid)
            except TypeError:
                error_log.error("Error: missing stid")
                #https://st.yandex-team.ru/CHEMODAN-22090
                processed_mid = ''

            #https://jira.yandex-team.ru/browse/CHEMODAN-8207
            processed_mid = re.sub(r'\+', r'-', processed_mid)
            processed_mid = re.sub(r'\/', r'_', processed_mid)

            processed_mid = urllib.quote(processed_mid, safe="")
        else:
            processed_mid = base64.b64encode(mid)

        url_params.update({
            'mid': processed_mid,
        })

        fsize = url_params['fsize']
        hid = url_params['hid']
        media_type = url_params['media_type']
        add = ''
        if fsize:
            add += '&fsize=%s' % fsize
        if hid:
            add += '&hid=%s' % hid
        if media_type:
            add += '&media_type=%s' % media_type
        if USE_TOKEN_V2:
            add += '&tknv=v2'
        # https://st.yandex-team.ru/CHEMODAN-22562
        if 'md5' in kwargs and kwargs['md5']:
            add += '&etag=%s' % kwargs['md5']
        return self.file_url % url_params + add

    def generate_folder_url(self, uid, path, filename, parser='zip', *args, **kwargs):
        kwargs['parser'] = parser
        filename = '%s.zip' % filename
        url_params = self.process_params(uid, path, filename, **kwargs)

        encoded_path = base64.b64encode(path.encode('utf-8'))
        url_params.update({
            'mid'       : encoded_path,
        })
        add = '&tknv=v2' if USE_TOKEN_V2 else ''
        url = self.folder_url % url_params + add
        return url

    def generate_zip_album_url(self, uid, album_public_key, album_title):
        return self.generate_folder_url(uid,
                                        album_public_key,
                                        album_title,
                                        parser='zip-album',
                                        hash=album_public_key,
                                        content_type = 'application/zip',
                                        inline=False)

    def _generate_archive_filename(self):
        current_datetime = datetime.datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
        return 'archive-%s' % current_datetime

    def generate_zip_files_url(self, uid, oid):
        archive_filename = self._generate_archive_filename()
        return self.generate_folder_url(uid, oid, archive_filename,
                                        parser='zip-files',
                                        hash=oid,
                                        content_type='application/zip',
                                        inline=False)

    def process_params(self, uid, mid, filename, **kwargs):
        content_type = kwargs.get('content_type', self.default_content_type)
        if not content_type:
            content_type = self.default_content_type

        # обрабатываем некорректные mimetype
        try:
            encoded_content_type = urllib.quote(content_type, safe='')
        except Exception:
            # content type (mimetype) в урле и в hmac подписи должны совпадать, иначе сверка на заберуне не пройдёт.
            content_type = self.default_content_type
            encoded_content_type = self.default_content_type

        inline = kwargs.get('inline', False)
        hash = kwargs.get('hash', '')
        limit = kwargs.get('limit', 0)
        eternal = kwargs.get('eternal', False)
        fsize = kwargs.get('fsize')
        parser = kwargs.get('parser', 'disk')
        hid = kwargs.get('hid')
        media_type = kwargs.get('media_type')
        expire_seconds = kwargs.get('expire_seconds')
        owner_uid = kwargs.get('owner_uid') or uid

        if expire_seconds is None:
            if parser == 'zip':
                expire_seconds = self.default_zip_expire_seconds
            else:
                expire_seconds = self.default_expire_seconds
        else:
            expire_seconds = int(expire_seconds)

        # собираем параметры
        infTimestamp = 'inf' if USE_TOKEN_V2 else 'mpfs'
        timestamp = infTimestamp if eternal else "%x" % (int(time.time()) + expire_seconds)
        disposition  = 'inline' if inline else 'attachment'
        if USE_TOKEN_V2:
            token = self.signed_token_v2(timestamp, mid, uid, filename, content_type, disposition, hash, limit)
        else:
            token = self.signed_token(timestamp, mid, uid, filename, content_type, disposition, hash, limit)
        encoded_name = filename.encode('utf-8') if isinstance(filename, unicode) else filename
        encoded_hash = hash.encode('utf-8') if isinstance(hash, unicode) else hash

        if uid in self.unlimited_users:
            limit = 0

        tld = self.tlds.get(parser, self.tlds['default'])

        url_params = {
            'uid'          : uid,
            'filename'     : encoded_name,
            'disposition'  : disposition,
            'hash'         : encoded_hash,
            'hid'          : hid,
        }

        for k, v in url_params.iteritems():
            url_params[k] = urllib.quote(str(v))

        url_params.update({
            'token'     : token,
            'timestamp' : timestamp,
            'limit'     : limit,
            'parser'    : parser,
            'content_type' : encoded_content_type,
            'media_type'   : media_type,
            'fsize'     : fsize,
            'hid'       : hid,
            'tld'       : tld,
            'owner_uid' : owner_uid,
        })

        service_log.info(
            '%s %s %s %s %s %s %s %s %s %s' %
            (timestamp, mid, uid, filename, content_type, disposition, hash, limit, fsize, hid)
        )
        return url_params

    def generate_public_url(self, mid, filename, **kwargs):
        return self.generate_file_url('0', mid, filename, **kwargs)

    def unpack_mid(self, hsh):
        hsh = urllib.unquote(hsh)
        hsh = re.sub(r'-', r'\+', hsh)
        hsh = re.sub(r'_', r'\/', hsh)
        return self.crypt_agent.decrypt(hsh)
