# -*- coding: utf-8 -*-

"""Модуль для работы с сигналами и таймерами uWSGI.

uWSGI сигналы по смыслу похожи на POSIX сигналы, но разная реализация.

Интерфейс построен поверх uWSGI Signal Framework, доступный через модуль uwsgi.
Цель:
    1. абстрагировать от наличия модуля uwsgi. Нужно в тестах и очередях.
    2. использовать символические константы перечислений (см :class:`~Target`)
    3. скрыть тонкости реализации интерфейса сигналов в модуле uwsgi и наделить дополнительными проверками.
        (см. коментарии к функциям)

:Example:
    Установить обработчик сигнала о проверке кеша каждые 60 секунд произвольным воркером.

    >>> class Signal(object):
    ...     CACHE = 1  # номер сигнала проверки кеша

    >>> def check_cache(signum):
    ...     # сделать проверку кеша
    ...     pass

    >>> register_signal(Signal.CACHE, check_cache, Target.RANDOM_WORKER)
    >>> add_timer(Signal.CACHE, 60)

.. warning:: Разрешается устанавливать обработчики только в мастер-процессе, т.е. в :mod:`~uwsgi_common.py`.
    Подробнее см. здесь: http://uwsgi-docs.readthedocs.org/en/latest/Signals.html#the-signals-table

.. seealso::
    uWSGI Signal Framework:
        http://uwsgi-docs.readthedocs.org/en/latest/Signals.html
    Модуль uwsgi:
        http://uwsgi-docs.readthedocs.org/en/latest/PythonModule.html

"""

import os

try:
    # автоматически добавляется мастер процессом
    import uwsgi
    UWSGI = True
except ImportError:
    UWSGI = False

__all__ = ['Target', 'register_signal', 'is_signal_registered', 'add_timer']


class Target(object):
    """Перечисление типов получателей сигнала.

    Список допустимых значений см.:
    http://uwsgi-docs.readthedocs.org/en/latest/PythonModule.html#uwsgi.register_signal
    """
    RANDOM_WORKER = "worker"
    ALL_WORKERS = "workers"


def register_signal(signum, handler, target=Target.RANDOM_WORKER):
    """Установить обработчик сигнала.

    :param int signum: номер сигнала
    :param str target: получатель сигнала, см. :class:`~Target`
    :param collections.Callable handler: обработчик сигнала
    :rtype: None
    """
    if not UWSGI:
        return

    if not isinstance(signum, int):
        raise TypeError("`int` is required, not `%s`" % type(signum))

    # значения вне границ мапятся на [0,255]
    if signum < 0 or signum > 255:
        raise ValueError("signum must be 0 <= signum <= 255, actual: %d " % signum)

    if not callable(handler):
        raise TypeError("handler must be callable")

    # см. коментарии к модулю
    if os.getpid() != uwsgi.masterpid():
        raise RuntimeError("Worker is not allowed to set signal handler")

    if uwsgi.signal_registered(signum):
        raise RuntimeError("Already registered %d" % signum)

    uwsgi.register_signal(signum, target, handler)


def is_signal_registered(signum):
    """Проверить установлен ли обработчик сигнала.

    :type signum: int
    :rtype: None | bool
    """
    if not UWSGI:
        return
    return uwsgi.signal_registered(signum)


def add_timer(signum, seconds):
    """Установить периодический таймер сигналов.

    Порождает сигналы типа `signum` каждые `seconds` секунд.

    :type signum: int
    :type seconds: int
    :rtype: None
    """
    if not UWSGI:
        return

    if seconds <= 0:
        raise ValueError("`seconds` must be greater then 0, actual: %d" % seconds)

    if not uwsgi.signal_registered(signum):
        raise RuntimeError("Signal is not registered")

    uwsgi.add_timer(signum, seconds)
