# -*- coding: utf-8 -*-
import itertools

from mpfs.common import errors
from mpfs.config import settings
from mpfs.core.bus import Bus
from mpfs.core.user.base import User
from mpfs.core.address import Address, ResourceId
from mpfs.core.factory import get_resources_by_resource_ids
from mpfs.metastorage.mongo.collections.filesystem import is_quick_move_enabled


SETPROP_SYMLINK_FIELDNAME = settings.system['setprop_symlink_fieldname']


def _version_diff_item_to_delta_item(version_diff_item):
    """Из элемента версионного diff формируем элемент delta
    """
    result = {
        'change_type': version_diff_item['op'],
        'path': version_diff_item['key'],
        'resource_id': version_diff_item['resource_id'],
        'type': version_diff_item['type'],
        'revision': version_diff_item['version'],
    }

    if 'new_key' in version_diff_item:
        result['new_path'] = version_diff_item['new_key']
    if 'resource' in version_diff_item:
        resource = version_diff_item['resource']
        if SETPROP_SYMLINK_FIELDNAME in resource.meta:
            result['discsw_symlink'] = resource.meta[SETPROP_SYMLINK_FIELDNAME]
        result['resource'] = resource
    #
    if 'shared' in version_diff_item:
        result['group'] = {
            'is_owned': version_diff_item['shared'] == 'owner',
            'rights': version_diff_item['rights']
        }

    return result


def _extend_version_diff(uid, version_diff, skip_resource_if_not_found=False):
    """Расширяем ответ версионного диффа

    Для `new` и `changed` изменений добавляем mpfs-ые ресурсы
    Меняется исходный документ
    """
    new_or_changed_docs = [i for i in version_diff['result'] if i['op'] != 'deleted']
    resource_ids = [ResourceId.parse(i['resource_id']) for i in new_or_changed_docs]
    resources = get_resources_by_resource_ids(uid, resource_ids)
    for new_or_changed_doc, resource in itertools.izip(new_or_changed_docs, resources):
        if resource is None:
            if not skip_resource_if_not_found:
                raise ValueError("Resource '%s' not found" % new_or_changed_doc['resource_id'])
        else:
            # меняем документ по ссылке
            new_or_changed_doc['resource'] = resource
    return version_diff


def deltas(uid, base_revision, allow_quick_move_deltas):
    user = User(uid)
    user_info = user.info()
    enable_410 = user_info.get('settings', {}).get('deltas', {}).get('enable_410', False)
    if enable_410:
        raise errors.StorageVersionNotFound()

    version_diff = Bus().diff(
        uid,
        Address.Make(uid, '/disk').id,
        base_revision,
        allow_quick_move_deltas,
    )

    should_skip_resource_if_not_found = allow_quick_move_deltas and is_quick_move_enabled(uid)
    version_diff = _extend_version_diff(uid, version_diff, skip_resource_if_not_found=should_skip_resource_if_not_found)

    deltas_result = {
        'items': [],
        'revision': version_diff['version']
    }

    for version_diff_item in version_diff['result']:
        deltas_result['items'].append(_version_diff_item_to_delta_item(version_diff_item))

    return deltas_result
