# -*- coding: utf-8 -*-

from copy import copy

import mpfs.engine.process
from mpfs.common import errors

from mpfs.common.util import from_json, to_json
from mpfs.common.errors import SnapshotInfiniteLoopDetected
from mpfs.common.util.crypt import AesCbcCryptAgent, DEFAULT_SECRET
from mpfs.common.util.filetypes import getGroupByName
from mpfs.common.util.urls import quote
from mpfs.config import settings
from mpfs.core.address import ResourceId, Address
from mpfs.core.factory import get_service_by_root_folder_path
from mpfs.core.metastorage.decorators import user_exists_and_not_blocked
from mpfs.core.services.disk_service import Disk
from mpfs.core.services.rate_limiter_service import rate_limiter
from mpfs.core.social.share import LinkToGroup, Group
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH, ATTACH_AREA_PATH, TRASH_AREA_PATH
from mpfs.metastorage.mongo.util import generate_version_number

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

RATE_LIMITER_GROUP_NAMES_SNAPSHOT = settings.rate_limiter['group_names']['snapshot']
SNAPSHOT_CHUNK_SIZE = settings.snapshot['chunk_size']
SNAPSHOT_SHARED_FOLDERS_TIME_LIMIT = settings.snapshot['shared_folders']['time_limit']
SNAPSHOT_CURRENT_PROTOCOL_VERSION = settings.snapshot['current_protocol_version']
SNAPSHOT_UIDS_TO_SWITCH_PROTOCOL_VERSION_FOR = settings.snapshot['uids_to_switch_protocol_version_for']
SETPROP_SYMLINK_FIELDNAME = settings.system['setprop_symlink_fieldname']
# `mtime` для системных папок проставляется в `Folder.__init__()`
# и берется из конфигов.
SYSTEM_FOLDER_DETAILS = settings.folders


class SnapshotIterationKey(object):
    _crypt_agent = AesCbcCryptAgent(DEFAULT_SECRET)
    DELIMITER = ';'
    INNER_DELIMITER = ','
    REVISION = 0
    FILE_ID = ''
    LAST_IDS = []
    ALL_PROCESSED_GIDS = set()
    LAST_FILE_TYPE = ''
    LEFT_COLLECTIONS = None
    "Содержит коллеции, которые осталось обойти. Если None, то будет работать только с /disk"
    LENGTH = 5

    def __init__(self):
        self._revision = self.REVISION
        self._file_id = self.FILE_ID
        self._last_ids = self._get_default_last_ids()
        self._all_processed_gids = self._get_default_processed_gids()
        self._last_file_type = self.LAST_FILE_TYPE
        self._left_collections = self._get_default_left_collections()
        self._remaining_gids = []
        self._shared_folders_iterations_started = False
        self._protocol_version = SNAPSHOT_CURRENT_PROTOCOL_VERSION
        self._uid = None
        self._session_id = None

    @classmethod
    def _get_default_last_ids(cls):
        return copy(cls.LAST_IDS)

    @classmethod
    def _get_default_processed_gids(cls):
        return copy(cls.ALL_PROCESSED_GIDS)

    @classmethod
    def _get_default_left_collections(cls):
        if cls.LEFT_COLLECTIONS is None:
            return None
        return copy(cls.LEFT_COLLECTIONS)

    def _initialize(self, iteration_key_dict):
        self._revision = iteration_key_dict['revision']
        self._file_id = iteration_key_dict['file_id']
        self._last_ids = iteration_key_dict.get('last_ids') or self._get_default_last_ids()
        self._last_file_type = iteration_key_dict.get('last_file_type') or self.LAST_FILE_TYPE
        self._remaining_gids = iteration_key_dict.get('remaining_gids') or []
        self._shared_folders_iterations_started = iteration_key_dict.get('shared_folders_iterations_started', False)
        self._uid = iteration_key_dict.get('uid')
        self._session_id = iteration_key_dict.get('session_id')
        self._protocol_version = 2

    def serialize(self):
        if self.is_default():
            return None
        if self._protocol_version == 2:
            iteration_key_dict = {
                'revision': self._revision,
                'file_id': self._file_id,
                'last_ids': self._last_ids,
                'last_file_type': self._last_file_type,
                'protocol_version': self._protocol_version,
                'remaining_gids': self._remaining_gids,
                'shared_folders_iterations_started': self._shared_folders_iterations_started,
                'uid': self._uid,
                'session_id': self._session_id,
            }
            if self._left_collections is not None:
                iteration_key_dict['left_collections'] = self._left_collections
            return self._crypt_agent.encrypt(to_json(iteration_key_dict))
        raise NotImplementedError('Unsupported protocol version %s' % self._protocol_version)

    def is_default(self):
        if self._file_id != self.FILE_ID:
            return False
        if self._revision != self.REVISION:
            return False
        if self._last_ids != self.LAST_IDS:
            return False
        if self._all_processed_gids != self.ALL_PROCESSED_GIDS:
            return False
        if self._last_file_type != self.LAST_FILE_TYPE:
            return False

        return True

    @classmethod
    def parse(cls, value):
        if not value:
            return cls()

        iteration_key_dict = from_json(cls._crypt_agent.decrypt(value))
        key = cls()
        key._initialize(iteration_key_dict)
        return key

    def get_file_id(self):
        return self._file_id

    def set_file_id(self, file_id):
        self._file_id = file_id

    def get_revision(self):
        return self._revision

    def set_revision(self, revision):
        if self._revision:
            raise RuntimeError("Already set")
        self._revision = revision

    def set_last_ids(self, last_ids):
        self._last_ids = copy(last_ids)

    def get_last_ids(self):
        return copy(self._last_ids)

    def add_processed_gid(self, gid):
        self._all_processed_gids.add(gid)

    def get_all_processed_gids(self):
        return copy(self._all_processed_gids)

    def get_last_file_type(self):
        return self._last_file_type

    def set_last_file_type(self, last_file_type):
        self._last_file_type = last_file_type

    def get_left_collections(self):
        return copy(self._left_collections)

    def set_left_collections(self, left_collections):
        self._left_collections = left_collections

    def get_remining_gids(self):
        return copy(self._remaining_gids)

    def set_remaining_gids(self, remaining_gids):
        self._remaining_gids = remaining_gids

    def get_shared_folders_iteration_started(self):
        return self._shared_folders_iterations_started

    def set_shared_folders_iteration_started(self, val):
        self._shared_folders_iterations_started = val

    def get_protocol_version(self):
        return self._protocol_version

    def get_uid(self):
        return self._uid

    def set_uid(self, uid):
        self._uid = uid

    def get_session_id(self):
        return self._session_id

    def set_session_id(self, session_id):
        self._session_id = session_id

    def init_revision(self, uid):
        self.set_revision(Disk().get_version(uid))

    def reset_cursor(self):
        pass


class IndexerSnapshotIterationKey(SnapshotIterationKey):
    LEFT_COLLECTIONS = ['/disk', PHOTOUNLIM_AREA_PATH, ATTACH_AREA_PATH, TRASH_AREA_PATH]
    LENGTH = 6

    def init_revision(self, uid):
        self.set_revision(generate_version_number())

    def reset_cursor(self):
        if self._left_collections:
            self._left_collections.pop()
        if not self._left_collections:
            # последним обходится /disk, после которого обходятся ОП пользователя, резетить ничего больше не надо
            return
        self._file_id = self.FILE_ID
        self._last_ids = self.LAST_IDS
        self._last_file_type = self.LAST_FILE_TYPE

    def _initialize(self, iteration_key_dict):
        super(IndexerSnapshotIterationKey, self)._initialize(iteration_key_dict)
        self._left_collections = iteration_key_dict.get('left_collections') or []


@user_exists_and_not_blocked
def snapshot(req):
    """
    Query string аргументы:
      * uid [обязательный]\
      * session_id [необязательный] - нужен для возможности лимитирования, требуется только при первом запросе, после
      будет зашит в ключ, при отсутствии игнорируем.
    Body аргументы:
      * iteration_key
    """
    uid = req.uid
    data = _get_request_body(req)

    raw_iteration_key = data.get('iteration_key')
    iteration_key = SnapshotIterationKey.parse(raw_iteration_key)

    if iteration_key.is_default() and uid and req.session_id:
        iteration_key.set_uid(uid)
        iteration_key.set_session_id(req.session_id)

    if iteration_key.get_uid() and iteration_key.get_session_id() and raw_iteration_key:
        encoded_iter_key = quote(raw_iteration_key)
        rate_limiter.check_limit_exceeded(RATE_LIMITER_GROUP_NAMES_SNAPSHOT, encoded_iter_key,
                                          error_class=errors.RequestsLimitExceeded429)

    if iteration_key.is_default():
        _set_revision(uid, iteration_key)

    iteration_key, chunk = _get_next_snapshot_chunk(uid, iteration_key)
    return {'iteration_key': iteration_key.serialize(),
            'items': BaseSnapshotChunkSerializer().serialize(chunk),
            'revision': iteration_key.get_revision()}


@user_exists_and_not_blocked
def indexer_snapshot(req):
    """
    Query string аргументы:
      * uid [обязательный]
    Body аргументы:
      * iteration_key
    """
    uid = req.uid
    data = _get_request_body(req)

    raw_iteration_key = data.get('iteration_key')
    iteration_key = IndexerSnapshotIterationKey.parse(raw_iteration_key)
    if iteration_key.is_default():
        _set_revision(uid, iteration_key)

    iteration_key, chunk = _get_next_snapshot_chunk(uid, iteration_key)
    return {'iteration_key': iteration_key.serialize(),
            'items': IndexerSnapshotChunkSerializer().serialize(chunk),
            'revision': iteration_key.get_revision()}


def _get_request_body(req):
    """Получить тело запроса.

    :rtype: dict
    """
    data = {}
    if req.http_req.data:
        data = from_json(req.http_req.data)
        if data is None:
            data = {}
    return data


def _set_revision(uid, iteration_key):
    """Установить текущую ревизию диска пользователя,
    если в ключе итерации еще не была установлена ревизия.

    :type uid: str
    :type iteration_key: :class:`~SnapshotIterationKey`
    """
    revision = iteration_key.get_revision()
    if not revision:
        iteration_key.init_revision(uid)


def _get_next_snapshot_chunk(uid, iteration_key):
    """Вернуть обновленный ключ итерации и следующий чанк снепшота
    из диска пользователя или ОП.

    :raise SnapshotInfiniteLoopDetected: если очередной чанк заканчивается на file_id
            который был получен в предыдущем чанке. В таком случае прогресса нет.

    :type uid: str
    :type iteration_key: :class:`~SnapshotIterationKey`
    :rtype: tuple[:class:`~SnapshotIterationKey`, list[dict]]
    """
    remaining_gids = set()
    chunk = []
    if iteration_key.get_remining_gids() and iteration_key.get_shared_folders_iteration_started():
        remaining_gids, last_folder_file_id, chunk = find_snapshot_chunk_from_share(
            uid, iteration_key.get_remining_gids(), iteration_key.get_file_id())
        iteration_key.set_remaining_gids(remaining_gids)
        iteration_key.set_file_id(last_folder_file_id)
    elif not iteration_key.get_shared_folders_iteration_started():
        file_id = iteration_key.get_file_id()
        last_ids = iteration_key.get_last_ids()
        last_file_type = iteration_key.get_last_file_type()

        left_collections = iteration_key.get_left_collections()
        if left_collections:
            current_service = get_service_by_root_folder_path(left_collections[-1])
        else:
            current_service = Disk()

        new_file_id, new_last_ids, new_last_file_type, chunk = \
            find_snapshot_chunk_from_disk(uid, file_id, last_ids, last_file_type, current_service)
        iteration_key.set_file_id(new_file_id)
        iteration_key.set_last_ids(new_last_ids)
        iteration_key.set_last_file_type(new_last_file_type)
        if new_file_id and file_id == new_file_id:
            raise SnapshotInfiniteLoopDetected()

        if not chunk:
            iteration_key.reset_cursor()
            if not iteration_key.get_left_collections():
                remaining_gids = _get_remaining_gids(uid)
                remaining_gids, last_folder_file_id, chunk = find_snapshot_chunk_from_share(
                    uid, remaining_gids, iteration_key.get_file_id())

                iteration_key.set_remaining_gids(remaining_gids)
                iteration_key.set_file_id(last_folder_file_id)
                iteration_key.set_shared_folders_iteration_started(True)

    if not chunk and not remaining_gids and not iteration_key.get_left_collections():
        iteration_key = SnapshotIterationKey()

    return iteration_key, chunk


def find_snapshot_chunk_from_disk(uid, file_id, last_ids, last_file_type, resource_service):
    """Получить чанк ресурсов пользователя без учета ОП.

    :type uid: str
    :type file_id: str
    :type last_ids: list
    :type last_file_type: str
    :type resource_service: MPFSStorageService
    :rtype: tuple[str, str, list[tuple[str, dict]]]
    """
    chunk = resource_service.find_snapshot_chunk(uid, file_id, last_ids, last_file_type, SNAPSHOT_CHUNK_SIZE)
    new_file_id = ''
    new_last_ids = ''
    new_last_file_type = ''
    docs = []

    if chunk:
        _, doc = chunk[-1]
        new_last_file_type = doc['type']
        new_file_id = doc['data']['meta']['file_id']
        new_last_ids = [doc_id for doc_id, doc in chunk if doc['data']['meta']['file_id'] == new_file_id]

        path_to_group_mapping = {group.path: group for group in Group.iter_all(uid)}
        path_to_grouplink_mapping = {link.path: link for link in LinkToGroup.iter_all(uid)}

        for _, doc in chunk:
            path = doc['key']
            if path in path_to_group_mapping:
                doc['data']['meta']['group'] = {'is_owned': 1}
            elif path in path_to_grouplink_mapping:
                doc['data']['meta']['group'] = {'is_owned': 0, 'rights': path_to_grouplink_mapping[path].rights}
            docs.append(doc)

    return new_file_id, new_last_ids, new_last_file_type, docs


def find_snapshot_chunk_from_share(uid, remaining_gids, last_folder_file_id=None):
    """Найти полный снепшот поддерева очередной ОП, если такая найдется.

    :type uid: str
    :type processed_gids: set
    :rtype: tuple[str, list[dict]]
    """
    if not remaining_gids:
        return remaining_gids, '', []

    current_gid = remaining_gids[0]
    link = next(link for link in LinkToGroup.iter_all(uid) if link.gid == current_gid)

    disk_service = Disk()
    group = link.group
    chunk, last_folder_file_id = disk_service.find_snapshot_from_subtree(group.owner, group.path,
                                                                         SNAPSHOT_SHARED_FOLDERS_TIME_LIMIT,
                                                                         SNAPSHOT_CHUNK_SIZE,
                                                                         last_folder_file_id)
    if last_folder_file_id is None:
        remaining_gids = remaining_gids[1:]
        last_folder_file_id = ''
    return remaining_gids, last_folder_file_id, [(link, doc) for _, doc in chunk],


def _get_remaining_gids(uid):
    return sorted({link.gid for link in LinkToGroup.iter_all(uid)})


class BaseSnapshotChunkSerializer(object):
    """Базовый класс сериализаторов чанков снепшота.

    Чанки содержат либо документы из диска пользователя,
    либо документы из ОП, доступной пользователю.
    """
    def serialize(self, chunk):
        """Сериализовать ``chunk`` снепшота.

        :type chunk: list[dict | tuple[:class:`~LinkToGroup`, dict]]
        :return: list[dict]
        """
        result = []
        for item in chunk:
            if len(item) == 2 and isinstance(item[0], LinkToGroup):
                resource = self.serialize_document_from_share(*item)
            else:
                resource = self.serialize_document(item)
            result.append(resource)
        return result

    def serialize_document(self, document):
        """Сериализовать документ из диска пользователя.

        :type document: dict
        :rtype: dict
        """
        result = {
            'path': document['key'],
            'type': document['type'],
            'meta': {
                'revision': long(document['version'])
            }
        }

        document_meta = document['data']['meta']

        if result['type'] == 'file':
            result['meta']['size'] = document['data']['size']
            result['meta']['md5'] = document_meta['md5']
            result['meta']['sha256'] = document_meta['sha256']

        address = Address.Make(document['uid'], result['path'])
        if address.is_storage:
            # Корневые папки не имеют `data.mtime`, она проставляется в `Folder.__init__`
            # и берется из конфигов.
            result['mtime'] = long(SYSTEM_FOLDER_DETAILS[result['path']]['mtime'])
        elif result['type'] == 'dir' and 'mtime' not in document['data']:
            pass
        else:
            result['mtime'] = long(document['data']['mtime'])

        file_id = document_meta['file_id']
        result['meta']['resource_id'] = ResourceId(document['uid'], file_id).serialize()

        if ('public' in document_meta and document_meta['public'] == 1 and 'short_url' in document_meta):
            result['meta']['short_url'] = document_meta['short_url']

        if SETPROP_SYMLINK_FIELDNAME in document_meta:
            result['meta']['discsw_symlink'] = document_meta[SETPROP_SYMLINK_FIELDNAME]

        group_info = document['data']['meta'].get('group')
        if group_info:
            serialized_group_info = {
                'is_owned': group_info['is_owned'],
            }
            if 'rights' in group_info:
                serialized_group_info['rights'] = group_info['rights']
            result['meta']['group'] = serialized_group_info

        return result

    def serialize_document_from_share(self, link, document):
        """Сериализовать документ, полученный из поддерева ОП.

        :type document: dict
        :type link: :class:`~LinkToGroup`
        :rtype: dict
        """
        serialized = self.serialize_document(document)
        serialized['path'] = link.get_link_path(serialized['path'])
        return serialized


class IndexerSnapshotChunkSerializer(BaseSnapshotChunkSerializer):

    def serialize_document(self, document):
        serialized = super(IndexerSnapshotChunkSerializer, self).serialize_document(document)
        if serialized['type'] == 'file':
            serialized['meta']['stid'] = document['data']['meta']['file_mid']
            serialized['meta']['media_type'] = document['data'].get('media_type')
            if serialized['meta']['media_type'] is None:
                serialized['meta']['media_type'] = getGroupByName(serialized['path'], serialized['meta'].get('mimetype'))
        return serialized
