# -*- coding: utf-8 -*-
import time
from datetime import datetime

from mpfs.common.util import hashed
from mpfs.core.office.static import OfficeAccessStateConst
from mpfs.dao.base import BaseDAO, BaseDAOItem, PostgresBaseDAOImplementation, MongoBaseDAOImplementation
from mpfs.dao.fields import (DateTimeField, MSKDateTimeField, Md5Field,
                             IntegerField, NullableStringField, ResourceTypeField,
                             StringField, UidField, FileIdField, EnumField, BoolField, IntArrayField)
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import (
    SQL_CREATE_LINK_DATA_ITEM,
    SQL_LINK_DATA_UPDATE_OFFICE_ACCESS_STATE,
    SQL_LINK_DATA_UPDATE_OFFICE_FIELDS,
    SQL_LINK_DATA_UPDATE_OFFICE_DOC_SHORT_ID,
    SQL_GET_PREVIOUS_OFFICE_DOC_SHORT_ID, SQL_FIND_EXPIRED_LINKS
)
from mpfs.metastorage.postgres.schema import link_data, OfficeActionState


class LinkDataDAOItem(BaseDAOItem):
    mongo_collection_name = 'link_data'
    postgres_table_obj = link_data
    is_sharded = True

    id = Md5Field(mongo_path='_id', pg_path=link_data.c.id)
    uid = UidField(mongo_path='uid', pg_path=link_data.c.uid)
    path = StringField(mongo_path='key', pg_path=link_data.c.path)
    version = IntegerField(mongo_path='version', pg_path=link_data.c.version, default_value=None)
    user_ip = NullableStringField(mongo_path='data.user_ip', pg_path=link_data.c.user_ip, default_value=None)
    public_uid = UidField(mongo_path='data.public_uid', pg_path=link_data.c.public_uid, default_value=None)
    target = NullableStringField(mongo_path='data.tgt', pg_path=link_data.c.target, default_value=None)
    type = ResourceTypeField(mongo_path='type', pg_path=link_data.c.type)
    parent = Md5Field(mongo_path='parent', pg_path=link_data.c.parent, default_value=None)
    date_created = DateTimeField(mongo_path='data.ctime', pg_path=link_data.c.date_created, default_value=None)
    date_modified = DateTimeField(mongo_path='data.mtime', pg_path=link_data.c.date_modified, default_value=None)
    date_deleted = MSKDateTimeField(mongo_path='data.dtime', pg_path=link_data.c.date_deleted, default_value=None)
    file_id = FileIdField(mongo_path='data.file_id', pg_path=link_data.c.file_id, default_value=None)
    resource_id_uid = UidField(mongo_path='data.resource_id_uid', pg_path=link_data.c.resource_id_uid, default_value=None)
    office_access_state = EnumField(mongo_path='data.office_access_state',
                                    pg_path=link_data.c.office_access_state,
                                    default_value=None,
                                    enum_class=OfficeActionState)
    office_doc_short_id = NullableStringField(mongo_path='data.office_doc_short_id',
                                              pg_path=link_data.c.office_doc_short_id,
                                              default_value=None)
    password = StringField(mongo_path='data.password', pg_path=link_data.c.password, default_value=None)
    read_only = BoolField(mongo_path='data.read_only', pg_path=link_data.c.read_only, default_value=None)
    available_until = DateTimeField(mongo_path='data.available_until', pg_path=link_data.c.available_until, default_value=None)
    external_organization_ids = IntArrayField(mongo_path='data.external_organization_ids',
                                              pg_path=link_data.c.external_organization_ids,
                                              default_value=None)


    exclude_keys_after_conversion_to_mongo = {
        'data': {
            'ctime': None,
            'dtime': None,
            'mtime': None,
            'public_uid': None,
            'tgt': None,
            'user_ip': None,
            'file_id': None,
            'resource_id_uid': None,
        },
        'parent': None,
        'version': None,
    }

    def get_mongo_representation(self, skip_missing_fields=False):
        mongo_dict = super(LinkDataDAOItem, self).get_mongo_representation(skip_missing_fields=skip_missing_fields)
        if 'data' not in mongo_dict:
            mongo_dict['data'] = {}  # workaround для унификации возвращаемых из монги и из постгреса данных
        return mongo_dict

    @classmethod
    def build_id(cls, uid, key):
        return hashed(str(uid) + ':' + cls.build_key(key))

    @staticmethod
    def build_key(key):
        return '/' + '/'.join(filter(None, key.split('/')))


class LinkDataDAO(BaseDAO):
    dao_item_cls = LinkDataDAOItem

    def __init__(self):
        super(LinkDataDAO, self).__init__()
        self._pg_impl = PostgresLinkDataDAOImplementation(self.dao_item_cls)
        self._mongo_impl = MongoLinkDataDAOImplementation(self.dao_item_cls)

    def create(self, uid, dao_item):
        return self._get_impl(uid).create(dao_item)

    def update_office_fields(self, uid, link_data_path, data):
        return self._get_impl(uid).update_office_fields(uid, link_data_path, data)

    def find_previous_office_doc_short_id(self, uid, file_id):
        return self._get_impl(uid).find_previous_office_doc_short_id(uid, file_id)

    def find_expired(self, shard_endpoint, expiration_dt):
        return self._get_impl_by_shard_endpoint(shard_endpoint).find_expired(shard_endpoint, expiration_dt)


class MongoLinkDataDAOImplementation(MongoBaseDAOImplementation):
    def create(self, item):
        coll = self.get_collection_by_uid(item.uid)
        coll.insert(item.get_mongo_representation(skip_missing_fields=True))

    def update_office_fields(self, uid, link_data_path, data):
        raise NotImplementedError()


class PostgresLinkDataDAOImplementation(PostgresBaseDAOImplementation):
    def __preprocess(self, document):
        if 'data' in document:
            if 'user_ip' in document['data'] and document['data']['user_ip'] == '':
                document['data']['user_ip'] = None

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, list):
            for doc in doc_or_docs:
                self.__preprocess(doc)
        else:
            self.__preprocess(doc_or_docs)
        return super(PostgresLinkDataDAOImplementation, self).insert(doc_or_docs, manipulate, continue_on_error, **kwargs)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        self.__preprocess(document)
        return super(PostgresLinkDataDAOImplementation, self).update(spec, document, upsert, multi, **kwargs)

    def create(self, item):
        params = {c.name: v for c, v in item.get_postgres_representation().iteritems()}
        session = Session.create_from_uid(params['uid'])
        session.execute(SQL_CREATE_LINK_DATA_ITEM, params)

    def update_office_fields(self, uid, link_data_path, data):
        params = {'path': LinkDataDAOItem.build_key(link_data_path), 'uid': uid}
        if 'office_access_state' in data and 'office_doc_short_id' in data:
            query = SQL_LINK_DATA_UPDATE_OFFICE_FIELDS
            params['office_access_state'] = data['office_access_state']
            params['office_doc_short_id'] = data['office_doc_short_id']
        elif 'office_access_state' in data:
            query = SQL_LINK_DATA_UPDATE_OFFICE_ACCESS_STATE
            params['office_access_state'] = data['office_access_state']
        elif 'office_doc_short_id' in data:
            query = SQL_LINK_DATA_UPDATE_OFFICE_DOC_SHORT_ID
            params['office_doc_short_id'] = data['office_doc_short_id']
        else:
            raise NotImplementedError()

        session = Session.create_from_uid(uid)
        session.execute(query, params)

    def find_previous_office_doc_short_id(self, uid, file_id):
        pg_file_id = LinkDataDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', file_id)[1]
        pg_uid = LinkDataDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1]
        session = Session.create_from_uid(uid)
        result = session.execute(SQL_GET_PREVIOUS_OFFICE_DOC_SHORT_ID, {'uid': pg_uid, 'file_id': pg_file_id}).fetchone()
        if result:
            return result

    def find_expired(self, shard_endpoint, expiration_dt):
        session = Session.create_from_shard_endpoint(shard_endpoint)
        result = session.execute(SQL_FIND_EXPIRED_LINKS, {'now': expiration_dt})
        if result:
            return [self.doc_to_item(doc) for doc in result]
