# -*- coding: utf-8 -*-
"""

MPFS
CORE

Публикатор файлов

"""
import itertools
import os
import time
import traceback
from urlparse import urlparse, urlunparse

import mpfs.engine.process
import mpfs.common.util.logger as logger
from mpfs.common.errors import ResourceNotFound
from mpfs.common.util.experiments.logic import change_experiment_context_with

from mpfs.config import settings
from mpfs.common.util import mailer, hashed, filter_uid_by_percentage
from mpfs.common import errors
from mpfs.core.address import ResourceId
from mpfs.core.albums.errors import AlbumItemMissed
from mpfs.core.albums.logic.common import get_resource_from_album_item, get_album_by_public_key_unchecked
from mpfs.core.bus import Bus
from mpfs.core.event_history.logger import CatchHistoryLogging
from mpfs.core.factory import get_resource, get_resources
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.global_gallery.logic.controller import GlobalGalleryController
from mpfs.core.office.logic.sharing_url import sync_office_fields_from_link_data, sync_public_fields_from_link_data
from mpfs.core.office.static import OfficeClientIDConst, EditorConst, OfficeAccessStateConst
from mpfs.core.office.util import (
    generate_office_doc_short_id,
    SharingURLAddressHelper,
    build_office_online_url,
    get_editor,
)
from mpfs.core.services.passport_service import Passport
from mpfs.core.services.lenta_loader_service import lenta_loader
from mpfs.core.services.logreader_service import LogreaderService
from mpfs.core.services.mulca_service import Mulca
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.address import Address, PublicAddress, SymlinkAddress
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.filesystem.resources.disk import DiskFolder, append_meta_to_office_files
from mpfs.common.util.crypt import CryptAgent
from mpfs.core.user.base import User
from mpfs.core.filesystem.quota import Quota
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.video_service import video
from mpfs.common.static.tags import *
from mpfs.core.metastorage.decorators import user_exists_and_not_blocked
from mpfs.common.static import tags
from mpfs.core.filesystem.events import FilesystemSetPublicEvent, FilesystemSetPrivateEvent
from mpfs.core.filesystem.resources.disk import MPFSFile
from mpfs.core.event_history.logger import SAVE_FILE_FROM_PUBLIC_LOG_MESSAGE_RE
from mpfs.core.user.constants import PUBLIC_UID, YATEAM_DIR_PATH
from mpfs.core.yateam.logic import check_yateam_subtree_access, is_yateam_subtree, user_has_nda_rights


FEATURE_TOGGLES_BLOCKINGS_ON_OVERDRAW = settings.feature_toggles['blockings_on_overdraw']
FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED = settings.feature_toggles['setting_yarovaya_mark_enabled']
FEATURE_TOGGLES_DISABLE_PUBLIC_LINKS_PERCENTAGE = settings.feature_toggles['disable_public_links_percentage']
FEATURE_TOGGLES_ALLOW_PUBLIC_WHILE_LOCKED_BY_OFFICE = settings.feature_toggles['allow_public_while_locked_by_office']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
listing_log = logger.get('stat-listing')

crypt_agent = CryptAgent()
passport = Passport()
logreader = LogreaderService()
quota = Quota()
mulca = Mulca()
push_queue = PushQueue()


class Publicator(object):
    """
    Класс публикатора
    Обладает методами выставления ресурса публичным или приватным
    Позволяет получить/скопировать данные по секретному хешу
    """

    NDA_SHORT_URL_FORMAT = 'https://disk.yandex.ru/public/nda/?%s'

    def __init__(self, *args, **kwargs):
        self.request = None
        for k, v in kwargs.iteritems():
            setattr(self, k, v)
        self.fs = Bus(request=self.request)

    @staticmethod
    def _get_nda_short_url(resource):
        public_url = resource.get_public_url()
        parsed = urlparse(public_url)
        return Publicator.NDA_SHORT_URL_FORMAT % parsed.query

    @staticmethod
    def replace_yadisk_cc_url(url):
        """В случае если переданный URL в домене yadisk.cc, то преобразует его URL
        в домене yadi.sk. В противном случае возвращает URL как есть. Дополнительно возвращает
        флаг о том был ли преобразован URL или нет.
        """
        # https://st.yandex-team.ru/CHEMODAN-36162
        parsed_url = urlparse(url)
        if parsed_url.netloc != 'yadisk.cc':
            return False, url

        parsed_url = parsed_url._replace(netloc='yadi.sk')
        return True, urlunparse(parsed_url)

    @staticmethod
    def get_short_url(resource):
        if is_yateam_subtree(resource.address.path) and user_has_nda_rights(resource.uid):
            return Publicator._get_nda_short_url(resource)
        else:
            return resource.get_short_url()

    @staticmethod
    def get_short_url_named(resource):
        if is_yateam_subtree(resource.address.path) and user_has_nda_rights(resource.uid):
            return Publicator._get_nda_short_url(resource)
        else:
            return resource.get_short_url_named()

    def set_public(self, uid, rawaddress, **kwargs):
        """
        Публикация ресурса: делаем симлинк и ставим публичный статус
        """
        tgt_address = Address(rawaddress)
        if uid != tgt_address.uid:
            raise ValueError("Uid mismatch: '%s' != '%s'" % (uid, tgt_address.id))

        parents_addresses = [a for a in tgt_address.get_parents() if not a.is_root and a.storage_path == '/disk']
        resources = get_resources(tgt_address.uid, parents_addresses + [tgt_address])
        for address, r in zip(parents_addresses, resources[:-1]):
            if r is None:
                raise ResourceNotFound(address.id)
            r.assert_blocked()

        tgt_resource = resources[-1]
        if tgt_resource.visible_address.id != tgt_address.id:
            raise ResourceNotFound(
                    "Addresses mismatch: '%s' != '%s'" % (tgt_resource.visible_address.id, tgt_address.id)
            )

        if tgt_resource.is_file:
            hid = getattr(tgt_resource, 'hid', None)
            if hid:
                self.fs.check_hids_blockings([hid])
            if tgt_resource.is_infected():
                raise errors.ViralFilesNotAllowedToPublicate(rawaddress)

        tgt_resource.check_rw(operation='set_public')
        tgt_resource.assert_blocked()

        if FEATURE_TOGGLES_BLOCKINGS_ON_OVERDRAW:
            user = User(tgt_resource.uid)
            user.check_overdrawn()

        if tgt_resource.is_fully_public():
            result = {
                'hash': tgt_resource.get_public_hash(),
                'url': tgt_resource.get_public_url(),
                'short_url': Publicator.get_short_url(tgt_resource),
                'short_url_named': Publicator.get_short_url_named(tgt_resource),
            }
        else:
            resync_public = False
            if FEATURE_TOGGLES_ALLOW_PUBLIC_WHILE_LOCKED_BY_OFFICE and tgt_resource.is_file:
                lock = LockHelper.get_lock(tgt_address.id)
                if lock and lock.get('data', {}).get('op_type') == 'office':
                    resync_public = True
                elif lock:
                    self.fs.check_lock(tgt_address.id)
            else:
                self.fs.check_lock(tgt_address.id)
            user_ip = kwargs.get('user_ip', '')
            result = self.make_one_element_public(resource=tgt_resource, user_ip=user_ip, tld=kwargs.get('tld', 'ru'))
            if FEATURE_TOGGLES_ALLOW_PUBLIC_WHILE_LOCKED_BY_OFFICE and resync_public:
                job_data = {
                    'uid': tgt_resource.owner_uid,
                    'file_id': tgt_resource.meta['file_id']
                }
                deduplication_id = '%s_%s_%s' % ('sync_public_fields_from_link_data', tgt_resource.owner_uid, tgt_resource.meta['file_id'])
                mpfs_queue.put(job_data, 'sync_public_fields_from_link_data', deduplication_id=deduplication_id, delay=65)
            if tgt_resource.is_group_root or tgt_resource.is_shared_root:
                job_data = {
                    'gid': tgt_resource.group.gid,
                    'actor': tgt_resource.uid
                }
                mpfs_queue.put(job_data, 'make_shared_root_public')
            # Создаем событие только при реальной публикации,
            # а не при отдаче информации о уже опубликованном ресурсе
            FilesystemSetPublicEvent(uid=uid, tgt_resource=tgt_resource,
                                     tgt_address=Address(rawaddress),
                                     type=kwargs.get('oper_type', None),
                                     subtype=kwargs.get('oper_subtype', None)).send_self_or_group(resource=tgt_resource)
        self._append_office_info(result, tgt_resource, kwargs.get('tld', 'ru'), actor_uid=uid)
        return result

    def set_public_settings(self, uid, rawaddress,  data):
        fields = ['read_only', 'available_until', 'external_organization_id', 'password']
        if not data.viewkeys() & fields:
            raise errors.BadRequestError('empty body')
        if data.get('read_only') is True or data.get('available_until') is not None \
                or data.get('external_organization_id') or data.get('password') is not None:
            if not User(uid).get_public_settings_enabled():
                raise errors.PublicLinkSettingsDisabled()

        if 'available_until' in data:
            available_until = data['available_until']
            if available_until is not None and not isinstance(available_until, int):
                raise errors.BadRequestError()

        # пока нужно шарить только на одну оргу, но хотим сделать задел на будущее, поэтому перекладываем оргу в массив
        if 'external_organization_id' in data:
            external_organization_id = data['external_organization_id']
            if external_organization_id is None:
                data['external_organization_ids'] = None
            else:
                if not isinstance(external_organization_id, int):
                    raise errors.BadRequestError()
                else:
                    user_info = User(uid).get_user_info()
                    if external_organization_id not in user_info.get('external_organization_ids', []):
                        raise errors.Forbidden('you can create links only for organizations in which you are member')
                    data['external_organization_ids'] = [external_organization_id]

        if 'password' in data and data['password'] == '':
            raise errors.BadRequestError()

        resource = self.fs.resource(uid, rawaddress)
        resource.check_rw(operation='set_public_settings')
        if not resource.is_fully_public():
            raise errors.ResourceNotFound()
        symlink = Symlink.find(resource.address)
        if not symlink:
            raise errors.ResourceNotFound()
        if 'read_only' in data and data['read_only'] is True and symlink.get_office_access_state() == OfficeAccessStateConst.ALL:
            raise errors.Forbidden('Can\'t set read_only for office_access_state=all')

        if 'password' in data:
            data['password'] = symlink.hash_password(data['password'])

        symlink.update(data)

    def get_public_settings(self, uid, rawaddress=None, hashed_address=None):
        if hashed_address is None and rawaddress is None:
            raise errors.BadRequestError('Must be given path or hashed_address')
        if rawaddress:
            resource = self.fs.resource(uid, rawaddress)
        else:
            public_addr = PublicAddress(hashed_address)
            symlink_hash = crypt_agent.decrypt(public_addr.hash)
            resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        if not resource.is_fully_public():
            raise errors.ResourceNotFound()
        symlink = Symlink.find(resource.address)
        if not symlink:
            raise errors.ResourceNotFound()
        return {
            'have_password': symlink.have_password(),
            'read_only': symlink.get_read_only(),
            'available_until': symlink.get_available_until(),
            'external_organization_ids': symlink.get_external_organization_ids()
        }

    def _append_office_info(self, result, resource, tld, actor_uid):
        if not isinstance(resource, MPFSFile):
            return

        filename, ext = os.path.splitext(resource.name)
        ext = ext[1:].lower()
        SHARING_URL_SUPPORTED_EXT = {'docx', 'xlsx', 'pptx'}
        owner_editor = get_editor(resource.owner_uid)
        if (ext not in SHARING_URL_SUPPORTED_EXT or
                not owner_editor or
                actor_uid != resource.owner_uid and owner_editor.type_label != EditorConst.ONLY_OFFICE):
            result.pop('office_online_sharing_url', None)
            result.pop('office_access_state', None)
            return
        if (owner_editor.type_label != EditorConst.ONLY_OFFICE or
                actor_uid != resource.owner_uid and resource.office_access_state != OfficeAccessStateConst.ALL):
            result.pop('office_online_sharing_url', None)

        if not resource.office_doc_short_id:
            return

        if (owner_editor.type_label == EditorConst.ONLY_OFFICE and
                (actor_uid == resource.owner_uid or resource.office_access_state == OfficeAccessStateConst.ALL)):
            sharing_url_addr = SharingURLAddressHelper.build_sharing_url_addr(
                resource.owner_uid,
                resource.office_doc_short_id
            )
            result['office_online_sharing_url'] = build_office_online_url(
                client_id=OfficeClientIDConst.SHARING_URL,
                document_id=sharing_url_addr.serialize(),
                tld=tld
            )
        result['office_access_state'] = resource.office_access_state

    def make_one_element_public(self, uid=None, rawaddress=None, resource=None, user_ip=None, tld='ru'):
        if not resource:
            resource = self.fs.resource(uid, rawaddress)
        symlink = None
        if not resource.is_fully_public():
            try:
                symlink = self.fs.symlink_create(resource=resource, user_ip=user_ip)
                if resource.is_group_internal or resource.is_shared_internal:
                    uids = resource.group.all_uids()
                    old_version = resource._service.get_version(uids)
                else:
                    old_version = resource.prev_version
                resource.make_public(symlink)
                log.info("published %s %s" % (resource.type, resource.address.id))
                self.notify_xiva(resource, 'published', old_version=old_version)
            except Exception, e:
                error_log.info(traceback.format_exc())
                if symlink:
                    symlink.delete()
                resource.make_private()
                if isinstance(e, errors.ClckNoResponse):
                    raise e
                raise errors.PublicationError()
        if resource.is_fully_public():
            result = {
                'hash': resource.get_public_hash(),
                'url': resource.get_public_url(),
                'short_url': Publicator.get_short_url(resource),
                'short_url_named': Publicator.get_short_url_named(resource),
            }
            if symlink:
                prev = Symlink.find_previous_office_doc_short_id(symlink)
                if prev:
                    prev_office_access_state, prev_office_doc_short_id = prev
                    data = {}
                    if prev_office_doc_short_id:
                        data['office_doc_short_id'] = prev_office_doc_short_id
                    if prev_office_access_state:
                        data['office_access_state'] = prev_office_access_state
                    if data:
                        symlink.update_office_fields(data)
                        sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])
                elif isinstance(resource, MPFSFile):
                    _, ext = os.path.splitext(resource.name)
                    ext = ext[1:].lower()
                    SHARING_URL_SUPPORTED_EXT = {'docx', 'xlsx', 'pptx'}
                    if ext in SHARING_URL_SUPPORTED_EXT:
                        if symlink.get_office_doc_short_id() is None:
                            data_to_update = {'office_doc_short_id': generate_office_doc_short_id()}
                            symlink.update_office_fields(data_to_update)
                            sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])

                            sharing_url_addr = SharingURLAddressHelper.build_sharing_url_addr(
                                resource.owner_uid,
                                data_to_update['office_doc_short_id']
                            )
                            result['office_online_sharing_url'] = build_office_online_url(
                                client_id=OfficeClientIDConst.SHARING_URL,
                                document_id=sharing_url_addr.serialize(),
                                tld=tld
                            )
                            result['office_access_state'] = resource.office_access_state

            return result
        else:
            raise errors.PublicationError()

    def make_one_element_private(self, uid=None, rawaddress=None, resource=None, old_version=None):
        if not resource:
            resource = self.fs.resource(uid, rawaddress)
        if resource.is_smth_public():
            try:
                self.fs.remove_symlink(uid, rawaddress, resource=resource)
            except Exception:
                pass
            logreader.reset_counter(resource.meta.get('public_hash'))
            if FEATURE_TOGGLES_SETTING_YAROVAYA_MARK_ENABLED:
                resource.set_yarovaya_mark()
            resource.make_private()
            log.info("unpublished %s %s" % (resource.type, resource.address.id))
            if not old_version:
                old_version = resource.prev_version
            self.notify_xiva(resource, 'unpublished', old_version=old_version)

    def set_private(self, uid=None, rawaddress=None, resource=None, return_info=False):
        """
        Убирание публичности ресурса: снимаем статус, удаляем симлинк
        """
        if not resource:
            resource = self.fs.resource(uid, rawaddress)
        resource.check_rw(operation='set_private')

        if resource.is_smth_public():
            resync_public = False
            if FEATURE_TOGGLES_ALLOW_PUBLIC_WHILE_LOCKED_BY_OFFICE and resource.is_file:
                lock = LockHelper.get_lock(resource.address.id)
                if lock and lock.get('data', {}).get('op_type') == 'office':
                    resync_public = True
                elif lock:
                    self.fs.check_lock(resource.address.id)
            else:
                self.fs.check_lock(resource.address.id)
            try:
                self.fs.remove_symlink(uid, resource.storage_address.id)
            except Exception:
                error_log.warn(traceback.format_exc())
            if resource.is_group_internal or resource.is_shared_internal:
                uids = resource.group.all_uids()
                old_version = resource._service.get_version(uids)
            else:
                old_version = resource.prev_version
            self.make_one_element_private(resource=resource, old_version=old_version)

            if FEATURE_TOGGLES_ALLOW_PUBLIC_WHILE_LOCKED_BY_OFFICE and resync_public:
                job_data = {
                    'uid': resource.owner_uid,
                    'file_id': resource.meta['file_id'],
                }
                deduplication_id = '%s_%s_%s' % ('sync_public_fields_from_link_data', resource.owner_uid, resource.meta['file_id'])
                mpfs_queue.put(job_data, 'sync_public_fields_from_link_data', deduplication_id=deduplication_id, delay=65)
            if resource.is_folder:
                if resource.is_group_root or resource.is_shared_root:
                    job_data = {
                        'gid': resource.group.gid,
                        'actor': resource.uid,
                    }
                    mpfs_queue.put(job_data, 'make_shared_root_private')

                job_data = {
                    'uid': resource.address.uid,
                    'path': resource.address.id
                }
                mpfs_queue.put(
                    job_data,
                    'make_folder_content_private',
                    deduplication_id='make_folder_content_private_%s' % hashed(job_data),
                )
            FilesystemSetPrivateEvent(uid=uid, tgt_resource=resource,
                                      tgt_address=Address(rawaddress)).send_self_or_group(resource=resource)

        if return_info:
            return {'path': resource.address.path}
        else:
            return {}

    def print_to_listing_log(self, file):
        if self.request and isinstance(file, MPFSFile):
            file.print_to_listing_log()

    def get_fully_public_resource(self, public_addr, uid=None):
        """
        Получить ресурс по публичному хешу со всеми проверками публичности

        :param str uid: id пользователя, от имени которого делается запрос
        """
        if not isinstance(public_addr, PublicAddress):
            raise TypeError("`PublicAddress` required. Got: %s" % type(public_addr))

        symlink_raw_addr = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_raw_addr)
        check_yateam_subtree_access(uid, resource.path, resource.uid)
        user_exists_and_not_blocked(resource.uid)
        if resource.is_blocked():
            raise errors.ResourceBlocked()
        if not resource.is_fully_public():
            raise errors.ResourceNotFound()

        symlink_addr = SymlinkAddress(symlink_raw_addr)
        symlink = Symlink(symlink_addr)
        self.check_access(symlink, uid)

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)
            resource.set_request(self.request, force=True)

        return resource

    def public_info(self, hashed_address, uid=None, password_info=None):
        """
        Получение информации о публичном ресурсе

        :param str uid: id пользователя, от имени которого делается запрос
        :param dict password_info: Содержит в себе password или token от симлинка

        """
        public_address = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_address.hash)
        try:
            resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
            check_yateam_subtree_access(uid, resource.path, resource.uid)
            append_meta_to_office_files(resource,
                                        self.request)  # можем пропускать файлы вида <private_hash>:/<relative_file_path>, потому что этот ресурс - папка. Надо протестировать и поправить - перенести это в конец функции
        except errors.ResourceNotFound:
            from mpfs.core.support import comment

            comment_data = comment.find_by_hash(hashed_address)
            if comment_data:
                data = {
                    'view': comment_data.get('view', ''),
                    'link': comment_data.get('ext_link', ''),
                }
                raise errors.ResourceBlocked(data=data)
            else:
                raise

        user = User(resource.uid)

        # https://st.yandex-team.ru/CHEMODAN-36240
        # https://st.yandex-team.ru/CHEMODAN-42695
        if user.is_missing_or_blocked_in_passport():
            raise errors.ResourceNotFound()

        resource.assert_blocked()
        user.check_blocked()
        if not resource.is_fully_public():
            raise errors.ResourceNotFound()

        if (filter_uid_by_percentage(user.uid, FEATURE_TOGGLES_DISABLE_PUBLIC_LINKS_PERCENTAGE)
                and user.is_in_hard_overdraft()):
            raise errors.OverdraftUserPublicLinkDisabled()

        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)
        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)

        if public_address.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_address.path)
            resource = self.fs.resource(resource.uid, origin_address.id)
            resource.set_request(self.request, force=True)

        else:
            origin_address = Address.Make(resource.uid, resource.path)

        resource.assert_blocked()
        parents_resources = get_resources(resource.uid, origin_address.get_parents(),
                                          available_service_ids=('/disk',))
        for r in parents_resources:
            r.assert_blocked()

        user_info = user.public_info()

        self.print_to_listing_log(resource)

        resource.load_views_counter()
        resource.meta[SPEED_LIMITED] = int(quota.download_speed_limited(resource.address.uid))
        if isinstance(resource, MPFSFile):
            resource.setup_previews(PUBLIC_UID)

        resource_info = resource.info()
        if user.is_yateam():
            # Заблоченность по Анти ФО верстка определяет по наличию поля blockings в мете
            resource.meta.pop('blockings', None)
            resource_info['this']['meta'].pop('blockings', None)
        if symlink.get_read_only():
            resource.clean_urls()
        result = {RESOURCE: resource_info, USER: user_info}
        if symlink.have_password():
            token, _ = symlink.generate_password_token(symlink_hash, symlink.get_file_id())
            result[PUBLIC_PASSWORD_TOKEN] = token
        return result

    def public_fulltree(self, hashed_address, deep_level, uid=None, password_info=None):
        """Получить дерево публичной папки или ее подпапки

        :param str uid: id пользователя, от имени которого делается запрос
        :param dict password_info: Содержит в себе password или token от симлинка
        """
        public_addr = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)

        user = User(resource.uid)

        # https://st.yandex-team.ru/CHEMODAN-36240
        # https://st.yandex-team.ru/CHEMODAN-42695
        if user.is_missing_or_blocked_in_passport():
            raise errors.ResourceNotFound()

        check_yateam_subtree_access(uid, resource.path, resource.uid)
        user_exists_and_not_blocked(resource.uid)
        if resource.is_blocked():
            raise errors.ResourceBlocked
        if not resource.is_folder:
            raise errors.NotFolder()
        if not resource.is_fully_public():
            return

        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)
            resource.set_request(self.request, force=True)
            if not resource.is_folder:
                raise errors.NotFolder()

        if symlink.get_read_only():
            resource.clean_urls()

        return self.fs.fulltree(
            PUBLIC_UID,
            resource.address.id,
            deep_level=deep_level,
            relative=True,
            rfunc=PublicAddress.MakeRelative,
            clean=True,
            prefix=public_addr.hash,
            symlink_read_only=symlink.get_read_only(),
        )

    def public_content(self, hashed_address, uid=None, load_source_ids=False, password_info=None):
        """
        Получение листинга публичной папки

        :param str uid: id пользователя, от имени которого делается запрос
        :param dict password_info: Содержит в себе password или token от симлинка
        """
        public_address = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_address.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)

        user = User(resource.uid)

        # https://st.yandex-team.ru/CHEMODAN-36240
        # https://st.yandex-team.ru/CHEMODAN-42695
        if user.is_missing_or_blocked_in_passport():
            raise errors.ResourceNotFound()

        check_yateam_subtree_access(uid, resource.path, resource.uid)
        resource.assert_blocked()
        user.check_blocked()

        if not resource.is_fully_public():
            raise errors.ResourceNotFound()
        resource.load_views_counter()

        if (filter_uid_by_percentage(user.uid, FEATURE_TOGGLES_DISABLE_PUBLIC_LINKS_PERCENTAGE)
                and user.is_in_hard_overdraft()):
            raise errors.OverdraftUserPublicLinkDisabled()

        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)
        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)

        if public_address.has_relative():
            # проверяем не заблокированы ли какие либо папки-предки
            relative_address = Address.Make(resource.uid, resource.path + public_address.path)
            relative_parents_resources = get_resources(resource.uid, relative_address.get_parents(),
                                                       available_service_ids=('/disk',))

            for r in relative_parents_resources:
                r.assert_blocked()
            origin_resource = self.fs.resource(PUBLIC_UID, relative_address.id)
            origin_resource.assert_blocked()
        else:
            origin_resource = resource

        origin_resource.set_request(self.request, force=True)

        def set_relative_address(item, is_origin=False):
            root = public_address.path or '/'
            if is_origin:
                rel_address = PublicAddress.Make(public_address.hash, root)
            else:
                rel_address = PublicAddress.Make(public_address.hash, os.path.join(root, item.name))

            item.id = rel_address.id
            item.key = rel_address.id
            item.path = rel_address.path
        is_read_only_symlink = symlink.get_read_only()
        if is_read_only_symlink:
            origin_resource.clean_urls()
        origin_resource.list()
        folders = []
        files = []
        blocked_items_num = 0
        if hasattr(origin_resource, 'children_items'):
            folders = origin_resource.children_items['folders']
            files = origin_resource.children_items['files']

            cleaned_folders = []
            for item in folders:
                if not item.is_blocked():
                    set_relative_address(item)
                    cleaned_folders.append(item)
                else:
                    blocked_items_num += 1

            origin_resource.children_items['folders'] = cleaned_folders

            cleaned_files = []
            for item in files:
                if not item.is_blocked() and not item.is_infected():
                    set_relative_address(item)
                    cleaned_files.append(item)
                else:
                    blocked_items_num += 1

            origin_resource.children_items['files'] = cleaned_files

            for item in itertools.chain(cleaned_files, cleaned_folders):
                item.meta['page_blocked_items_num'] = blocked_items_num
                if is_read_only_symlink:
                    item.clean_urls()

        set_relative_address(origin_resource, is_origin=True)
        origin_resource.meta['page_blocked_items_num'] = blocked_items_num
        if load_source_ids:
            with change_experiment_context_with(uid=resource.uid):
                GlobalGalleryController.load_and_set_source_ids_for_resources([origin_resource] + folders + files)
        return map(
            lambda x: x.dict(),
            folders + files
        )

    def get_public_not_blocked_resource(self, hashed_address, uid=None, password_info=None):
        public_addr = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)

        check_yateam_subtree_access(uid, resource.path, resource.uid)

        user_exists_and_not_blocked(resource.uid)
        if resource.is_blocked():
            raise errors.ResourceBlocked()

        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)
            if resource.is_blocked():
                raise errors.ResourceBlocked()

        if symlink.get_read_only():
            resource.clean_urls()
        return resource

    def public_direct_url(self, hashed_address, modified, uid=None):
        """
        Формирование прямой ссылки на файл в мульке для Docviewer
        """
        resource = self.get_public_not_blocked_resource(hashed_address, uid=uid)
        return resource.get_direct_url(modified)

    def public_url(self, hashed_address, inline, uid=None, check_blockings=True, password_info=None):
        """
        Получение ссылки на публичный файл

        :param str uid: id пользователя, от имени которого делается запрос
        :param dict password_info: Содержит в себе password или token от симлинка
        """
        public_addr = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)
        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)
        if symlink.get_read_only():
            resource.clean_urls()

        owner = User(resource.uid)

        # https://st.yandex-team.ru/CHEMODAN-36240
        # https://st.yandex-team.ru/CHEMODAN-42695
        if owner.is_missing_or_blocked_in_passport():
            raise errors.ResourceNotFound()

        check_yateam_subtree_access(uid, resource.path, resource.uid)

        owner.check_blocked()

        resource.assert_blocked()

        if not resource.is_fully_public():
            raise errors.FileNotFound()

        if (filter_uid_by_percentage(owner.uid, FEATURE_TOGGLES_DISABLE_PUBLIC_LINKS_PERCENTAGE)
                and owner.is_in_hard_overdraft()):
            raise errors.OverdraftUserPublicLinkDisabled()

        user = None
        if uid:
            try:
                user = User(uid)
            except errors.StorageInitUser:
                pass

        if (check_blockings and hasattr(resource, 'get_blockings') and resource.get_blockings() and
                resource.uid not in resource.unlimited_users and (user is None or user.antifo_enabled) and
                not owner.is_yateam()):
            raise errors.FileNotFound()

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(resource.uid, origin_address.id)
            resource.assert_blocked()
        else:
            origin_address = Address.Make(resource.uid, resource.path)

        parents_resources = get_resources(resource.uid, origin_address.get_parents(),
                                          available_service_ids=('/disk',))
        for r in parents_resources:
            r.assert_blocked()
        result = resource.get_url_with_hash(
            override_uid=PUBLIC_UID,
            hash=hashed_address,
            inline=inline,
            owner_uid=resource.uid,
        )
        if symlink.get_read_only():
            result.pop('file', None)
            result.pop('folder', None)
            result['read_only'] = True
        return result

    def public_notification(self, uid, rawaddress, emails, message, locale=None):
        """
        Отправка email
        """
        template_params = {}
        resource = self.fs.resource(uid, rawaddress)
        if resource.is_fully_public():
            uinfo = passport.userinfo(uid)
            user_name = uinfo['public_name']
            template_params = {
                'username': user_name,
                'filename': resource.name,
                'fileurl': resource.get_public_url(),
                'message': message,
                'locale': locale,
            }
            for email in emails.split(','):
                mailer.send(email, 'publish', template_params, sender_name=user_name)

    def _grab(self, meth, uid, hashed_address, new_name, save_path=None, password_info=None):
        """
        Выполняет проверку предусловий и в случае успеха пердаёт управление непосредственно методу копирования `meth`.

        :param callable meth: метод копирования
        :param str uid: id пользователя, которому сохраняем ресурс
        :param str hashed_address: он же private_hash, он же public_key
        :param str new_name: какое имя будет у сохраненного файла
        :param str save_path: Путь до директории, куда сохранять файл. По умолчанию системная папка "Загрузки".
                              Публичные файлы яндексоидов из НДА папки сохраняются в НДА папку.
        :param dict password_info: Содержит в себе password или token от симлинка
        """
        public_addr = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)

        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)
        if symlink.get_read_only():
            raise errors.Forbidden()

        user = User(resource.uid)

        # https://st.yandex-team.ru/CHEMODAN-36240
        # https://st.yandex-team.ru/CHEMODAN-42695
        if user.is_missing_or_blocked_in_passport():
            raise errors.ResourceNotFound()

        check_yateam_subtree_access(uid, resource.path, resource.uid)
        if not resource.is_fully_public():
            return

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)

        if resource.is_infected():
            raise errors.CopyInfectedFileError()

        if is_yateam_subtree(resource.path) and user_has_nda_rights(resource.uid):
            if save_path is not None and not is_yateam_subtree(save_path):
                raise errors.BadArguments('cannot save nda file to non-nda folder')
            if save_path is None:
                save_path = YATEAM_DIR_PATH

        if save_path is None:
            locale = User(uid).get_supported_locale()
            incoming_address = self.fs.get_downloads_address(uid, locale)
            self.fs.mksysdir(uid, type='downloads', locale=locale)
        else:
            save_dir_address = Address.Make(uid, save_path)
            save_dir_folder = get_resource(uid, save_dir_address)
            if not isinstance(save_dir_folder, DiskFolder):
                raise errors.FolderNotFound()
            incoming_address = save_dir_folder.address

        self.fs.check_available_space(uid, incoming_address.id)
        tgt = incoming_address.get_child_resource(resource.name)
        if new_name:
            tgt = tgt.rename(new_name)
        tgt = self.fs.autosuffix_address(tgt)

        parent_folder = self.fs.resource(PUBLIC_UID, tgt.get_parent().id)
        parent_folder.check_rw()
        self.fs.check_lock(tgt.id)

        ret = meth(self.request, uid, resource.address, tgt)
        if ret:
            User(uid).set_state(key='first_file', value=1)
            resource.download_counter_inc()
        return ret

    def grab(self, uid, hashed_address, new_name, save_path=None, password_info=None):
        """
        Скопировать что угодно к другому пользователю
        """

        def grabber(request, uid, src_address, tgt_address):
            force = True

            with CatchHistoryLogging(catch_messages=True) as catcher:
                ret = self.fs.copy_resource(
                    uid, src_address.id, tgt_address.id, force=force, lock_source=False,
                    make_visible=True, force_djfs_albums_callback=True
                )
                logged_messages = catcher.get_messages(
                    pattern=SAVE_FILE_FROM_PUBLIC_LOG_MESSAGE_RE
                )

                lenta_block_id = None
                if logged_messages:
                    for msg in logged_messages:
                        try:
                            lenta_block_id = lenta_loader.save_file_from_public(msg)
                        except Exception:
                            error_log.info(traceback.format_exc())
                        break

                if request and lenta_block_id:
                    # пропихиваем в форматтер
                    request.form.meta[LENTA_BLOCK_ID] = lenta_block_id

            return ret

        return self._grab(grabber, uid, hashed_address, new_name, save_path=save_path, password_info=password_info)

    def async_grab(self, uid, hashed_address, new_name, save_path=None, password_info=None):
        """
        Асинхронно скопировать что угодно к другому пользователю
        """

        def grabber(request, uid, src_address, tgt_address):
            connection_id = request.connection_id
            subtype = '%s_%s' % (src_address.storage_name, tgt_address.storage_name)
            from mpfs.core.operations import manager

            src_resource = get_resource(uid, src_address)

            operation_data = dict(
                target=tgt_address.id,
                source=src_address.id,
                connection_id=connection_id,
                force=True,
                lock_source=False,
                make_visible=True,
                force_djfs_albums_callback=True,
            )

            lenta_block_id = None
            if src_resource.is_folder:
                # в случае папки мы заранее генерируем file_id и отправляем его Ленте
                try:
                    predefined_file_id = Resource.generate_file_id(uid, tgt_address.id)
                    operation_data['predefined_file_id'] = predefined_file_id
                    # идем в Ленту
                    raw_resource_id = ResourceId(uid=uid, file_id=predefined_file_id).serialize()
                    lenta_block_id = lenta_loader.start_saving_folder_from_public(uid, raw_resource_id, tgt_address.path)
                    if lenta_block_id:
                        operation_data[tags.LENTA_BLOCK_ID] = lenta_block_id
                except Exception:
                    error_log.info(traceback.format_exc())

            operation = manager.create_operation(uid, 'copy', subtype, odata=operation_data)
            result = {
                'oid': operation.id,
                'type': operation.type,
                tags.AT_VERSION: operation.at_version,
                'target_path': tgt_address.path
            }
            if lenta_block_id:
                result[LENTA_BLOCK_ID] = lenta_block_id
            return result

        return self._grab(grabber, uid, hashed_address, new_name, save_path=save_path, password_info=password_info)

    @staticmethod
    def parse_private_hash(private_hash):
        """Получить из публичного(приватного) хеша адреса: PublicAddress и SymlinkAddress

        Этого метода не должно быть, надо перенести эту логику в mpfs.core.address
        """
        public_addr = PublicAddress(private_hash)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        return public_addr, symlink_addr

    @classmethod
    def get_symlink(cls, hashed_address, allow_deleted=False):
        _, symlink_addr = cls.parse_private_hash(hashed_address)
        return Symlink(symlink_addr, allow_deleted=allow_deleted)

    def block(self, hashed_address):
        """Заблокировать ресурс по публичному хешу.

        Данный метод умеет работать как с хешами на корень публичной папки, так и с хешами на ресурс
        внутри публичной папки, то есть имеющими вид `hash:/path/to/relative/dir`.
        """
        public_address = PublicAddress(hashed_address)
        symlink = self.get_symlink(hashed_address)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink.address.id)
        uid = resource.uid

        raw_address = resource.address.id

        if resource.is_fully_public():
            # корневая папка действительно публичная
            if not public_address.has_relative():
                # блокируют корень публичной папки
                self.fs.remove_symlink(uid, raw_address)
            else:
                # передали публичную папку и путь относительно нее
                path = resource.path
                relative_path = public_address.path
                sep = os.sep
                if not path.endswith(sep):
                    path += sep
                if relative_path.startswith(sep):
                    relative_path = relative_path.lstrip(sep)
                relative_address = Address.Make(uid, os.path.join(path, relative_path))
                raw_address = relative_address.id
            resource = self.fs.resource(uid, raw_address)
            if resource.is_group_root or resource.is_shared_root:
                for root_folder in resource.group.get_all_root_folders(cached=False):
                    root_folder.make_blocked()
                    root_folder.make_private()
            else:
                resource.make_private()
                resource.make_blocked()

        return {
            'uid': uid,
            'rawaddress': raw_address,
            'user_ip': symlink.user_ip,
            'ptime': symlink.ctime,
            'stids': resource.all_storage_ids(),
            'hid': getattr(resource, 'hid', ''),
        }

    def unblock(self, uid, rawaddress):
        """
        Разблокировка ресурса
        """
        resource = self.fs.resource(uid, rawaddress)
        resource.make_unblocked()

    def notify_xiva(self, resource, action, old_version=None):
        if old_version is None:
            old_version = resource.prev_version
        rawaddress = resource.address
        xiva_data = {
            'xiva_data': [{
                'op': action,
                'key': rawaddress.path,
                'folder': rawaddress.parent_id,
                'resource_type': resource.type,
                'fid': resource.meta['file_id'],
            }, ],
            'uid': resource.uid,
            'old_version': old_version,
            'new_version': resource.version,
            'operation': 'action',
        }
        if self.request and hasattr(self.request, 'connection_id'):
            xiva_data['connection_id'] = self.request.connection_id
        if resource.is_shared_internal or resource.is_group_internal:
            xiva_data['committer'] = resource.uid
            xiva_data['gid'] = resource.group.gid
            xiva_data['action_name'] = 'diff_group'
        else:
            xiva_data['action_name'] = 'diff'
        push_queue.put(xiva_data)

    def download_counter_inc(self, hashed_address, bytes_downloaded, count):
        """
        Инкрементация счётчика скачиваний
        """
        public_addr = PublicAddress(hashed_address)
        symlink_addr = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_addr)

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)

        if resource.is_group_internal or resource.is_shared_internal:
            uids = resource.group.all_uids()
            old_version = resource._service.get_version(uids)
        else:
            old_version = resource.version

        quota.update_traffic(resource.owner_uid, bytes_downloaded)
        resource.download_counter_inc(count)
        if not settings.push['disable_for']['published_download']:
            self.notify_xiva(resource, 'published_download', old_version=old_version)

    def get_address(self, hashed_address, password_info=None):
        """
        Принимает публичный хэш
        Возвращает uid и address
        """
        public_addr = PublicAddress(hashed_address)
        symlink_hash = crypt_agent.decrypt(public_addr.hash)
        resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
        symlink_addr = SymlinkAddress(symlink_hash)
        symlink = Symlink(symlink_addr)

        if public_addr.has_relative():
            origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
            resource = self.fs.resource(PUBLIC_UID, origin_address.id)

        if symlink.public_uid:
            uid = symlink.public_uid
        else:
            uid = resource.uid

        self.check_access(symlink, uid)
        self.check_passworded_symlink(symlink, symlink_hash, password_info)

        return {
            'uid': uid,
            'rawaddress': resource.address.id,
            'user_ip': symlink.user_ip,
            'ptime': symlink.ctime,
        }

    def public_video_url(self, uid, hashed_address, yandexuid):
        public_addr = PublicAddress(hashed_address)

        album = get_album_by_public_key_unchecked(public_addr.hash)
        if album:
            if not public_addr.has_relative():
                raise AlbumItemMissed()
            resource = get_resource_from_album_item(album, public_addr.path.strip('/'), uid)
        else:
            symlink_addr = crypt_agent.decrypt(public_addr.hash)
            resource = self.fs.get_symlink_resource(PUBLIC_UID, symlink_addr)
            symlink_address = SymlinkAddress(symlink_addr)
            symlink = Symlink(symlink_address)
            self.check_access(symlink, uid)

            user = User(resource.uid)

            # https://st.yandex-team.ru/CHEMODAN-36240
            # https://st.yandex-team.ru/CHEMODAN-42695
            if user.is_missing_or_blocked_in_passport():
                raise errors.ResourceNotFound()

            check_yateam_subtree_access(uid, resource.path, resource.uid)
            user_exists_and_not_blocked(resource.uid)
            if resource.is_blocked():
                raise errors.ResourceBlocked
            user.check_blocked()

            if public_addr.has_relative():
                origin_address = Address.Make(resource.uid, resource.path + public_addr.path)
                resource = self.fs.resource(PUBLIC_UID, origin_address.id)

        consumer = uid if uid is not None else PUBLIC_UID
        return video.generate_url(
            consumer, resource.uid, resource.file_mid(), resource.hid,
            public=True, video_info=resource.get_video_info(), consumer_yandexuid=yandexuid
        )

    @staticmethod
    def check_access(symlink, uid):
        symlink_org_ids = symlink.get_external_organization_ids()
        if len(symlink_org_ids) == 0:
            return

        if uid is None or uid == PUBLIC_UID:
            raise errors.ResourceNotFound()

        user_info = User(uid).get_user_info()
        user_org_ids = user_info.get('external_organization_ids', [])

        if len(set(symlink_org_ids) & set(user_org_ids)) == 0:
            raise errors.ResourceNotFound()

    @staticmethod
    def check_passworded_symlink(symlink, public_hash, password_info=None):
        if symlink.have_password():
            if password_info is None:
                raise errors.Forbidden()
            password = password_info.get('password')
            password_token = password_info.get('token')
            if not password and not password_token:
                raise errors.Forbidden()
            if password:
                symlink.password_check(password)
            if password_token:
                symlink.password_token_check(password_token, public_hash, symlink.get_file_id())

