# -*- coding: utf-8 -*-
from os.path import sep
import time

import mpfs.engine.process
import mpfs.common.errors as common_errors
import mpfs.common.errors.share as errors

from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.social.share.entity import SharedEntity
from mpfs.core.metastorage.control import groups, group_links, group_invites
from mpfs.core.address import Address
from mpfs.common.util import hashed
from mpfs.core.factory import get_resource
from mpfs.core.filesystem.quota import Quota
from mpfs.core.social.share.link import LinkToGroup
from mpfs.core.social.share.invite import GroupInvites
from mpfs.core.services.disk_service import Disk
from mpfs.core.queue import mpfs_queue
from mpfs.core.social.share.utils import safely_check_has_shared_folders
from mpfs.core.user.base import User
from mpfs.core.bus import Bus

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class Group(SharedEntity):

    __all__ = ('_id', 'path', 'owner', 'size')
    controller = groups
    _loaded = {}
    _instances = {'id': {}, 'uid_path': {}}
    # не очищайте бездумно этот сторадж, тк к нему может идти обращение в момент, когда
    # в базе данные уже удалены (используется по сути как временное хранилище)
    _storage = {}

    def __init__(self, *args, **kwargs):
        super(Group, self).__init__(*args, **kwargs)
        self.gid = self._id
        self.links = []
        self.invites = []

    @classmethod
    def Create(cls, uid, addr, gid=None):
        if groups.get_count(owner=uid) >= settings.social['max_groups_number']:
            e = errors.ShareGroupsLimitReached('User "%s" reached group number limit.' % uid)
            # цикл импорт
            from mpfs.core.social.share.processor import b2b_group_limits_reached_alarm
            b2b_group_limits_reached_alarm(uid, e)
            raise e
        if not isinstance(addr, Address):
            addr = Address(addr)
        if Quota().free(uid=addr.uid) <= 0:
            raise errors.GroupNoFreeSpace('Tried to create a shared folder but user exceeded its limits')

        if not gid:
            cur_time = int(time.time())
            rounded_cur_time = cur_time - cur_time % 60  # round up to minutes
            gid = hashed('%s:%s' % (addr.id, rounded_cur_time))
        folder = get_resource(uid, addr)
        from mpfs.core.filesystem.resources.disk import DiskFolder
        if folder.address.is_storage or not isinstance(folder, DiskFolder):
            raise common_errors.AddressError()
        #=======================================================================
        # Check existing groups
        #TODO: проверка выглядит некрасиво, но так немного понятнее
        parents = set()
        id_split = filter(None, addr.path.split('/'))[:-1]
        for i in xrange(0, len(id_split)):
            parents.add('/' + '/'.join(id_split[:i+1]))
        for group in groups.get_all(owner=uid):
            if group['path'] == addr.path:
                return cls(**group)
            elif group['path'].startswith(addr.path + '/'):
                raise errors.GroupConflict(group['path'])
            elif group['path'] in parents:
                raise errors.GroupConflict(group['path'])
        for link in group_links.get_all(uid=uid):
            if link['path'] == addr.path:
                raise errors.GroupConflict(addr.path)
            elif link['path'].startswith(addr.path + '/'):
                raise errors.GroupConflict(addr.path)
            elif addr.path.startswith(link['path'] + '/'):
                raise errors.GroupConflict(addr.path)
        #=======================================================================
        fs = Bus()
        fs.check_lock(addr.id)

        data = {
                '_id' : gid,
                'path' : addr.path,
                'owner' : uid,
                'size' : 0,
                }
        group = cls(**data)
        User(uid).set_has_shared_folders()
        group.save()
        folder = group.get_folder()
        folder.set_shared()
        cls._instances['uid_path'][(group.owner, group.path)] = group
        cls._instances['id'][group.gid] = group
        mpfs_queue.put({'gid' : gid}, 'group_setsize')
        return group

    def is_full(self):
        return self.user_count() - 1 >= settings.social['max_users_number']

    def check_is_full(self):
        if self.is_full():
            e = errors.ShareUserLimitReached('Group "%s" is full. Owner uid: "%s"' % (self.gid, self.owner))
            # цикл импорт
            from mpfs.core.social.share.processor import b2b_group_limits_reached_alarm
            b2b_group_limits_reached_alarm(self.owner, e)
            raise e

    @classmethod
    def load(cls, gid=None, owner=None, path=None):

        group_data = None
        if owner and path:
            if (owner, path) in cls._instances['uid_path']:
                group = cls._instances['uid_path'][(owner, path)]
                group.links = []
                return group
            else:
                group_data = groups.get_one(owner=owner, path=path)
        elif gid:
            if gid in cls._instances['id']:
                group = cls._instances['id'][gid]
                group.links = []
                return group
            else:
                group_data = groups.get_one(_id=gid)

        if not group_data:
            raise errors.GroupNotFound(gid)

        group = cls(**group_data)
        cls._instances['uid_path'][(group.owner, group.path)] = group
        cls._instances['id'][group.gid] = group
        return group

    @classmethod
    def load_all(cls, uid):
        """
        Load to cache and return all groups by owner UID.
        :param uid: Owner UID.
        :return: {'uid_path': {(group.owner, group.path): group, ...}, 'id': {group.gid: group, ...}}
        """
        if safely_check_has_shared_folders(uid) and not cls._loaded.get(uid):
            for group_data in groups.get_all(owner=uid):
                group = cls(**group_data)
                cls._instances['uid_path'][(group.owner, group.path)] = group
                cls._instances['id'][group.gid] = group
        cls._loaded[uid] = True
        return cls._instances

    @classmethod
    def has_data_in_cache_for(cls, uid):
        if not cls._loaded.get(uid):
            return None

        for loaded_group in cls._instances['id'].values():
            try:
                if loaded_group.owner == uid:
                    return True
            except Exception:
                pass

        return False

    @classmethod
    def iter_all(cls, uid):
        for group in cls.load_all(uid)['id'].itervalues():
            if group.owner == uid:
                yield group

    @classmethod
    def find(cls, **kw):
        find_group = kw.get('group')
        find_link = kw.get('link')
        try:
            gid = kw['gid']
        except KeyError:
            try:
                uid = kw['uid']
                path = kw['path']
            except KeyError:
                raise errors.GroupNotFound()
            else:
                path_split = path.split('/')
                if not find_link or find_group:
                    cls.load_all(uid)
                    for i in xrange(len(path_split), 2, -1):
                        path = '/'.join(path_split[:i])
                        try:
                            group = cls._instances['uid_path'][(uid, path)]
                        except KeyError:
                            pass
                        else:
                            return group
                if not find_group or find_link:
                    LinkToGroup.load_all(uid)
                    for i in xrange(len(path_split), 2, -1):
                        if kw.get('find_links_correctly'):
                            # я не дебил, этот флажок я тут сделал, потому что мне нужен правильный код (а именно поиск
                            # группы по груп линку у участника, если путь ОП у него не совпадает с владельцем), а как
                            # оно проработало 7 лет и работает сейчас в других местах - я не знаю
                            path = '/'.join(path_split[:i])
                        try:
                            link = LinkToGroup._instances['uid_path'][(uid, path)]
                        except KeyError:
                            link = None
                        else:
                            if find_link and not find_group:
                                return link
                            else:
                                return link.group
                raise errors.GroupNotFound()
        else:
            try:
                group = cls._instances['id'][gid]
            except KeyError:
                return Group.load(gid)
            else:
                return group

    def update_disk_size(self, val):
        Disk().update_counter(self.owner, val)
        self.update_group_size(val)

    def update_group_size(self, val):
        self.size += val
        # Сейчас честный шаринг и размер группы хранится только для статистики по историческим причинам
        try:
            groups.increment(self.owner, self.gid, val, 'size')
        except common_errors.UserIsReadOnly:
            # https://st.yandex-team.ru/CHEMODAN-75397 Не ломать операции перемещения, если группы в ридонли
            error_log.error("groups.size increment failed (owner=%s, gid=%s). groups is in readonly mode" % (self.owner, self.gid))

    def iteruids(self):
        """Iterate through UIDs having access to group."""
        for each in group_links.get_all(gid=self.gid):
            yield each['uid']

    def get_link_by_uid(self, uid):
        """Получение LinkToGroup по uid

        Если такого линка нет, то бросается исключение ShareNotFound
        """
        return LinkToGroup.load(gid=self.gid, uid=uid)

    def iterlinks(self):
        """Получить итератор по всем линкам

        Использует кеширование
        """
        if not self.links:
            self.links = list(self.load_links())
        for link in self.links:
            yield link

    def _get_excluded_b2b_key(self):
        user = self.owner_user
        if user.is_b2b():
            return user.b2b_key

    @property
    def owner_user(self):
        return User(self.owner)

    def load_links(self, exclude_b2b=False, offset=0, limit=None):
        """Загружает линки из базы и возвращает итератор.

        Отличается от метода `iterlinks` отсутствием кэширования
        """
        if limit is not None and limit <= 0:
            raise StopIteration()

        excluded_b2b_key = None if not exclude_b2b else self._get_excluded_b2b_key()
        raw_links = group_links.iter_by_gid(self.gid, excluded_b2b_key=excluded_b2b_key, offset=offset, limit=limit)

        for raw_link in raw_links:
            raw_link['group'] = self
            link = LinkToGroup(**raw_link)
            link.update_instances()
            yield link

    def count_links(self, exclude_b2b=False):
        excluded_b2b_key = None if not exclude_b2b else self._get_excluded_b2b_key()
        return group_links.count_by_gid(self.gid, excluded_b2b_key=excluded_b2b_key)

    def get_all_root_folders(self, cached=False):
        """Загружает все рутовые папки группы(включая папку владельца)

        Использует методы, которые кешируют результаты

        :param cached: Принудительно загрузить данные из кэша и не ходить в базу за ними.
        """
        rf_key = 'root_folders'
        if cached:
            if rf_key in self._storage and self.gid in self._storage[rf_key]:
                return self._storage[rf_key][self.gid]

        result = [self.get_folder()]
        for link in self.iterlinks():
            result.append(link.get_folder())

        self._storage.setdefault(rf_key, {})[self.gid] = result
        return result

    def iterinvites(self):
        """Получить итератор по всем инвайтам

        Использует кеширование
        """
        if not self.invites:
            self.invites = list(self.load_invites())
        for invite in self.invites:
            yield invite

    def load_invites(self, statuses=None, offset=0, limit=None):
        """Загружает инвайты из базы и возвращает итератор.

        Отличается от метода `iterinvites` отсутствием кэширования
        """
        if limit is not None and limit <= 0:
            raise StopIteration()
        raw_invites = group_invites.iter_by_gid(self.gid, statuses=statuses, offset=offset, limit=limit)
        for raw_invite in raw_invites:
            raw_invite['group'] = self
            invite = GroupInvites(**raw_invite)
            yield invite

    def count_invites(self, statuses=None):
        return group_invites.count_by_gid(self.gid, statuses=statuses)

    def first_invited(self, universe_login, universe_service):
        if not len(list(self.iterlinks())):
            group_invites = list(self.iterinvites())
            invites_count = len(group_invites)
            if invites_count == 0:
                return True
            elif invites_count == 1:
                invite_hash = self.generate_invite_id(universe_login, universe_service)
                if group_invites[0].hash == invite_hash:
                    return True
        return False

    def all_uids(self):
        """Return UIDs of owner and users having access to group."""
        result = [self.owner]
        result.extend(self.iteruids())
        return result

    def get_folder(self):
        from mpfs.core.filesystem.resources.group import GroupRootFolder
        address = Address('%s:%s' % (self.owner, self.path))
        return GroupRootFolder(self.owner, address, group=self)

    def get_group_path(self, link_path, uid):
        link = LinkToGroup.load(uid=uid, gid=self.gid)
        if link_path.startswith(link.path):
            return self.path + self._get_relative_path(link.path, link_path)
        else:
            return link_path

    def get_group_link_base_version(self, uid):
        link = LinkToGroup.load(uid=uid, gid=self.gid)
        return link.get_search_indexer_base_version()

    def get_relative_path(self, address):
        if address.uid == self.owner:
            base_path = self.path
        else:
            link = LinkToGroup.load(uid=address.uid, gid=self.gid)
            base_path = link.path
        return self._get_relative_path(base_path, address.path)

    def get_group_address(self, address):
        path = self.get_group_path(address.path, address.uid)
        return Address.Make(self.owner, path)

    @staticmethod
    def _get_relative_path(base_path, path):
        path_split = filter(None, path.split('/'))
        base_len = len(filter(None, base_path.split('/')))
        relative_path_split = path_split[base_len:]
        if relative_path_split:
            return '/' + '/'.join(relative_path_split)
        else:
            return ''

    @staticmethod
    def get_link_path(invited_uid, owner_uid, owner_path):
        """Получить путь ресурса у Приглашенного по пути Владельца"""
        resource_address = Address.Make(owner_uid, owner_path)
        resource = factory.get_resource(owner_uid, resource_address)

        if not resource.is_group:
            raise errors.GroupNotFound()

        relative_path = resource.group.get_relative_path(resource_address)
        link = LinkToGroup.load(uid=invited_uid, gid=resource.group.gid)
        shared_root_path = link.path
        # Чистим путь от лишних разделителей /
        result = sep + sep.join(filter(None, sep.join([shared_root_path, relative_path]).split(sep)))
        return result

    def get_address(self):
        return Address('%s:%s' % (self.owner, self.path))

    def remove_links(self):
        for link in self.iterlinks():
            link.remove()
            try:
                link.remove_folder()
            except errors.ResourceNotFound:
                log.error('Remove group_link, associated folder not found')

    def remove_invites(self):
        for invite in self.iterinvites():
            invite.remove()

    def user_count(self):
        '''Return number of users in this group + owner'''
        return group_links.get_count(gid=self.gid) + 1

    def invites_count(self):
        return group_invites.get_count(gid=self.gid, status='new')

    def generate_invite_id(self, universe_login, universe_service):
        return hashed('%s:%s:%s' % (self.gid, universe_login, universe_service))

    def remove(self):
        self._instances['id'].pop(self.gid, None)
        self._instances['uid_path'].pop((self.owner, self.path), None)
        SharedEntity.remove(self)

    def rename(self, new_path):
        old_path = self.path
        super(Group, self).rename(new_path)
        self._instances['uid_path'].pop((self.owner, old_path), None)
        self._instances['uid_path'][(self.owner, self.path)] = self

    def set_size(self):
        folder = self.get_folder()
        self.size = folder.get_size()
        self.save()

    @classmethod
    def reset(cls):
        super(Group, cls).reset()

    @classmethod
    def reset_storage(cls):
        cls._storage = {}

    def is_member(self, uid):
        try:
            LinkToGroup.load(gid=self.gid, uid=uid)
        except errors.ShareNotFound:
            return False
        else:
            return True


class Groups(object):
    def preload(self, gids):
        for group_data in groups.get_in(field='_id', array=gids):
            group = Group(**group_data)
            Group._instances['uid_path'][(group.owner, group.path)] = group
            Group._instances['id'][group.gid] = group
