# -*- coding: utf-8 -*-


import time


import mpfs.common.errors.share as errors

from mpfs.core.social.share.entity import SharedEntity
from mpfs.core.metastorage.control import group_invites
from mpfs.core.social.share.link import LinkToGroup
from mpfs.core.services.passport_service import Passport
from mpfs.common.util import hashed


passport_entity = Passport()


def get_invite_hash(gid, userid, provider):
    if provider == 'ya_directory':
        provider = 'email'
        # используем 'email', как и для приглашения через email, чтобы нельзя было два раза пригласить пользователя

    return hashed('%s:%s:%s' % (gid, userid, provider))


class GroupInvites(SharedEntity):

    class Status(object):
        ACTIVATED = 'activated'
        NEW = 'new'
        REJECTED = 'rejected'

    __all__ = (
        '_id',
        'gid',
        'rights',
        'ctime',
        'universe_login', 
        'universe_service',
        'uid',
        'avatar',
        'name',
        'status'
    )
    
    controller = group_invites

    def __init__(self, *args, **kwargs):
        super(GroupInvites, self).__init__(*args, **kwargs)
        if not hasattr(self, 'group'):
            from mpfs.core.social.share.group import Group
            self.group = Group.load(self.gid)
        for k in ('uid', 'avatar', 'name'):
            if not hasattr(self, k):
                setattr(self, k, None)
        if not hasattr(self, 'status'):
            self.status = self.Status.NEW
        self.hash = self._id

    @classmethod
    def load(cls, hsh):
        invite_data = group_invites.get_one(_id=hsh)
        if invite_data:
            return cls(**invite_data)
        else:
            raise errors.GroupInviteNotFound(hsh)

    def is_new(self):
        return self.status == self.Status.NEW

    def is_activated(self):
        return self.status == self.Status.ACTIVATED

    def is_rejected(self):
        return self.status == self.Status.REJECTED

    def renew(self, **kw):
        for k,v in kw.iteritems():
            setattr(self, k, v)
        self.ctime = int(time.time())

    def set_activated(self, uid):
        if self.uid:
            for invite_data in group_invites.get_all(uid=self.uid, status='new', gid=self.gid):
                if invite_data['_id'] != self._id:
                    invite_data['group'] = self.group
                    invite = GroupInvites(**invite_data)
                    invite.renew(status=self.Status.ACTIVATED)
                    invite.uid = uid
                    invite.save()
        self.renew(status=self.Status.ACTIVATED)
        self.uid = uid
        self.save()

    def set_rejected(self):
        self.renew(status=self.Status.REJECTED)
        self.save()

    def info(self, uid):
        if self.status == self.Status.ACTIVATED and uid:
            if self.uid == uid:
                link = LinkToGroup.load(uid=uid, gid=self.group.gid, group=self.group)
                data = {
                    'id': link.path,
                    'gid': self.group.gid
                }
            else:
                raise errors.ShareError()
        else:
            passport_data = passport_entity.userinfo_summary(self.group.owner)
            
            data = {
                'status': self.status,
                'owner_uid': self.group.owner,
                'owner_login': passport_data['login'],
                'owner_name': passport_data['public_name'],
                'folder_name': self.group.folder_name,
                'rights': self.rights,
                'size': self.group.size,
                'gid': self.group.gid
            }

        return data
