# -*- coding: utf-8 -*-

import time

import mpfs.engine.process
import mpfs.common.errors.share as errors

from mpfs.core.metastorage.control import group_links
from mpfs.core.social.share.entity import SharedEntity
from mpfs.common.util import hashed
from mpfs.core.address import Address
from mpfs.core.social.share.utils import safely_check_has_shared_folders
from mpfs.core.user.base import User
from mpfs.metastorage.mongo.util import generate_version_number

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class LinkToGroup(SharedEntity):

    __all__ = ('_id', 'gid', 'uid', 'rights', 'path', 'universe_login', 'universe_service', 'ctime', 'b2b_key')
    controller = group_links
    _loaded = {}
    _instances = {'id': {}, 'uid_path': {}}

    def __init__(self, *args, **kwargs):
        super(LinkToGroup, self).__init__(*args, **kwargs)
        group = kwargs.get('group')
        self.lid = self._id
        if group:
            self.group = group
        else:
            from mpfs.core.social.share.group import Group
            self.group = Group.load(self.gid)

    @staticmethod
    def build_id(uid, gid):
        return hashed('%s:%s' % (uid, gid))

    @classmethod
    def load_for_user(cls, user, group):
        """Загрузить group_link

        :param User user: пользователь, для которого загрузить link
        :param Group group: группа
        """
        lid = cls.build_id(user.uid, group.gid)
        return cls.load(lid)

    @classmethod
    def load(cls, *a, **kw):
        link = None
        spec = {}
        if not a:
            try:
                spec['_id'] = cls.build_id(kw['uid'], kw['gid'])
            except KeyError:
                spec = kw
                try:
                    spec['_id'] = spec.pop('lid')
                except KeyError:
                    pass
        else:
            spec['_id'] = a[0]

        try:
            lid = spec['_id']
        except Exception:
            try:
                uid = spec['uid']
                path = spec['path']
            except KeyError:
                pass
            else:
                link = cls._instances['uid_path'].get((uid, path))
        else:
            link = cls._instances['id'].get(lid)

        # https://jira.yandex-team.ru/browse/CHEMODAN-7488
        uid = kw.get('uid')
        if not link and uid and cls._loaded.get(uid):
            raise errors.ShareNotFound(spec)

        if not link:
            link_data = group_links.get_one(**spec)
            if link_data:
                link_data['group'] = kw.get('group')
                link = cls(**link_data)
                cls._instances['id'][link.lid] = link
                cls._instances['uid_path'][(link.uid, link.path)] = link
            else:
                raise errors.ShareNotFound(spec)
        else:
            link._folder = None
        return link

    @classmethod
    def load_all(cls, uid):
        """
        Load to cache and return all group links by owner UID.
        :param uid: Link owner UID.
        :return: {'uid_path': {(link.uid, link.path): link, ...}, 'id': {link.lid: link, ...}}
        """
        if safely_check_has_shared_folders(uid) and not cls._loaded.get(uid):
            all_links = group_links.get_all(uid=uid)
            all_link_gids = map(lambda x: x.get('gid'), all_links)
            from mpfs.core.social.share.group import Groups
            Groups().preload(all_link_gids)
            for link_data in all_links:
                try:
                    link = cls(**link_data)
                except errors.GroupNotFound:
                    error_log.error('Group for link not found, uid: %s gid: %s' % (link_data['uid'], link_data['gid']))
                else:
                    cls._instances['id'][link.lid] = link
                    cls._instances['uid_path'][(link.uid, link.path)] = link
        cls._loaded[uid] = True
        return cls._instances

    @classmethod
    def has_data_in_cache_for(cls, uid):
        if not cls._loaded.get(uid):
            return None

        for item in cls._instances['id'].values():
            try:
                if item.uid == uid:
                    return True
            except Exception:
                pass

        return False

    @classmethod
    def iter_all(cls, uid):
        for each in cls.load_all(uid)['id'].itervalues():
            if each.uid == uid:
                yield each

    @classmethod
    def Create(cls, uid, gid, path, rights):
        lid = cls.build_id(uid, gid)
        data = {
            '_id': lid,
            'gid': gid,
            'rights': rights,
            'uid': uid,
            'path': path,
            'ctime': int(time.time()),
        }
        user = User(uid)
        if user.is_b2b():
            data['b2b_key'] = user.b2b_key
        link = cls(**data)
        user.set_has_shared_folders()
        link.save()
        cls._instances['id'][link.lid] = link
        cls._instances['uid_path'][(link.uid, link.path)] = link
        from mpfs.core.last_files.logic import SharedLastFilesProcessor
        SharedLastFilesProcessor().update_for_group_async(gid)
        return link

    def remove_folder(self):
        return self.get_folder().rm(keep_group=True)

    def remove(self):
        self._instances['id'].pop(self.lid, None)
        self._instances['uid_path'].pop((self.uid, self.path), None)
        super(LinkToGroup, self).remove()
        from mpfs.core.last_files.logic import SharedLastFilesProcessor
        SharedLastFilesProcessor().update_for_group_async(self.gid)

    def is_rw(self):
        return self.rights == 660

    def get_folder(self):
        try:
            folder = self._folder
        except AttributeError:
            folder = None
        if not folder:
            from mpfs.core.filesystem.resources.share import SharedRootFolder
            address = Address('%s:%s' % (self.uid, self.path))
            folder = SharedRootFolder(self.uid, address, link=self)
            self._folder = folder
        return folder

    def get_link_path(self, group_path):
        group_path_split = filter(None, group_path.split('/'))
        path_tail_split = group_path_split[self.group.path.count('/'):]
        if path_tail_split:
            return self.path + '/' + '/'.join(path_tail_split)
        else:
            return self.path

    def rename(self, new_path):
        old_path = self.path
        super(LinkToGroup, self).rename(new_path)
        self._instances['uid_path'].pop((self.uid, old_path), None)
        self._instances['uid_path'][(self.uid, self.path)] = self

    def update_instances(self):
        self._instances['id'][self.lid] = self
        self._instances['uid_path'][(self.uid, self.path)] = self

    def get_b2b_key(self):
        try:
            return self.b2b_key
        except AttributeError:
            return None

    def get_search_indexer_base_version(self):
        # search_indexer_base_version - монотонно возрастающая цифра, равная времени создания груп линка
        # нужна для того, чтобы при присоединении (и последующем отсоединении) к одной и той же ОП одним и тем же
        # участником в поиск улетали не версии ресурсов владельца (всегда одни и те же), а версии ресурсов, которые
        # старше, чем улетали в прошлый раз. Сделано через суммирование search_indexer_base_version и версии
        # ресурса у владельца. Работает только для отправки в индексатор, для всего остального отдается версия
        # владельца.
        ctime = getattr(self, 'ctime', 0)
        if ctime:
            return ctime * 1000000
        return 0
