# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.static.tags.push import *
from mpfs.core.queue import mpfs_queue
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.common.util import name_from_path
from mpfs.core.services.passport_service import Passport
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.core.filesystem.quota import Quota

log = mpfs.engine.process.get_default_log()
passport_entity = Passport()
push_queue = PushQueue()


class MailNotifier(object):

    def user_left_folder(self, user_name, user_email, link):
        user_info = passport_entity.userinfo(link.group.owner)
        to = user_info['email']
        template = 'sharedFolder/left'

        args = {
            'folderName': link.group.folder_name,
            'username': user_name,
            'userEmail': user_email,
            'locale': user_info['language'],
            'gid': link.gid,
        }

        data = {
            'email_to': to,
            'template_name': template,
            'template_args': args,
            'sender_name': user_name,
            'sender_email': user_email,
        }

        mpfs_queue.put(data, 'send_email')

    def user_activated_invite(self, user_name, user_email, link):
        user_info = passport_entity.userinfo(link.group.owner)
        to = user_info['email']
        template = 'sharedFolder/accepted'

        args = {
            'folderName': link.group.folder_name,
            'username': user_name,
            'rights': str(link.rights),
            'userEmail': user_email,
            'locale': user_info['language'],
            'gid': link.gid,
        }

        data = {
            'email_to': to,
            'template_name': template,
            'template_args': args,
            'sender_name': user_name,
            'sender_email': user_email,
        }

        mpfs_queue.put(data, 'send_email')

    def rights_changed(self, link):
        user_info = passport_entity.userinfo(link.uid)
        owner_info = passport_entity.userinfo(link.group.owner)
        template = 'sharedFolder/rights'

        args = {
            'friendName': owner_info['public_name'],
            'friendEmail': owner_info['email'],
            'folderName': link.folder_name,
            'rights': str(link.rights),
            'locale': user_info['language'],
            'gid': link.gid,
        }

        data = {
            'email_to': user_info['email'],
            'template_name': template,
            'template_args': args,
            'sender_name': owner_info['public_name'],
            'sender_email': owner_info['email'],
        }

        mpfs_queue.put(data, 'send_email')

    def user_kicked(self, link):
        user_info = passport_entity.userinfo(link.uid)
        owner_info = passport_entity.userinfo(link.group.owner)
        template = 'sharedFolder/kicked'

        args = {
            'friendName': owner_info['public_name'],
            'friendEmail': owner_info['email'],
            'folderName': link.folder_name,
            'locale': user_info['language'],
            'gid': link.gid,
        }

        data = {
            'email_to': user_info['email'],
            'template_name': template,
            'template_args': args,
            'sender_name': owner_info['public_name'],
            'sender_email': owner_info['email'],
        }

        mpfs_queue.put(data, 'send_email')

    def group_removed(self, group):
        owner_info = passport_entity.userinfo(group.owner)
        for link in group.iterlinks():
            user_info = passport_entity.userinfo(link.uid)
            template = 'sharedFolder/deleted'

            args = {
                'friendName': owner_info['public_name'],
                'friendEmail': owner_info['email'],
                'folderName': link.folder_name,
                'locale': user_info['language'],
                'gid': link.gid,
            }

            data = {
                'email_to': user_info['email'],
                'template_name': template,
                'template_args': args,
                'sender_name': owner_info['public_name'],
                'sender_email': owner_info['email'],
            }

            mpfs_queue.put(data, 'send_email')


class XivaNotifier(object):

    def push_actor_invite_created(self, user_uid, owner_uid, group, group_rights, invite_hash):
        owner_name = passport_entity.userinfo_summary(owner_uid)['public_name']
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_new', 'for': 'actor'}},
            'values': [
                {
                    'tag': 'owner',
                    'value': '',
                    'parameters': {
                        'uid': owner_uid,
                        'name': owner_name
                    }
                },
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'name': name_from_path(group.path),
                        'rights': group_rights,
                        'hash': invite_hash
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_NEW,
            'uid': user_uid,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        push_queue.put(data)

    def push_owner_invite_created(self, invite, group):
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_new', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'universe_login': invite.universe_login,
                        'universe_service': invite.universe_service,
                        'first_invitation': int(
                            group.first_invited(invite.universe_login, invite.universe_service)
                        )
                    }
                },
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'name': group.folder_name,
                        'path': group.path + '/',
                        'gid': group.gid,
                        'rights': invite.rights
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_NEW,
            'uid': group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)

    def push_invite_approved(self, uid, user_path, group, user_name, old_version, new_version):
        xiva_data = {
            'root': {
                'tag': 'share', 'parameters': {
                    'type': 'invite_approved',
                    'for': 'actor',
                    'new': new_version,
                    'old': old_version
                }
            },
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': user_path + '/',
                        'gid': group.gid
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_APPROVED,
            'uid': uid,
            'new_version': new_version,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_approved', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': group.path + '/',
                        'gid': group.gid,
                        'name': name_from_path(group.path)
                    }
                },
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'uid': uid,
                        'name': user_name
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_APPROVED,
            'uid': group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        push_queue.put(data)

        # отправка остальным пользователям каталога
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_approved', 'for': 'user'}},
            'values': [],
        }
        xiva_value_folder = {
            'tag': 'folder',
            'value': '',
            'parameters': {
                'path': group.path + '/',
                'gid': group.gid,
                'name': name_from_path(group.path)
            }
        }
        xiva_value_user = {
            'tag': 'user',
            'value': '',
            'parameters': {
                'uid': uid,
                'name': user_name
            }
        }
        data = {
            'class': SHARE_INVITE_APPROVED,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        for link in group.iterlinks():
            if link.uid == uid:
                continue
            else:
                xiva_value_folder['parameters']['path'] = link.path + '/'
                xiva_value_folder['parameters']['name'] = name_from_path(link.path)
                data['xiva_data']['values'] = [xiva_value_folder, xiva_value_user]
                data['uid'] = link.uid
                push_queue.put(data)

    def push_leave_folder(self, uid, user_path, group, action, user_name=None):
        # отправка тому, кто вышел из группы
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': action, 'for': 'actor'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': user_path + '/',
                        'gid': group.gid
                    }
                }
            ]
        }
        data = {
            'class': 'share_%s' % action,
            'uid': uid,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)
        if user_name is None:
            user_name = passport_entity.userinfo_summary(uid)['public_name']

        # отправка владельцу группы
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': action, 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': group.path + '/',
                        'gid': group.gid,
                        'name': name_from_path(group.path)
                    }
                },
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'uid': uid,
                        'name': user_name
                    }
                }
            ]
        }
        data = {
            'class': 'share_%s' % action,
            'uid': group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        push_queue.put(data)

        # отправка остальным пользователям каталога
        xiva_value_folder = {
            'tag': 'folder',
            'value': '',
            'parameters': {
                'gid': group.gid
            }
        }
        xiva_value_user = {
            'tag': 'user',
            'value': '',
            'parameters': {
                'uid': uid,
                'name': user_name
            }
        }
        data = {
            'class': 'share_%s' % action,
            'uid': None,
            'new_version': 1,
            'xiva_data': {
                'root': {
                    'tag': 'share',
                    'parameters': {
                        'type': action,
                        'for': 'user'
                    }
                },
                'values': []
            },
            'connection_id': '',
            'operation': 'action'
        }
        for link in group.iterlinks():
            if link.uid == uid:
                continue
            else:
                xiva_value_folder['parameters']['path'] = link.path + '/'
                xiva_value_folder['parameters']['name'] = name_from_path(link.path)
                data['xiva_data']['values'] = [xiva_value_folder, xiva_value_user]
                data['uid'] = link.uid
                push_queue.put(data)

    def push_invite_rights_changed(self, invite):
        # отправка владельцу группы
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'rights_changed', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': invite.group.path + '/',
                        'name': invite.group.folder_name,
                        'gid': invite.group.gid,
                        'rights': invite.rights
                    }
                }
            ]
        }
        data = {
            'class': SHARE_RIGHTS_CHANGED,
            'uid': invite.group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)

    def push_folder_rights_changed(self, uid, user_path, rights, group):
        # отправка тому, у кого поменяли права
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'rights_changed', 'for': 'actor'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': user_path + '/',
                        'name': name_from_path(user_path),
                        'gid': group.gid,
                        'rights': rights
                    }
                }
            ]
        }
        data = {
            'class': SHARE_RIGHTS_CHANGED,
            'uid': uid,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        push_queue.put(data)
        user_name = passport_entity.userinfo_summary(uid)['public_name']

        # отправка владельцу группы
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'rights_changed', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': group.path + '/',
                        'name': name_from_path(group.path),
                        'gid': group.gid,
                        'rights': rights
                    }
                },
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'uid': uid,
                        'name': user_name
                    }
                }
            ]
        }
        data = {
            'class': SHARE_RIGHTS_CHANGED,
            'uid': group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)

        # отправка остальным пользователям каталога
        xiva_value_folder = {
            'tag': 'folder',
            'value': '',
            'parameters': {
                'gid': group.gid,
                'rights': rights
            }
        }
        xiva_value_user = {
            'tag': 'user',
            'value': '',
            'parameters': {
                'uid': uid,
                'name': user_name
            }
        }
        data = {
            'class': SHARE_RIGHTS_CHANGED,
            'uid': None,
            'new_version': 1,
            'xiva_data': {
                'root': {
                    'tag': 'share',
                    'parameters': {
                        'type': 'rights_changed',
                        'for': 'user'
                    }
                },
                'values': []
            },
            'connection_id': '',
            'operation': 'action'
        }
        for link in group.iterlinks():
            if link.uid == uid:
                continue
            else:
                xiva_value_folder['parameters']['path'] = link.path + '/'
                xiva_value_folder['parameters']['name'] = name_from_path(link.path)
                data['xiva_data']['values'] = [xiva_value_folder, xiva_value_user]
                data['uid'] = link.uid
                push_queue.put(data)

    def push_invite_rejected(self, invite):
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_rejected', 'for': 'actor'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'gid': invite.group.gid,
                        'hash': invite.hash,
                        'name': invite.group.folder_name
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_REJECTED,
            'uid': invite.uid,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)

        if invite.uid:
            user_name = passport_entity.userinfo_summary(invite.uid)['public_name']
        else:
            user_name = invite.universe_login

        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_rejected', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'gid': invite.group.gid,
                        'name': name_from_path(invite.group.path),
                        'path': invite.group.path + '/'
                    }
                },
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'uid': invite.uid,
                        'name': user_name
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_REJECTED,
            'uid': invite.group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': '',
            'operation': 'action'
        }
        push_queue.put(data)

    def push_folder_unshared(self, group, notify_owner=True):
        if notify_owner:
            # отправка владельцу группы
            xiva_data = {
                'root': {'tag': 'share', 'parameters': {'type': 'folder_unshared', 'for': 'owner'}},
                'values': [
                    {
                        'tag': 'folder',
                        'value': '',
                        'parameters': {
                            'path': group.path + '/',
                            'gid': group.gid,
                            'name': name_from_path(group.path)
                        }
                    }
                ]
            }
            data = {
                'class': SHARE_FOLDER_UNSHARED,
                'uid': group.owner,
                'new_version': 1,
                'xiva_data': xiva_data,
                'connection_id': self.connection_id,
                'operation': 'action'
            }
            push_queue.put(data)

        # отправка остальным пользователям каталога
        xiva_value_folder = {
            'tag': 'folder',
            'value': '',
            'parameters': {
                'gid': group.gid
            }
        }
        data = {
            'class': SHARE_FOLDER_UNSHARED,
            'new_version': 1,
            'xiva_data': {
                'root': {
                    'tag': 'share',
                    'parameters': {
                        'type': 'folder_unshared',
                        'for': 'user'
                    }
                },
                'values': []
            },
            'connection_id': '',
            'operation': 'action'
        }
        for link in group.iterlinks():
            xiva_value_folder['parameters']['path'] = link.path + '/'
            xiva_value_folder['parameters']['name'] = name_from_path(link.path)
            data['xiva_data']['values'] = [xiva_value_folder]
            data['uid'] = link.uid
            push_queue.put(data)

        # Отправка по инвайтам
        for invite in group.iterinvites():
            if invite.uid and invite.status == 'new':
                xiva_data = {
                    'root': {'tag': 'share', 'parameters': {'type': 'invite_removed', 'for': 'actor'}},
                    'values': [
                        {
                            'tag': 'folder',
                            'value': '',
                            'parameters': {
                                'path': '/disk/%s/' % invite.group.folder_name,
                                'gid': invite.group.gid,
                                'name': invite.group.folder_name
                            }
                        }
                    ]
                }
                data = {
                    'class': SHARE_INVITE_REMOVED,
                    'uid': invite.uid,
                    'new_version': 1,
                    'xiva_data': xiva_data,
                    'connection_id': '',
                    'operation': 'action'
                }
                push_queue.put(data)

    def push_remove_invite(self, invite):
        # отправка владельцу группы
        xiva_data = {
            'root': {'tag': 'share', 'parameters': {'type': 'invite_removed', 'for': 'owner'}},
            'values': [
                {
                    'tag': 'folder',
                    'value': '',
                    'parameters': {
                        'path': invite.group.path + '/',
                        'gid': invite.group.gid,
                        'name': invite.group.folder_name
                    }
                },
                {
                    'tag': 'user',
                    'value': '',
                    'parameters': {
                        'universe_login': invite.universe_login,
                        'universe_service': invite.universe_service,
                        'name': invite.name,
                        'avatar': invite.avatar
                    }
                }
            ]
        }
        data = {
            'class': SHARE_INVITE_REMOVED,
            'uid': invite.group.owner,
            'new_version': 1,
            'xiva_data': xiva_data,
            'connection_id': self.connection_id,
            'operation': 'action'
        }
        push_queue.put(data)

        # отправка пользователю
        if invite.uid is not None:
            xiva_data = {
                'root': {'tag': 'share', 'parameters': {'type': 'invite_removed', 'for': 'actor'}},
                'values': [
                    {
                        'tag': 'folder',
                        'value': '',
                        'parameters': {
                            'path': '/disk/%s/' % invite.group.folder_name,
                            'gid': invite.group.gid,
                            'name': invite.group.folder_name
                        }
                    }
                ]
            }
            data = {
                'class': SHARE_INVITE_REMOVED,
                'uid': invite.uid,
                'new_version': 1,
                'xiva_data': xiva_data,
                'connection_id': '',
                'operation': 'action'
            }
            push_queue.put(data)

    def push_diff(self, data):
        if settings.feature_toggles['send_xiva_push_notification']:
            mpfs_queue.put(data, 'xiva')

    def push_space(self, uid):
        quota = Quota()
        space_info = quota.report(uid)
        push_tag = quota.get_push_type(uid)
        if push_tag:
            limit = space_info['limit']
            used = space_info['used']
            free = space_info['free']
            xiva_data = {
                'root': {
                    'tag': SPACE,
                    'parameters': {
                        'type': push_tag.type,
                        'limit': limit,
                        'free': free,
                        'used': used
                    }
                },
                'values': []
            }
            data = {
                'class': push_tag.klass,
                'uid': uid,
                'new_version': 1,
                'xiva_data': xiva_data,
                'connection_id': self.connection_id,
                'operation': 'action',
                'action_name': SPACE
            }
            push_queue.put(data)


class IndexerNotifier(object):

    @staticmethod
    def push_invite_activated(link):
        mpfs_queue.put({'uid': link.uid, 'gid': link.gid}, 'push_invite_activated')

    def push_folder_unshared(self, group, indexes):
        indexer = DiskDataIndexer()
        if indexes:
            folder_index = indexes[group.owner]
            group_folder = group.get_folder()

            indexer.push_tree(
                group.owner, folder_index.values(),
                'modify', version=group_folder.version,
                operation='folder_unshared'
            )

            for link in group.iterlinks():
                folder_index = indexes[link.uid]
                link_folder = link.get_folder()
                indexer.push_tree(
                    link.uid, folder_index.values(),
                    'delete', version=link_folder.deleted_version,
                    operation='kicked_by_unshare',
                    owner_uid=group.owner,
                    shared_guest_root_folder=link.path
                )
            indexer.flush_index_data(group_data_first=True)

    @staticmethod
    def push_leave_folder(folder_index, action, uid, version=None, link_folder=None):
        if not version and link_folder:
            version = link_folder.deleted_version

        indexer = DiskDataIndexer()
        indexer.push_tree(
            uid, folder_index.values(),
            action, version=version,
            operation='leave_folder'
        )

        indexer.flush_index_data()

mail = MailNotifier()
xiva = XivaNotifier()
index = IndexerNotifier()
