# -*- coding: utf-8 -*-
import time
import traceback
import functools

import mpfs.engine.process
import mpfs.common.errors.share as errors
import mpfs.common.errors as common_errors
import events as share_events

from mpfs.common.util.mailer import DEFAULT_LOCALE
from mpfs.config import settings
from mpfs.common.util.slice_tools import SequenceChain, AdhocSequence, offset_limit_to_slice
from mpfs.core.address import Address
from mpfs.core.social.dao.group_invites import GroupInvitesDAO
from mpfs.core.social.dao.group_links import GroupLinksDAO
from mpfs.core.social.share.group import Group, Groups
from mpfs.core.social.share.link import LinkToGroup
from mpfs.core.social.share.invite import GroupInvites, get_invite_hash
from mpfs.core.metastorage.control import groups, group_links, group_invites
from mpfs.core.services.passport_service import Passport
from mpfs.common.util import hashed, fulltime, name_from_path, email_valid
from mpfs.common.errors import FolderAlreadyExist, UserIsNotB2bError, SpamError
from mpfs.core.social.share.notifier import XivaNotifier, MailNotifier, IndexerNotifier
from mpfs.core import factory
from mpfs.core.user.base import User
from mpfs.core.filesystem.base import Filesystem
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.core.filesystem.helpers.counter import Counter
from mpfs.core.user.utils import ignores_shared_folders_space
from mpfs.core.yateam.logic import is_yateam_root
from mpfs.metastorage import DESCENDING
from mpfs.core.services.directory_service import DirectoryService, DirectoryContact
from mpfs.core.services.spam_checker_service import spam_checker
from mpfs.core.queue import mpfs_queue
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move
from mpfs.core.services.index_service import SearchIndexer
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.static.tags.experiment_names import NOTIFY_INDEX_ON_SHARING_DELETE_SHORT


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
SOCIAL_B2B_LIMITS_MONITORING = settings.social['b2b_limits_monitoring']
FEATURE_TOGGLES_IGNORE_SHARED_FOLDER_IN_QUOTA = settings.feature_toggles['ignore_shared_folder_in_quota']
FEATURE_TOGGLES_CHECK_SHARED_FOLDER_INVITE_FOR_SPAM = settings.feature_toggles['check_shared_folder_invite_for_spam']

directory_service = DirectoryService()
passport_entity = Passport()
AVAILABLE_RIGHTS = {
    640: 'owner rw, group r',
    660: 'owner rw, group rw',
}


def b2b_group_limits_reached_alarm(owner_user_or_uid, exception):
    """Отправляем письмо о превышении лимитов ОП для b2b пользователей(костыль)"""
    if not SOCIAL_B2B_LIMITS_MONITORING['enabled'] or not SOCIAL_B2B_LIMITS_MONITORING['emails']:
        return
    if not isinstance(exception, (errors.ShareFoldersLimitReached,
                                  errors.ShareGroupsLimitReached,
                                  errors.ShareUserLimitReached)):
        return

    owner_user = owner_user_or_uid
    if isinstance(owner_user_or_uid, basestring):
        owner_user = User(owner_user_or_uid)

    if not owner_user.is_b2b():
        return

    for email in SOCIAL_B2B_LIMITS_MONITORING['emails']:
        data = {
            'email_to': email,
            'template_name': 'diskSupport',
            'template_args': {
                'title': "Sharing folder limits reached for B2B users",
                'body': str(exception),
            },
        }
        mpfs_queue.put(data, 'send_email')


class ShareProcessor(object):
    def __init__(self, *args, **kwargs):
        self.connection_id = kwargs.get('connection_id', '')
        self.notifier_xiva = XivaNotifier()
        self.notifier_xiva.connection_id = self.connection_id
        self.notifier_mail = MailNotifier()
        self.notifier_index = IndexerNotifier()
        self.counter = Counter()
        self.request = None
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    @staticmethod
    def _create_group(uid, address):
        if not isinstance(address, Address):
            address = Address(address)

        if is_yateam_root(address.path):
            raise common_errors.YaTeamDirModifyError('Tried to share YaTeam root directory')

        group = Group.Create(uid, address)
        share_events.ShareCreateGroupEvent(uid=uid, group=group).send()
        return group

    @classmethod
    def create_group(cls, uid, address):
        return {'gid': cls._create_group(uid, address).gid}

    def check_folder_with_shared(self, uid, path, group_container=[], link_container=[]):
        Group.load_all(uid)
        result = None
        for group in Group._instances['uid_path'].itervalues():
            if group.owner == uid and group.path.startswith(path + '/'):
                group_container.append(group)
                result = 'owner'
        LinkToGroup.load_all(uid)
        for link in LinkToGroup._instances['uid_path'].itervalues():
            if link.uid == uid and link.path.startswith(path + '/'):
                link_container.append(link)
                if result in (None, 'owner') or link.rights < result:
                    result = link.rights
        return result

    def list_owned_groups(self, owner, path=None):
        Group.load_all(owner)
        result = filter(lambda g: g.owner == owner, Group._instances['uid_path'].itervalues())
        if path:
            result = filter(lambda g: g.path.startswith(path + '/') or g.path == path, result)
        return result

    def list_owned_links(self, uid, path=None):
        LinkToGroup.load_all(uid)
        result = filter(lambda l: l.uid == uid, LinkToGroup._instances['uid_path'].itervalues())
        if path:
            result = filter(lambda l: l.path.startswith(path + '/') or l.path == path, result)
        return result

    @classmethod
    def _format_group(cls, group, extend_from_passport=True):
        item = {
            'status': 'owner',
            'uid': group.owner,
        }
        if extend_from_passport:
            item.update(cls._extra_data_from_passport(group.owner))
        return item

    @classmethod
    def _format_link(cls, link, extend_from_passport=True):
        item = {
            'status': 'approved',
            'uid': link.uid,
            'rights': link.rights,
            'service': link.universe_service,
        }
        if extend_from_passport:
            item.update(cls._extra_data_from_passport(link.uid))
        return item

    @staticmethod
    def _format_invite(invite):
        item = {
            'status': 'proposed',
            'userid': invite.universe_login,
            'rights': invite.rights,
            'service': invite.universe_service,
        }
        # опциональные поля, перенесено как было
        for k in ('uid', 'avatar', 'name'):
            v = getattr(invite, k, None)
            if v:
                item[k] = v
        return item

    @staticmethod
    def _extra_data_from_passport(uid):
        user_info = passport_entity.userinfo(uid=uid)
        return {
            'name': user_info['display_name'],
            'userid': user_info['decoded_email'],
            'avatar': user_info['avatar'],
        }

    @classmethod
    def format_group_entities(cls, entities, extend_from_passport=True):
        formated_entities = []
        for entity in entities:
            if isinstance(entity, LinkToGroup):
                formated_entity = cls._format_link(entity, extend_from_passport=extend_from_passport)
            elif isinstance(entity, GroupInvites):
                formated_entity = cls._format_invite(entity)
            elif isinstance(entity, Group):
                formated_entity = cls._format_group(entity, extend_from_passport=extend_from_passport)
            else:
                raise TypeError('Can format only groups, links and invites. Got: %r' % type(entity))
            formated_entities.append(formated_entity)
        return formated_entities

    @staticmethod
    def list_users_in_group(gid, uid, exclude_b2b=False, offset=0, limit=None):
        group = Group.load(gid)
        # Нам необходимо отдавать такой список и правильно пагинироваться по нему с помощью offset/limit:
        #     [<владелец>, <участник_1>, ..., <участник_N>, <приглашенный_1>, ..., <приглашенный_N>]
        # Полный список участников и приглашенных пользователей лежит в БД и мы не хотим на каждый
        # запрос вытаскивать все данные, формировать единый список и делать по нему slice.
        # Поэтому мы формируем нужные ряды(sequences) данных: группа, участники и приглашенные
        # и отдаем их `SequenceChain`, который умеет минимальным кол-вом вызовом
        # методов: `__len__` и `__getitem__(slice)` забрать и отдать только нужные данные.
        sequences = [
            [group],
            AdhocSequence(
                functools.partial(group.load_links, exclude_b2b=exclude_b2b),
                functools.partial(group.count_links, exclude_b2b=exclude_b2b)
            )
        ]
        if uid == group.owner:
            # для владельца дополнительно добавляем приглашенных, но не
            # отказавшихся пользователей
            sequences.append(
                AdhocSequence(
                    functools.partial(group.load_invites, statuses=['new']),
                    functools.partial(group.count_invites, statuses=['new'])
                )
            )
        slice_obj = offset_limit_to_slice(offset, limit)
        return SequenceChain(*sequences)[slice_obj]

    def invite_to_group(self, gid, raw_address, foreign_uid, userid, provider, locale,
                        user_avatar, user_name, rights=640, auto_accept=False, user_ip=None):
        """
        Создать операцию приглашения пользователя в папку.

        :param foreign_uid: Обязательный. uid пользователя, от которого происходит запрос
        :param gid: Необходим gid или raw_address. id группы.
        :param raw_address: Необходим gid или raw_address. Путь до папки, в которую нужно пригласить пользователя. Если группы не существует, то создает ее(историч.)
        """
        GroupInvitesDAO().check_readonly()

        if gid:
            # если передан gid, то на raw_address не смотрим
            group = Group.load(gid)
            if group.owner != foreign_uid:
                raise errors.GroupConflict('Input uid: %s, group owner uid: %s' % (foreign_uid, group.owner))
        elif raw_address:
            address = Address(raw_address)
            if address.uid != foreign_uid:
                raise errors.GroupConflict('Input uid: %s, folder owner uid: %s' % (foreign_uid, address.uid))

            try:
                group = Group.load(owner=address.uid, path=address.path)
            except errors.GroupNotFound:
                # такой группы нет, но мы создаем ее(исторически сложившееся поведение)
                # это недокументированное фича и кто этим пользуется - сам себе
                # враг - пусть теперь сидит и ждет пока она создастся
                group = self._create_group(address.uid, address)
        else:
            raise NotImplementedError("Need 'gid' or 'raw_address'")

        # Проверка ограничений общих папок, связанные с уже существующей группой.
        group.check_is_full()
        # далее все оставленно как было, только все данные о группе: gid,
        # owner_uid и group_path берутся из одного места - объекта group.

        owner_info = passport_entity.userinfo(group.owner)

        # https://st.yandex-team.ru/CHEMODAN-57844
        so_resolution = None
        try:
            so_resolution = spam_checker.check(group.owner,
                                               owner_info['public_name'],
                                               owner_info['login'],
                                               userid,
                                               name_from_path(group.path), user_ip)
        except Exception as e:
            # do not fail if spam checker fails
            pass

        if provider == 'ya_directory':
            odata = {
                'entity': userid,
                'rights': rights,
                'path': group.path,
                'owner': group.owner,
                'gid': group.gid,
                'folder_name': name_from_path(group.path),
                'user_ip': user_ip,
                'auto_accept': auto_accept,
                'connection_id': self.connection_id,
            }
            from mpfs.core.operations import manager
            return manager.create_operation(group.owner, 'invites', 'invite_group_contacts', odata=odata)
        else:
            hsh = get_invite_hash(group.gid, userid, provider)

            link_args = {
                'gid': group.gid,
                'universe_login': userid,
                'universe_service': provider,
            }

            user_uid = None
            if provider == 'email' and email_valid(userid):
                user_info = passport_entity.userinfo(login=userid)
                if (user_info and
                            'uid' in user_info and
                            user_info['uid'] is not None):
                    user_uid = user_info['uid']
                    user_locale = user_info['language']

                    if not user_locale:
                        user_locale = owner_info['language']
                    if not user_locale:
                        user_locale = locale
                    if not user_locale:
                        user_locale = DEFAULT_LOCALE
                    locale = user_locale

                    link_args = {
                        'gid': gid,
                        'uid': user_uid,
                    }
                else:
                    user_locale = owner_info['language']
                    if not user_locale:
                        user_locale = locale
                    if not user_locale:
                        user_locale = DEFAULT_LOCALE
                    locale = user_locale

            if group:
                try:
                    LinkToGroup.load(**link_args)
                except errors.ShareNotFound:
                    pass
                else:
                    raise errors.GroupUserInvited()

            odata = {
                'provider': provider,
                'userid': userid,
                'template': 'sharedFolder/access',
                'rights': rights,
                'path': group.path,
                'owner': group.owner,
                'gid': group.gid,
                'hash': hsh,
                'user_avatar': user_avatar,
                'user_name': user_name,
                'connection_id': self.connection_id,
                'user_uid': user_uid,
                'auto_accept': auto_accept,
                'so_is_spam': so_resolution.is_spam if so_resolution else False,
                'so_receipt': so_resolution.receipt if so_resolution else None,
                'params': {
                    'rights': str(rights),
                    'folderName': name_from_path(group.path),
                    'hash': hsh,
                    'friendName': owner_info['public_name'],
                    'friendEmail': owner_info['decoded_email'] or owner_info['email'],
                    'friendLogin': owner_info['login'],
                    'locale': locale,
                    'gid': group.gid,
                },
            }
            from mpfs.core.operations import manager
            return manager.create_operation(group.owner, 'shared_invites', 'send', odata=odata)

    def create_group_invite(self, data):
        gid = data['gid']
        # на этом этапе группа должна существовать
        group = Group.load(gid)

        rights = data['rights']
        universe_service = data['provider']
        universe_login = data['userid']
        user_avatar = data['user_avatar']
        user_name = data['user_name']
        hsh = get_invite_hash(group.gid, universe_login, universe_service)
        user_uid = data['user_uid']
        try:
            invite = GroupInvites.load(hsh)
        except errors.GroupInviteNotFound:
            invite_data = {
                'gid': group.gid,
                'rights': rights,
                'ctime': int(time.time()),
                '_id': hsh,
                'universe_login': universe_login,
                'universe_service': universe_service,
                'group': group,
                'avatar': user_avatar,
                'name': user_name,
            }
            invite = GroupInvites(**invite_data)
        else:
            invite.renew(rights=rights, status='new')
        invite.uid = user_uid

        if user_uid:
            self.notifier_xiva.push_actor_invite_created(user_uid, group.owner, group, rights, hsh)
        self.notifier_xiva.push_owner_invite_created(invite, group)
        invite.save()

        share_events.ShareCreateInviteEvent(uid=group.owner, invite=invite).send()

        return {'hash': hsh, 'gid': group.gid}

    def join_group(self, user, group, rights, universe_service='', universe_login=''):
        """Вступить пользователем в группу

        Внимание! Нет проверки прав на действие, это должен делать вызывающий код.
        Этот метод не использует механизм инвайтов

        :param User user: пользователь, который вступает в группу
        :param Group group: группа, в которую вступает пользователь
        :param int rights: права пользователя в групппе: 660 - rw, 640 - ro
        :param str universe_service: ошметки от инвайтов, каким сервисом был приглашен пользователь
        :param str universe_login: ошметки от инвайтов, логин, на который было отправлено приглашение
        """
        self.check_rights(rights)

        user_free_space = user.info()['space']['free']
        if group.owner == user.uid:
            raise errors.GroupConflict()

        required_free_space = group.size
        ignore_shared = ignores_shared_folders_space(user=user)
        if ignore_shared:
            required_free_space = 0

        if (not ignore_shared and user_free_space <= 0) or \
                        user_free_space < required_free_space:
            raise common_errors.StorageNoSpace()

        group.check_is_full()
        if group_links.get_count(uid=user.uid) >= settings.social['max_folders_number']:
            e = errors.ShareFoldersLimitReached('User "%s" reached membership limit.' % user.uid)
            b2b_group_limits_reached_alarm(group.owner_user, e)
            raise e

        try:
            link = LinkToGroup.load_for_user(user, group)
        except errors.ShareNotFound:
            pass
        else:
            # если нужно сменить права используй `self.change_rights`
            raise errors.GroupUserInvited()

        group_root_folder = group.get_folder()
        share_root_folder_name = group_root_folder.address.name
        # https://jira.yandex-team.ru/browse/CHEMODAN-8894
        if group_root_folder.is_photostream():
            share_root_folder_name += ' (%s)' % group.owner_user.get_user_info().get('login')
        disk_root_folder = factory.get_resource(user.uid, '/disk')
        share_root_folder_addr = Filesystem.autosuffix_address(
            disk_root_folder.address.get_child(share_root_folder_name)
        )
        link = LinkToGroup.Create(user.uid, group.gid, share_root_folder_addr.path, rights)
        from mpfs.core.filesystem.resources import share as share_resources
        share_root_folder = share_resources.SharedRootFolder.Create(user.uid, share_root_folder_addr, disk_root_folder, link=link)
        share_root_folder.set_shared()
        link.universe_login = universe_login
        link.universe_service = universe_service
        link.save()
        self.counter.commit()

        #TODO создать событие вступления в группу и унести нижележащий код туда
        new_folder_data = {
            'uid': user.uid,
            'xiva_data': [
                {
                    'op': 'new',
                    'key': share_root_folder.id,
                    'fid': share_root_folder.meta['file_id']
                },
            ],
            'old_version': disk_root_folder.version,
            'new_version': share_root_folder.version,
            'operation': 'action',
            'connection_id': self.connection_id,
            'action_name': 'diff',
        }
        self.notifier_xiva.push_diff(new_folder_data)
        # на самом деле это уведомление поиска о новой папке
        # к инвайту это имеет слабое отношение
        self.notifier_index.push_invite_activated(link)
        self.notifier_xiva.push_space(user.uid)

        link.get_folder().set_request(self.request)
        return link

    def activate_group_invite(self, uid, hsh):
        user = User(uid)
        invite = GroupInvites.load(hsh)
        if not invite.is_new():
            raise errors.GroupInviteNotFound()

        try:
            link = LinkToGroup.load_for_user(user, invite.group)
        except errors.ShareNotFound:
            # новый участник группы
            link = self.join_group(
                user, invite.group, invite.rights,
                universe_service=invite.universe_service,
                universe_login=invite.universe_login
            )
        else:
            try:
                link = self.change_member_rights(user, invite.group, invite.rights)
            except errors.ShareMemberRightsNotChanged:
                raise errors.GroupUserInvited()

        invite.set_activated(uid)

        #TODO вынести в подписку на ShareActivateInviteEvent
        user_info = user.get_user_info()
        user_name = user_info['public_name']
        user_email = user_info['email']
        version = link.get_folder().version
        self.notifier_xiva.push_invite_approved(
            uid, link.path, link.group, user_name,
            version, version
        )

        share_events.ShareActivateInviteEvent(uid=uid, user=user, link=link, invite=invite).send()

        #TODO вынести в подписку на ShareActivateInviteEvent
        #===================================================================
        # send mail to owner
        owner = User(link.group.owner)
        if not owner.is_b2b():
            self.notifier_mail.user_activated_invite(user_name, user_email, link)
        #===================================================================

    @staticmethod
    def check_rights(rights):
        """Провалидировать права доступа"""
        if rights not in AVAILABLE_RIGHTS:
            raise errors.GroupConflict('Bad rights. Expected: %s. Got: %r' % (AVAILABLE_RIGHTS.keys(), rights))

    def change_member_rights(self, member_user, group, new_rights):
        """Сменить права участника группы

        Внимание! Нет проверки прав на действие, это должен делать вызывающий код.

        :param User member_user: участник группы, которому меняют права
        :param Group group: группа
        :param int new_rights: новые права
        :rtype: LinkToGroup
        """
        self.check_rights(new_rights)

        link = LinkToGroup.load_for_user(member_user, group)
        if link.rights == new_rights:
            raise errors.ShareMemberRightsNotChanged()

        prev_rights = link.rights
        link.rights = new_rights
        link.save()
        link.get_folder().update_rights()
        self.notifier_xiva.push_folder_rights_changed(member_user.uid, link.path, new_rights, link.group)

        share_events.ShareChangeRightsEvent(uid=group.owner, link=link, prev_rights=prev_rights).send()
        # ===================================================================
        # send mail to user
        self.notifier_mail.rights_changed(link)
        #===================================================================
        return link

    def change_invite_rights(self, universe_login, universe_service, group, new_rights):
        """Сменить права доступа в инвайте

        Внимание! Нет проверки прав на действие, это должен делать вызывающий код.

        :param str universe_login: логин, на который было отправлено приглашение
        :param str universe_service: каким сервисом был приглашен пользователь
        :param Group group: группа
        :param int new_rights: новые права
        :rtype: GroupInvites
        """
        self.check_rights(new_rights)

        hsh = get_invite_hash(group.gid, universe_login, universe_service)
        invite = GroupInvites.load(hsh)
        if invite.group.gid != group.gid:
            raise errors.GroupNoPermit()
        if not invite.is_new():
            raise errors.GroupInviteExpired()
        if invite.rights == new_rights:
            raise errors.ShareInviteRightsNotChanged()

        prev_rights = invite.rights
        invite.rights = new_rights
        invite.save()
        share_events.ShareChangeInviteRightsEvent(uid=group.owner, invite=invite, prev_rights=prev_rights).send()
        self.notifier_xiva.push_invite_rights_changed(invite)
        return invite

    def leave_group(self, uid, gid, **kw):
        GroupLinksDAO().check_readonly()
        folder = kw.get('folder')
        indexer = DiskDataIndexer()
        version = None
        if folder:
            link = folder.link
            if indexer.search_on:
                folder_index = folder.get_full_index()
            version = int(folder._service.remove(folder.storage_address, data=folder.dict(safe=True)).version)
        else:
            link = LinkToGroup.load(uid=uid, gid=gid, group=kw.get('group'))
            folder = link.get_folder()
            if indexer.search_on:
                folder_index = folder.get_full_index()
            version = link.remove_folder()
        link.remove()
        user_info = passport_entity.userinfo_summary(uid)
        user_name = user_info['public_name']
        user_email = user_info['email']

        self.notifier_xiva.push_leave_folder(uid, link.path, link.group, 'user_has_left', user_name)

        owner = User(link.group.owner)
        if not owner.is_b2b():
            self.notifier_mail.user_left_folder(user_name, user_email, link)

        if indexer.shared_index_on:
            self.notifier_index.push_leave_folder(folder_index, 'delete', link.uid, version=version, link_folder=folder)

        share_events.ShareLeaveGroupEvent(uid=uid, link=link).send()

    def kick_user(self, owner, gid, uid, leave_data=False):
        link = LinkToGroup.load(uid=uid, gid=gid)
        if link.group.owner != owner:
            raise errors.GroupNoPermit()
        else:
            notifier_action = 'delete'
            version = None
            link_folder = None
            indexer = DiskDataIndexer()
            try:
                link_folder = link.get_folder()
            except errors.ResourceNotFound:
                log.error('Remove group_link, associated folder not found')
            else:
                if leave_data:
                    notifier_action = 'modify'
                    link_folder.make_private_copy()
                    version = link_folder.deleted_version
                else:
                    version = link.remove_folder()
            finally:
                link.remove()
            # ===================================================================
            if indexer.search_on and indexer.shared_index_on:
                if link_folder:
                    mpfs_queue.put(
                        {
                            'gid': gid,
                            'path': link.path,
                            'file_id': link_folder.meta['file_id'],
                            'uid': uid,
                            'version': version,
                            'action': notifier_action
                        },
                        'group_user_kicked_notify_search_index'
                    )

            #===================================================================
            self.notifier_xiva.push_leave_folder(uid, link.path, link.group, 'user_was_banned')
            #===================================================================

            share_events.ShareKickFromGroupEvent(uid=owner, link=link).send()

            # send mail to user
            owner = User(link.group.owner)
            if not owner.is_b2b():
                self.notifier_mail.user_kicked(link)
            #===================================================================

    def change_owner(self, owner, gid, uid):
        if (mpfs.engine.process.usrctl().is_collection_in_postgres(owner, 'user_data') or
                mpfs.engine.process.usrctl().is_collection_in_postgres(uid, 'user_data')):
            raise errors.StorageEndpointError()

        link = LinkToGroup.load(uid=uid, gid=gid)
        if owner != link.group.owner:
            raise errors.GroupNoPermit()
        owner_path = link.group.path
        owners_folder = link.group.get_folder()
        link.get_folder().make_private_copy()
        link.group.owner = uid
        link.group.path = link.path
        link.group.save()
        link.remove()
        owners_folder.rm_content()
        new_link = LinkToGroup.Create(owner, gid, owner_path, 660)
        self.counter.commit()
        share_events.ShareChangeGroupOwnerEvent(uid=owner, link=link, new_link=new_link).send()
        # TODO: Xiva PUSH

    def list_all_folders(self, uid):
        result = {
            'owned': self.list_owned_folders(uid),
            'joined': self.list_joined_folders(uid),
        }
        return result

    def list_owned_folders(self, uid):
        result = []
        for each in groups.get_all(owner=uid):
            group = Group(**each)
            try:
                folder = group.get_folder()
            except errors.ResourceNotFound:
                error_log.error('Group folder not found owner: %s path: %s' % (group.owner, group.path))
            else:
                result.append(folder.form)
        return result

    def list_joined_folders(self, uid):
        result = []
        for each in group_links.get_all(uid=uid):
            link = LinkToGroup(**each)
            try:
                folder = link.get_folder()
            except errors.ResourceNotFound:
                error_log.error('Shared folder not found: uid: %s path: %s' % (link.uid, link.path))
            else:
                result.append(folder.form)
        return result

    def remove_folder_references(self, uid, path):
        for group_data in groups.get_all(owner=uid):
            if group_data['path'].startswith(path + '/'):
                try:
                    group = Group(**group_data)
                    self.delete_group_root(group)
                except Exception:
                    error_log.error("Can't remove group with refs")
                    error_log.error(traceback.format_exc())
        for link_data in group_links.get_all(uid=uid):
            if link_data['path'].startswith(path + '/'):
                try:
                    link = LinkToGroup(**link_data)
                    link.remove()
                    link.remove_folder()
                except Exception:
                    error_log.error("Can't remove link")
                    error_log.error(traceback.format_exc())
                    # TODO: Xiva notify, email notification

    def list_not_approved_folders(self, uid):
        result = []

        all_invites = group_invites.get_all(uid=uid)
        all_invite_gids = map(lambda x: x.get('gid'), all_invites)
        Groups().preload(all_invite_gids)

        for invite_data in all_invites:
            try:
                invite = GroupInvites(**invite_data)
            except errors.ShareError, she:
                error_log.error(she.message + ' gid: %(gid)s' % invite_data)
            else:
                if invite.is_new():
                    data = {
                        'owner_uid': invite.group.owner,
                        'owner_name': passport_entity.userinfo_summary(invite.group.owner)['public_name'],
                        'hash': invite.hash,
                        'ctime': invite.ctime,
                        'folder_name': invite.group.folder_name,
                        'status': 'proposed',
                        'rights': invite.rights,
                        'gid': invite.group.gid,
                        'size': invite.group.size,
                    }
                    result.append(data)
        return result

    @staticmethod
    def get_folder_indexes(curr_folder, only_root=False):
        curr_uid = curr_folder.uid
        if only_root:
            if is_reindexed_for_quick_move(curr_uid):
                res = {curr_folder.id: curr_folder.dict(safe=False)}
            else:
                res = curr_folder.get_full_index()
                try:
                    SearchIndexer().start_reindex_for_quick_move(curr_uid)
                except Exception:
                    error_log.warning("Can't start reindex for quick move for uid {} (optional)".format(curr_uid))
                    error_log.warning(traceback.format_exc())
        else:
            # previous logic
            res = curr_folder.get_full_index()
        return res

    def delete_group_root(self, group, notify_owner=True):
        if not isinstance(group, Group):
            group = Group.load(group)

        group_owner = group.owner
        group_folder = group.get_folder()
        link_folders = {link.uid: link.get_folder() for link in group.iterlinks()}

        indexer = DiskDataIndexer()
        indexes = {}
        is_notify_index_short_enabled = experiment_manager.is_feature_active(NOTIFY_INDEX_ON_SHARING_DELETE_SHORT)

        if indexer.search_on:
            for uid, folder in link_folders.iteritems():
                indexes[uid] = ShareProcessor.get_folder_indexes(folder, is_notify_index_short_enabled)

        group.remove()
        group_folder.set_unshared()
        group.remove_links()
        group.remove_invites()
        # ===================================================================
        # send mail to users
        self.notifier_mail.group_removed(group)
        #===================================================================
        self.notifier_xiva.push_folder_unshared(group, notify_owner=notify_owner)

        # делаем в конце, что бы в случае, когда папка ни на кого не расшарена строить индекс в конце, а не в начале
        if indexer.search_on:
            indexes[group_owner] = ShareProcessor.get_folder_indexes(group_folder, is_notify_index_short_enabled)

        self.notifier_index.push_folder_unshared(group, indexes=indexes)

    def unshare_folder(self, uid, gid):
        group = Group.load(gid)
        if group.owner != uid:
            raise errors.GroupNoPermit()
        else:
            self.delete_group_root(group)
        share_events.ShareUnshareFolderEvent(uid=uid, group=group).send()

    def reject_invite(self, uid, hsh):
        invite = GroupInvites.load(hsh)
        if not invite.uid:
            self.notifier_xiva.push_invite_rejected(invite)
            invite.set_rejected()
        elif invite.uid and invite.uid == uid:
            self.notifier_xiva.push_invite_rejected(invite)
            invite.set_rejected()
        else:
            raise errors.GroupNoPermit()
        share_events.ShareRejectInviteEvent(uid=uid, invite=invite).send()

    def remove_invite(self, uid, gid, universe_login, universe_service):
        hsh = get_invite_hash(gid, universe_login, universe_service)
        invite = GroupInvites.load(hsh)
        if invite.group.owner == uid:
            invite.remove()
            self.notifier_xiva.push_remove_invite(invite)
            share_events.ShareRemoveInviteEvent(uid=uid, invite=invite).send()
        else:
            raise errors.GroupNoPermit()

    def invite_info(self, uid, hsh):
        group_invite = GroupInvites.load(hsh)
        return group_invite.info(uid)

    def folder_info(self, uid, gid):
        try:
            shared_entity = LinkToGroup.load(uid=uid, gid=gid)
        except errors.ShareNotFound:
            shared_entity = Group.load(gid=gid, owner=uid)
            if shared_entity.owner != uid:
                raise
        shared_entity.get_folder()
        return {
            'id': shared_entity.path,
        }

    def set_group_size(self, gid):
        group = Group.load(gid)
        group.set_size()

    @classmethod
    def get_invite_like_objects(cls, status=GroupInvites.Status.ACTIVATED, gid=None, ctime_lte=None, offset=0, limit=20):
        """Получить список актуальных приглашений в расшаренную папку.

        :param status: Статус приглашения.
        :param gid: Идентификатор расшаренной папки.
        :param limit: Максимальное количество возвращаемых результатов.
        :param offset: Смещение от начала выборки подпадающей под условия запроса.
        :param ctime_lte: Максимальный таймстемп подтверждения приглашения.
        """
        if status not in (GroupInvites.Status.ACTIVATED, GroupInvites.Status.REJECTED, GroupInvites.Status.NEW):
            raise ValueError('Unexpected `status`. Got %s.' % status)

        if ctime_lte is not None and ctime_lte <= 0:
            return []

        # ATTENTION
        # Сейчас в базе есть приглашения с ctime и без, и нам надо как-то правильно
        # обрабатывать этот случай, мы приняли решение выбирать только с ctime,
        # чтоб мы могли корректно их сортировать и обращаться к этому атрибуту без AttributeError.
        spec = {'ctime': {'$exists': True}}
        if gid is not None:
            spec['gid'] = gid
        sort_ = [('ctime', DESCENDING)]
        if ctime_lte is not None:
            spec['ctime']['$lte'] = ctime_lte

        kwargs = {
            'sort': sort_,
            'limit': limit
        }
        if offset:
            kwargs['skip'] = offset

        collection = group_links
        wrapper_class = LinkToGroup

        if status in (GroupInvites.Status.REJECTED, GroupInvites.Status.NEW):
            collection = group_invites
            spec['status'] = status
            wrapper_class = GroupInvites

        objects_list = list(collection.find(spec=spec, **kwargs))
        if gid:
            group = Group.load(gid)
            for obj in objects_list:
                obj['group'] = group
        return [wrapper_class(**o) for o in objects_list]

    @staticmethod
    def check_synchronization_with_b2b_allowed(user, group):
        if not user.is_b2b():
            raise UserIsNotB2bError()
        if group.owner != user.uid:
            raise errors.GroupConflict()

    def synchronize_with_b2b(self, user, group):
        self.check_synchronization_with_b2b_allowed(user, group)
        rw_contacts, ro_contacts = directory_service.get_shared_folder_members(group.owner, group.gid)
        rw_contacts_map = {c.uid: c for c in rw_contacts if c.uid != group.owner}
        ro_contacts_map = {c.uid: c for c in ro_contacts if c.uid != group.owner}

        # достаем всех b2b-х участников группы и разбиваем их на rw и ro
        rw_group_uids, ro_group_uids = set(), set()
        for link in group.iterlinks():
            link_b2b_key = link.get_b2b_key()
            if (not link_b2b_key or
                    link_b2b_key != user.b2b_key or
                    link.uid == group.owner):
                continue
            if link.is_rw():
                rw_group_uids.add(link.uid)
            else:
                ro_group_uids.add(link.uid)

        all_contacts_uids = rw_contacts_map.viewkeys() | ro_contacts_map.viewkeys()
        all_group_uids = rw_group_uids | ro_group_uids

        error_msg_tmpl = 'synchronize_with_b2b error. b2b_key: "%s", gid: "%s", action: "%s", member_uid: "%s", exception: %r'
        error_logger = functools.partial(error_log.warning, error_msg_tmpl, user.b2b_key, group.gid)
        # удаляем из группы
        uids_to_kick = all_group_uids - all_contacts_uids
        for uid in uids_to_kick:
            try:
                self.kick_user(group.owner, group.gid, uid)
            except (errors.ShareError, common_errors.StorageInitUser) as e:
                error_logger('kick_user', uid, e)

        # у этих пользователей права сменились
        change_rights = (
            (660, rw_contacts_map.viewkeys() & ro_group_uids),
            (640, ro_contacts_map.viewkeys() & rw_group_uids)
        )
        for new_rights, uids in change_rights:
            for uid in uids:
                try:
                    self.change_member_rights(User(uid), group, new_rights)
                except (errors.ShareError, common_errors.StorageInitUser) as e:
                    error_logger('change_member_rights', uid, e)

        # новые участники ОП
        new_members = (
            (660, rw_contacts_map),
            (640, ro_contacts_map)
        )
        for rights, contacts_map in new_members:
            uids = contacts_map.viewkeys() - all_group_uids
            for uid in uids:
                contact = contacts_map[uid]
                try:
                    self.join_group(User(contact.uid), group, rights, universe_service='ya_directory', universe_login=contact.email)
                except (errors.ShareError, common_errors.StorageInitUser) as e:
                    error_logger('join_group', uid, e)
