# -*- coding: utf-8 -*-

import copy
import traceback
import time
import hashlib

import mpfs.engine.process

from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.filesystem.hardlinks.controllers import HardlinksController
from mpfs.core.services.download_stat_service import download_stat
from mpfs.core.metastorage.control import support_blocked_hids, support_block_history
from mpfs.core.support import user
from mpfs.core.support import comment
from mpfs.core.address import Address, PublicAddress
from mpfs.core.social.publicator import Publicator
from mpfs.core.services.passport_service import Passport
from mpfs.common.util import mailer
from mpfs.metastorage.mongo.util import id_for_key

UIDS_WITH_PUBLIC_HID_COUNT = settings.support['uids_with_public_hid_count']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

publicator = Publicator()
passport = Passport()
hid_whitelist = None


def _load_hid_whitelist():
    global hid_whitelist
    hid_whitelist = [x.get('hid') for x in download_stat.get_hid_whitelist()]


def hid_whitelisted(hid):
    """
    Проверить есть ли hid в белом списке
    """
    if not hid_whitelist:
        _load_hid_whitelist()

    if str(hid) in hid_whitelist:
        return True
    else:
        return False


def block_hid(hid, hid_block_type):
    if hid_whitelisted(hid):
        log.info("hid %s found in whitelist, not blocked" % hid)
    else:
        support_blocked_hids.put(hid, hid_block_type)


def uids_with_public_hid(hid):
    resources = HardlinksController().get_public_resources_by_hid(hid)
    uids = [res.uid for res in resources]
    return list(set(uids))


def block_uids_by_hid(hid, block_type, moderator, comment_text, additional_uids=None):
    """
    Блокировка пользователей, у которых есть публичный файл с определённым hid'ом

    Содержит защиту, если пользователей насобиралось больше значения из конфига,
    никого не блокирует и отправляет письмо
    """

    if hid_whitelisted(hid):
        log.info("hid %s found in whitelist, nothing blocked" % hid)
    else:
        block_hid(hid, block_type)

        public_resources = HardlinksController().get_public_resources_by_hid(hid)
        uids = set([res.uid for res in public_resources])
        uids_length = len(uids)
        additional_uids = set(additional_uids or []) - uids
        additional_uids_length = len(additional_uids)
        if additional_uids:
            uids.update(additional_uids)

        if len(uids) > UIDS_WITH_PUBLIC_HID_COUNT:
            text = u"Ссылка на публичный файл: %s <br> Uid'ы для блокировки: <br> %s <br>" % (public_resources[0].meta['short_url'], ' <br> '.join(uids))
            params = {
                'title': u"При блокировке по hid найдено больше %s uid'ов" % UIDS_WITH_PUBLIC_HID_COUNT,
                'body': text,
            }
            for email in settings.support['too_many_porno_notification_emails']:
                mailer.send(email, 'diskSupport', params)
            log.info("Too many uids found for hid %s, notification sended, nothing blocked" % hid)
        elif len(uids) > 0:
            for uid in uids:
                try:
                    user.block(uid, moderator, comment_text)
                except Exception:
                    error_log.error(traceback.format_exc())
            log.info("%s uids with public hid %s blocked." % (uids_length, hid))
            if additional_uids_length:
                log.info("%s additional uids blocked." % additional_uids_length)
        else:
            log.info("No public files with hid: %s found, only hid blocked" % hid)


def batch_block(uids, public_hashes, hids, comment_text, view, link, moderator, comment_type, notify, hid_block_type):
    for uid in uids.split(','):
        try:
            user.block(uid, moderator, comment_text)
        except Exception:
            error_log.error(traceback.format_exc())

    for public_hash in public_hashes.split(','):
        try:
            block_public_file(
                public_hash,
                comment_text,
                view,
                link,
                moderator,
                comment_type,
                notify,
                hid_block_type,
            )
        except Exception:
            error_log.error(traceback.format_exc())

    for hid in hids.split(','):
        try:
            block_hid(hid, 'only_view_delete')
        except Exception:
            error_log.error(traceback.format_exc())

    history_data = {
        'type': 'block',
        'ctime': int(time.time()),
        'moderator': moderator,
        'uids': uids,
        'hids': hids,
        'public_hashes': public_hashes,
    }
    history_data['_id'] = hashlib.md5(str(history_data)).hexdigest()
    support_block_history.insert(history_data)


def block_public_file(private_hash, comment_text, view, link, moderator, comment_type, notify, hid_block_type):
    """
    Блокировка публичного файла
    с записью коммента и логина модератора
    """
    file_dict = publicator.block(private_hash)
    data = {
        'hash': private_hash,
        'comment': comment_text,
        'address': file_dict['rawaddress'],
        'view': view,
        'ext_link': link,
        'id': id_for_key(file_dict['rawaddress']),
        'user_ip': file_dict['user_ip'],
        'ptime': file_dict['ptime'],
        'stids': file_dict['stids'],
        'hid': file_dict['hid'],
    }
    if not data['stids']:
        data.pop('stids')
    comment.create(
        file_dict['uid'],
        moderator,
        comment_type,
        data,
    )
    if hid_block_type:
        block_hid(file_dict['hid'], hid_block_type)

    if notify:
        address = Address(file_dict['rawaddress'])
        userinfo = passport.userinfo(file_dict['uid'])
        params = {
            'fileName': address.name,
            'locale': userinfo.get('language'),
        }
        mailer.send(userinfo['email'], 'sharedFile/blocked', params)

    return data
