# -*- coding: utf-8 -*-
from datetime import datetime

from bson import Binary, ObjectId

from mpfs.dao.base import (
    BaseDAO,
    BaseDAOItem, PostgresBaseDAOImplementation,
)
from mpfs.dao.fields import (
    DateTimeStructureField,
    HidField,
    ObjectIdField,
    StringField,
)
from mpfs.metastorage.postgres.queries import SQL_UPSERT_SUPPORT_BLOCKED_HIDS

from mpfs.metastorage.postgres.schema import support_blocked_hids


class SupportBlockedHidsDAOItem(BaseDAOItem):
    mongo_collection_name = 'support_blocked_hids'
    postgres_table_obj = support_blocked_hids
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=support_blocked_hids.c.id)
    block_type = StringField(mongo_path='block_type', pg_path=support_blocked_hids.c.block_type, default_value=None)
    ctime = DateTimeStructureField(mongo_path='ctime', pg_path=support_blocked_hids.c.ctime, default_value=None)
    hid = HidField(mongo_path='hid', pg_path=support_blocked_hids.c.storage_id)


class SupportBlockedHidsDAO(BaseDAO):
    dao_item_cls = SupportBlockedHidsDAOItem

    def __init__(self):
        super(SupportBlockedHidsDAO, self).__init__()
        self._pg_impl = PostgresSupportBlockedHidsDAOImplementation(self.dao_item_cls)

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, list):
            for doc in doc_or_docs:
                # Преобразовываем старые текстовые _id в ObjectId при миграции
                if '_id' not in doc or not ObjectId.is_valid(str(doc['_id'])):
                    doc['_id'] = ObjectId()
        else:
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()

        return super(SupportBlockedHidsDAO, self).insert(
            doc_or_docs, manipulate, continue_on_error, **kwargs)

    def put(self, hid, block_type):
        if not isinstance(hid, Binary):
            hid = Binary(str(hid), subtype=2)
        spec = {
            'hid': hid,
        }
        doc = {
            'hid': hid,
            'block_type': block_type,
            'ctime': datetime.now(),
        }
        super(SupportBlockedHidsDAO, self).update(self, spec, doc, upsert=True)

    def get(self, hids):
        spec = {'hid': {'$in': hids}}
        return super(SupportBlockedHidsDAO, self).find(spec)


class PostgresSupportBlockedHidsDAOImplementation(PostgresBaseDAOImplementation):
    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if not upsert or spec.keys() != ['hid']:
            return super(PostgresSupportBlockedHidsDAOImplementation, self).update(
                spec, document, upsert, multi, **kwargs)
        session = self._get_session()
        if not document.get('_id'):
            document['_id'] = ObjectId()
        doc = self.dao_item_cls.create_from_mongo_dict(document)
        with session.begin():
            return session.execute(SQL_UPSERT_SUPPORT_BLOCKED_HIDS, doc.as_raw_pg_dict())
