# -*- coding: utf-8 -*-
from datetime import datetime

from bson import ObjectId

from mpfs.dao.base import (
    BaseDAO,
    BaseDAOItem,
)
from mpfs.dao.fields import (
    DateTimeField,
    HidField,
    JsonField,
    ObjectIdField,
    StringField, DateTimeStructureField,
)

from mpfs.metastorage.postgres.schema import support_moderation_queue


class SupportModerationQueueDAOItem(BaseDAOItem):
    mongo_collection_name = 'support_moderation_queue'
    postgres_table_obj = support_moderation_queue
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=support_moderation_queue.c.id)
    created = DateTimeStructureField(mongo_path='created', pg_path=support_moderation_queue.c.created, default_value=None)
    description = StringField(mongo_path='description', pg_path=support_moderation_queue.c.description, default_value=None)
    hid = HidField(mongo_path='hid', pg_path=support_moderation_queue.c.storage_id, default_value=None)
    links = JsonField(mongo_path='links', pg_path=support_moderation_queue.c.links)
    moderation_time = DateTimeField(mongo_path='moderation_time', pg_path=support_moderation_queue.c.moderation_time, default_value=None)
    moderator = StringField(mongo_path='moderator', pg_path=support_moderation_queue.c.moderator, default_value=None)
    source = StringField(mongo_path='source', pg_path=support_moderation_queue.c.source, default_value=None)
    status = StringField(mongo_path='status', pg_path=support_moderation_queue.c.status, default_value=None)


class SupportModerationQueueDAO(BaseDAO):
    dao_item_cls = SupportModerationQueueDAOItem

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, list):
            for doc in doc_or_docs:
                # Преобразовываем старые текстовые _id в ObjectId при миграции
                if '_id' not in doc or not ObjectId.is_valid(str(doc['_id'])):
                    doc['_id'] = ObjectId()
        else:
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()

        return super(SupportModerationQueueDAO, self).insert(
            doc_or_docs, manipulate, continue_on_error, **kwargs)

    def put_public_link(self, short_url, resource_type):
        moderation_item = {
            "created": datetime.utcnow(),
            "source": "public_views",
            "links": [{"type": resource_type, "url": short_url}],
            "status": "not-moderated",
        }
        super(SupportModerationQueueDAO, self).insert(moderation_item)

    def moderation_queue(self, sort, order, bounds, filters):
        query = dict()
        for filter_, value in filters.iteritems():
            if filter_ in ("source", "status"):
                query[filter_] = value

        amount_args = dict()
        if sort:
            amount_args['sort'] = [(sort, order)]

        if bounds:
            if 'amount' in bounds:
                amount_args['limit'] = bounds['amount']
            if 'offset' in bounds:
                amount_args['skip'] = bounds['offset']

        result = []
        all_resources = super(SupportModerationQueueDAO, self).find(query, **amount_args)
        for item in all_resources:
            item['created'] = str(item['created'])
            item['_id'] = str(item['_id'])
            result.append(item)
        return result

    def moderation_queue_count(self, spec):
        return super(SupportModerationQueueDAO, self).find(spec).count()

    def set_status(self, _id, moderator, moderation_time, status):
        query = {
            '_id': ObjectId(_id),
        }
        update = {
            '$set': {'moderator': moderator,
                     'moderation_time': moderation_time,
                     'status': status},
        }
        super(SupportModerationQueueDAO, self).update(query, update)
        return {}
