# -*- coding: utf-8 -*-
from bson import ObjectId

from mpfs.common.util import to_json
from mpfs.dao.base import (
    BaseDAO,
    BaseDAOItem,
    PostgresBaseDAOImplementation,
    QueryWithParams,
)
from mpfs.dao.cursor import PostgresCursor
from mpfs.dao.fields import (
    JsonField,
    ObjectIdField,
    StringArrayField,
    StringField,
    UidField,
)
from mpfs.metastorage.postgres.queries import SQL_UPDATE_SUPPORT_MPFS, SQL_FIND_SUPPORT_MPFS

from mpfs.metastorage.postgres.schema import support_mpfs


class SupportMpfsDAOItem(BaseDAOItem):
    mongo_collection_name = 'support_mpfs'
    postgres_table_obj = support_mpfs
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=support_mpfs.c.id)
    data = JsonField(mongo_path='data', pg_path=support_mpfs.c.data)
    data_id = StringField(mongo_path='data.id', pg_path=support_mpfs.c.data_id, default_value=None)
    data_hash = StringField(mongo_path='data.hash', pg_path=support_mpfs.c.data_hash, default_value=None)
    data_stids = StringArrayField(mongo_path='data.stids', pg_path=support_mpfs.c.data_stids, default_value=None)
    uid = UidField(mongo_path='uid', pg_path=support_mpfs.c.uid)


class SupportMpfsDAO(BaseDAO):
    dao_item_cls = SupportMpfsDAOItem

    def __init__(self):
        super(SupportMpfsDAO, self).__init__()
        self._pg_impl = PostgresSupportMpfsDAOImplementation(self.dao_item_cls)

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, list):
            for doc in doc_or_docs:
                # Преобразовываем старые текстовые _id в ObjectId при миграции
                if '_id' not in doc or not ObjectId.is_valid(str(doc['_id'])):
                    doc['_id'] = ObjectId()
        else:
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()

        return super(SupportMpfsDAO, self).insert(
            doc_or_docs, manipulate, continue_on_error, **kwargs)

    def put(self, uid, rawdata):
        super(SupportMpfsDAO, self).insert({'uid': uid, 'data': rawdata, '_id': ObjectId()})

    def folder_content(self, uid, key):
        result = []
        params = {}
        if uid:
            params.update({'uid': uid})
        if key:
            for k, v in key.iteritems():
                params.update({'data.%s' % k: v})
        all_resources = super(SupportMpfsDAO, self).find(params)
        for item in all_resources:
            data = item.get('data')
            data['uid'] = item.get('uid')
            result.append(data)
        return result


class PostgresSupportMpfsDAOImplementation(PostgresBaseDAOImplementation):
    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        session = self._get_session()
        params_sql = ''
        values = {}
        if spec:
            params = []
            # Вытаскиваем из data id, hash и stids в отдельные поля
            for key in ('id', 'hash', 'stids'):
                key = 'data.%s' % key
                pg_key = key.replace('.', '_')
                if key in spec:
                    params.append('%(name)s=:%(name)s' % dict(name=pg_key))
                    values[pg_key] = spec.pop(key)

            # Поиск по остальным полям из data
            for k, v in spec.items():
                values['value%s' % len(params)] = v
                if k.startswith('data.'):
                    params.append("%s->>'%s'=:value%s" % (k.split('.')[0], k.split('.')[1], len(params)))
                else:
                    params.append('%s=:value%s' % (k, len(params)))

            params_sql = ' WHERE ' + ' AND '.join(params)
        with session.begin():
            return PostgresCursor(
                session,
                QueryWithParams(SQL_FIND_SUPPORT_MPFS + params_sql, values),
                self.dao_item_cls
            )

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if 'data.id' in spec and 'uid' in spec and len(spec.keys()) == 2:
            session = self._get_session()
            data = {k.split('.')[1]: v for k, v in document['$set'].items()}
            with session.begin():
                return session.execute(SQL_UPDATE_SUPPORT_MPFS,
                                       {'uid': spec['uid'], 'data_id': spec['data.id'], 'data': to_json(data)})
        return super(PostgresSupportMpfsDAOImplementation, self).update(spec, document, upsert, multi, **kwargs)

