# -*- coding: utf-8 -*-
import time

from bson import ObjectId

from mpfs.common.errors import SupportTooManyUsersWithProhibitedCleaningError
from mpfs.config import settings
from mpfs.dao.base import (
    BaseDAO,
    BaseDAOItem,
    PostgresBaseDAOImplementation,
)
from mpfs.dao.fields import (
    DateTimeField,
    ObjectIdField,
    StringField,
    UidField,
)
from mpfs.metastorage.postgres.queries import SQL_UPSERT_SUPPORT_PROHIBITED_CLEANING_USERS

from mpfs.metastorage.postgres.schema import support_prohibited_cleaning_users


class SupportProhibitedCleaningUsersDAOItem(BaseDAOItem):
    mongo_collection_name = 'support_prohibited_cleaning_users'
    postgres_table_obj = support_prohibited_cleaning_users
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=support_prohibited_cleaning_users.c.id)
    comment = StringField(mongo_path='comment', pg_path=support_prohibited_cleaning_users.c.comment, default_value=None)
    ctime = DateTimeField(mongo_path='ctime', pg_path=support_prohibited_cleaning_users.c.ctime, default_value=None)
    moderator = StringField(mongo_path='moderator', pg_path=support_prohibited_cleaning_users.c.moderator)
    uid = UidField(mongo_path='uid', pg_path=support_prohibited_cleaning_users.c.uid)


class SupportProhibitedCleaningUsersDAO(BaseDAO):
    dao_item_cls = SupportProhibitedCleaningUsersDAOItem

    def __init__(self):
        super(SupportProhibitedCleaningUsersDAO, self).__init__()
        self._pg_impl = PostgresSupportProhibitedCleaningUsersDAOImplementation(self.dao_item_cls)

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if isinstance(doc_or_docs, list):
            for doc in doc_or_docs:
                # Преобразовываем старые текстовые _id в ObjectId при миграции
                if '_id' not in doc or not ObjectId.is_valid(str(doc['_id'])):
                    doc['_id'] = ObjectId()
        else:
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()

        return super(SupportProhibitedCleaningUsersDAO, self).insert(
            doc_or_docs, manipulate, continue_on_error, **kwargs)

    def put(self, uid, comment, moderator):
        """Добавить одну запись в коллекцию.

        Если запись с таким `uid` уже есть, то она перезапишется.
        Максимальное количество записей в коллекции ограничено сеттингами.
        """
        count = super(SupportProhibitedCleaningUsersDAO, self).find().count()
        if count >= settings.support['prohibited_cleaning_users_max_count']:
            raise SupportTooManyUsersWithProhibitedCleaningError()

        result = super(SupportProhibitedCleaningUsersDAO, self).update({'uid': uid}, {
            'uid': uid,
            'ctime': int(time.time()),
            'comment': comment,
            'moderator': moderator
        }, upsert=True)
        return True


class PostgresSupportProhibitedCleaningUsersDAOImplementation(PostgresBaseDAOImplementation):
    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if not upsert or spec.keys() != ['uid']:
            return super(PostgresSupportProhibitedCleaningUsersDAOImplementation, self).update(
                spec, document, upsert, multi, **kwargs)
        session = self._get_session()
        if not document.get('_id'):
            document['_id'] = ObjectId()
        doc = self.dao_item_cls.create_from_mongo_dict(document)
        with session.begin():
            return session.execute(SQL_UPSERT_SUPPORT_PROHIBITED_CLEANING_USERS, doc.as_raw_pg_dict())
