# -*- coding: utf-8 -*-
"""

MPFS
SUPPORT

"""
import hashlib
import re
import time
import traceback
import urllib

import datetime

import mpfs.engine.process
from mpfs.common.errors import Forbidden
from mpfs.common.static.tags.push import LOG
from mpfs.common.util.limits.utils import UploadLimitPeriods
from mpfs.common.util.overdraft import OVERDRAFT_RESET_COUNT_FIELD
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.billing.processing import repair
from mpfs.core.bus import Bus
from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase
from mpfs.core.filesystem.quota import Quota
from mpfs.core.filesystem.resources.mail import MailFile
from mpfs.core.inactive_users_flow.logic import UNBLOCK_COMMENT_TEXT, can_be_unblock_inactive_flow
from mpfs.core.metastorage.control import support_block_history, support_blocked_hids, support_prohibited_cleaning_users
from mpfs.core.operations import manager
from mpfs.core.overdraft.utils import is_block_by_overdraft
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.clck_service import Clck
from mpfs.core.services.djfs_api_service import djfs_api_legacy
from mpfs.core.services.zaberun_service import Zaberun
from mpfs.core.social.publicator import Publicator
from mpfs.core.support import comment, controllers, errors, moderation_queue, user
from mpfs.core.user.base import User
from mpfs.core.user.constants import TRASH_AREA_PATH
from mpfs.metastorage.mongo.binary import Binary

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

zaberun = Zaberun()
publicator = Publicator()


def block_public_file(req):
    return controllers.block_public_file(req.private_hash, req.comment, req.view, req.link, req.moderator, req.type,
                                         req.notify, req.hid_block_type)


def uids_with_public_hid(req):
    return controllers.uids_with_public_hid(req.hid)


def batch_block(req):
    return controllers.batch_block(req.uids, req.public_hashes, req.hids, req.comment_text, req.view, req.link,
                                   req.moderator, req.comment_type, req.notify, req.hid_block_type)


def unblock_file(req):
    """
    Разблокировать заблокированный файл
    """
    publicator.unblock(req.uid, req.path)
    comment_type = 'unblock_file'
    data = {
        'address': req.path,
    }

    comment.create(
        req.uid,
        req.moderator,
        comment_type,
        data,
    )

    if req.unblock_hid:
        resource = factory.get_resource(req.uid, req.path)
        support_blocked_hids.remove(hid=resource.hid)


def get_comments(req):
    """
    Получить комменты по uid
    """
    return comment.select(req.uid)


def update_comment(req):
    """
    Изменить поля комментария
    """
    data = dict([(k, v) for (k, v) in req.data.iteritems() if v is not None])
    address = Address(req.address)
    return comment.update(address.uid, {'address': address.id}, data)


def find_by_hash(req):
    """
    Найти заблокированный файл по хэшу
    """
    c = comment.find_by_hash(req.hash)
    if c:
        return c

    symlink = publicator.get_symlink(req.hash, allow_deleted=True)
    return {
        'ptime': symlink.ctime,
        'user_ip': symlink.user_ip,
        'uid': symlink.public_uid,
    }


def block_user(req):
    """
    Заблокировать пользователя
    """
    return user.block(req.uid, req.moderator, req.comment)


def unblock_user(req):
    """
    Разблокировать пользователя
    """
    return user.unblock(req.uid, req.moderator, req.comment)


def get_public_file_address(req):
    """
    Получить адрес файла по публичному хэшу
    """
    return publicator.get_address(req.private_hash)


def download_url(req):
    """
    Получить адрес файла по публичному хэшу
    """
    address = Address(req.path)
    resource = factory.get_resource(req.uid, address)
    if isinstance(resource, MailFile):
        return resource.get_webattach_url()
    else:
        return resource.get_url()


def trash_restore_all(req):
    """
    Восстановить всё из корзины
    """
    mpfs_queue.put({'uid': req.uid}, 'trash_restore_all')
    return True


def restore_deleted(req):
    """
    Восстановить удалённые файлы
    """
    mpfs_queue.put(
        {
            'uid': req.uid,
            'dest': req.dest,
            'path': req.path,
            'force': req.force,
        },
        'restore_deleted'
    )
    return True


def async_block_uids_by_hid(req):
    """
    Поставить задачу на блокировку всех uid'ов, у которых был опубликован файл с заданным hid'ом
    """
    result = mpfs_queue.put(
        {
            'hid': req.hid,
            'type': req.type,
            'moderator': req.moderator,
            'comment_text': req.comment_text,
            'additional_uids': req.additional_uids
        },
        'block_uids_by_hid'
    )
    return str(result)


def push_log_notify(req):
    """
    Послать нотифай на получение лога от клиента
    """
    mpfs_queue.put(
        {
            'uid': req.uid,
            'class': LOG,
            'operation': 'action',
            'connection_id': '',
            'old_version': 0,
            'new_version': 0,
            'xiva_data': [
                {
                    'op': 'log',
                    'key': '/disk',
                    'type': 'dir',
                },
            ],
            'action_name': 'diff',
        },
        'xiva',
    )
    return 'ok'


def is_single_owner(req):
    owners = Link.ListOwners(req.md5, req.size, req.sha256)
    return True if len(owners) == 1 and req.uid in owners else False


def block_zaberun_url(req):
    data = {
        'stids': [zaberun.unpack_mid(req.hsh), ],
    }
    comment.create(
        req.uid,
        req.moderator,
        req.comment_type,
        data,
    )

    return True


def get_moderation_list(req):
    """
    Получить список для модерации
    """
    return moderation_queue.select(req.sort, req.order, req.bounds, req.filters)


def get_moderation_queue_count(req):
    """
    Получить список для модерации
    """
    return moderation_queue.count(status=req.status, created=req.created)


def set_status(req):
    """
    Поставить статус
    """
    return moderation_queue.set_status(req._id, req.moderator, req.status)


def set_limit_by_services(req):
    """
    Выставить лимит по услугам из billing_services

    https://st.yandex-team.ru/CHEMODAN-19881
    """
    log.info('recalculating services for %s, asked by %s' % (req.uid, req.moderator))
    repair.repair_services(req.uid)
    log.info('setting quota to %s by billing_services, asked by %s' % (req.uid, req.moderator))
    repair.recalculate_limit_by_services(req.uid)


def get_block_history(req):
    sort_field = req.sort
    if sort_field not in ["_id", "uids", "ctime", "moderator", "hids", "type", "public_hashes"]:
        sort_field = "ctime"
    order_field = req.order
    if order_field not in [0, -1, 1]:
        order_field = -1
    else:
        order_field = {0: -1, -1: -1, 1: 1}[order_field]

    total = support_block_history.find({}).count()
    result = list(support_block_history.find({}, skip=req.offset, limit=req.amount, sort=[(sort_field, order_field)]))
    return {'total': total, 'result': result}


def batch_unblock(req):
    block_operation = support_block_history.find_one({'_id': req.id})

    for uid in block_operation['uids'].split(','):
        try:
            user.unblock(uid, req.moderator, req.comment_text)
        except Exception:
            error_log.error(traceback.format_exc())

    for hid in block_operation['hids'].split(','):
        try:
            support_blocked_hids.remove(hid=Binary(str(hid), subtype=2))
        except Exception:
            error_log.error(traceback.format_exc())

    history_data = block_operation

    history_data['ctime'] = int(time.time())
    history_data['moderator'] = req.moderator
    history_data['type'] = 'unblock'
    history_data['_id'] = hashlib.md5(str(history_data)).hexdigest()

    support_block_history.insert(history_data)


def search(req):
    """
    Поиска файла по имени по коллекциям ('user_data', 'trash', 'hidden_data') напрямую в монге,
    пользоваться осторожно, т.к. индекс работает только по uid, дальше идет fullscan
    """

    name = req.query
    escaped_name = re.escape(name)

    query = {
        'uid': req.uid,
        '$or': [
            {
                'key': {'$regex': '/[^/]*%s[^/]*$' % escaped_name},
                # ищем только по именам файлов, поэтому после последнего слеша
                'type': 'file',
            },
            {
                'key': {'$regex': '/[^/]*%s[^/]*/?$' % escaped_name},
                # в конце может быть слеш, может не быть
                'type': 'dir',
            },
        ]
    }

    db = CollectionRoutedDatabase()
    collections = ('user_data', 'trash', 'hidden_data')

    counts = {}
    for coll in collections:
        total = db[coll].find(query).count()
        counts[coll] = total

    offset = req.offset
    amount = req.amount
    result = []

    for coll in collections:
        if offset > counts[coll]:
            offset -= counts[coll]
            continue

        items = db[coll].find(query, skip=offset, limit=amount)

        for item in items:
            result.append({
                'path': item['key'],
                'size': item.get('data', {}).get('size', None),
            })

        count = items.count()
        if count < amount:
            amount -= count
            offset = 0
            continue
        else:
            break

    return {'total': sum(counts.itervalues()), 'result': result}


def resolve_public_short_url(req):
    url = req.short_url
    url = urllib.unquote(url)
    public_hash = Clck().short_url_to_public_hash(url)
    return {
        'public_hash': public_hash
    }


def add_user_to_prohibited_cleaning(req):
    """Добавить пользователя в список пользователей с отключённой чисткой корзины и хидден даты."""
    added = support_prohibited_cleaning_users.put(
        uid=req.uid,
        comment=req.comment,
        moderator=req.moderator
    )
    if added:
        return {'success': True}
    else:
        return {'success': False}


def remove_user_from_prohibited_cleaning(req):
    """Удалить пользователя из списка пользователей с отключённой чисткой корзины и хидден даты."""
    support_prohibited_cleaning_users.remove(req.uid)


def list_prohibited_cleaning_users(req):
    return support_prohibited_cleaning_users.list_all()


def _check_startrek_issue(func):
    def wrapper(req):
        if re.match('\w+\-\d+', req.startrek_issue):
            func(req)
        else:
            raise errors.InvalidStartrekIssue()

    return wrapper


@_check_startrek_issue
def get_file_checksums(req):
    manager.create_operation(
        req.uid,
        'support',
        'get_file_checksums',
        odata={
            'path': req.path,
            'startrek_issue': req.startrek_issue,
        },
    )
    return {'success': True}


@_check_startrek_issue
def force_remove_folder(req):
    manager.create_operation(
        req.uid,
        'support',
        'force_remove_folder',
        odata={
            'path': req.path,
            'startrek_issue': req.startrek_issue,
        },
    )
    return {'success': True}


@_check_startrek_issue
def force_drop_trash(req):
    fs = Bus(request=req)
    fs.check_trash_lock_for(Address.Make(req.uid, TRASH_AREA_PATH).id)
    manager.create_operation(
        req.uid,
        'support',
        'force_drop_trash',
        odata={
            'path': TRASH_AREA_PATH,
            'startrek_issue': req.startrek_issue,
        },
    )
    return {'success': True}


@_check_startrek_issue
def force_restore_trash_folder(req):
    manager.create_operation(
        req.uid,
        'support',
        'force_restore_trash_folder',
        odata={
            'path': req.path,
            'startrek_issue': req.startrek_issue,
        },
    )
    return {'success': True}


@_check_startrek_issue
def force_restore_hidden_folder(req):
    manager.create_operation(
        req.uid,
        'support',
        'force_restore_hidden_folder',
        odata={
            'path': req.path,
            'startrek_issue': req.startrek_issue,
        },
    )
    return {'success': True}


@_check_startrek_issue
def reindex_faces_album(req):
    manager.create_operation(
        req.uid,
        'support',
        'reindex_faces_album',
        odata={
            'startrek_issue': req.startrek_issue,
        }
    )
    return {'success': True}


def overdraft_allow_unblock(req):
    return {'allowed': _allow_unblock_overdraft(User(req.uid))}


def overdraft_unblock(req):
    user = User(req.uid)
    if not _allow_unblock_overdraft(user):
        raise Forbidden('Not allowed unblock overdrafter')
    user.set_block(False, 'Unblock overdraft by support action')
    djfs_api_legacy.reset_overdraft(user.uid)


def _allow_unblock_overdraft(user):
    overdraft_reset_count = user.get_overdraft_info().get(OVERDRAFT_RESET_COUNT_FIELD, 0)
    return bool(user.is_blocked() and is_block_by_overdraft(user.uid) and overdraft_reset_count == 0)


def inactive_users_flow_allow_unblock(req):
    user_ = User(req.uid)
    return {'allowed': can_be_unblock_inactive_flow(user_)}


def inactive_users_flow_unblock(req):
    user_ = User(req.uid)
    if not can_be_unblock_inactive_flow(user_):
        raise Forbidden('Not allowed unblock inactive user')
    user_.set_block(False, UNBLOCK_COMMENT_TEXT)


def upload_traffic_counters(request):
    result = {}
    quota = Quota()
    for period in UploadLimitPeriods:
        counter = quota.upload_traffic_data(request.uid, period=period.value)
        if counter:
            result[period.value.name] = {
                'value': counter.data.get('bytes'),
                'started_at': str(datetime.datetime.fromtimestamp(int(counter.data.get('ctime', 0))))
            }

    return result
