# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from urllib2 import HTTPError

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.static import tags, codes
from mpfs.common.util import to_json, ctimestamp
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem.dao.folder import FolderDAO
from mpfs.core.filesystem.resources.disk import MPFSFile
from mpfs.core.metastorage.control import support_prohibited_cleaning_users
from mpfs.core.operations.base import Operation
from mpfs.core.services.djfs_albums import djfs_albums
from mpfs.core.operations.filesystem.bulk import BulkActionOperation
from mpfs.core.services.kladun_service import Kladun
from mpfs.core.services.startrek_service import startrek_service
from mpfs.core.user.base import User
from mpfs.core.user.constants import TRASH_AREA_PATH, DISK_AREA_PATH, HIDDEN_AREA_PATH


SUPPORT_FORCE_HANDLE_REENQUE_DELAY = settings.support['force_handle']['reenque_delay']
SUPPORT_FORCE_HANDLE_CHUNK_SIZE = settings.support['force_handle']['chunk_size']
SUPPORT_FORCE_HANDLE_CHUNK_COUNT = settings.support['force_handle']['chunk_count']
SUPPORT_FORCE_HANDLE_CREATE_OPERATION_COUNT_ON_ONE_HANDLE = settings.support['force_handle']['create_operation_count_on_one_handle']
SUPPORT_FORCE_HANDLE_BULK_OPERATION_TIMEOUT = settings.support['force_handle']['bulk_operation_timeout']

SUPPORT_REINDEX_ALBUMS_TIMEOUT = settings.support['reindex_albums']['timeout']
SUPPORT_REINDEX_ALBUMS_REENQUE_DELAY = settings.support['reindex_albums']['reenque_delay']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class SupportOperation(Operation):
    type = 'support'
    task_name = 'operation_support'

    def _process(self, *args, **kwargs):
        if self.data.get('is_first_exec', True):
            self._create_startrek_comment(
                u'Операция %s выполняется, идентификатор операции: %s' % (self.__class__.__name__, self.id)
            )

        self.data['is_first_exec'] = False
        self.save()

    def _create_startrek_comment(self, text):
        startrek_service.create_comment(self.data['startrek_issue'], text)

    def set_failed(self, error):
        try:
            self._create_startrek_comment(u'Операция закончилась с ошибкой')
        except Exception, e:
            error_log.error(e.__str__())

        super(SupportOperation, self).set_failed(error)


class SupportBulkActionOperation(BulkActionOperation):
    type = 'support'
    subtype = 'bulk_action'
    task_name = 'operation_support'

    @classmethod
    def is_pended(self):
        return True


class GetFileChecksumsOperation(SupportOperation):
    subtype = 'get_file_checksums'

    def _process(self, *args, **kwargs):
        super(GetFileChecksumsOperation, self)._process(*args, **kwargs)

        resource = Bus().get_resource(self.uid, self.data['path'])
        if not isinstance(resource, MPFSFile):
            raise errors.ResourceNotFound()

        checksums = Kladun().get_file_checksums(resource.file_mid())

        comment_text = u'Операция выполнена, чексумы посчитаны для пользователя %s, файл %s.\n\n' %\
                       (self.uid, self.data['path'])
        comment_text = u'%s\nЧексумы в сторадже: \n%s\n' % (
            comment_text,
            self._create_hash_text(checksums.md5, checksums.sha256, checksums.size)
        )
        comment_text = u'%s\nЧексумы в БД: \n%s\n' % (
            comment_text,
            self._create_hash_text(resource.md5(), resource.sha256(), resource.size)
        )
        if checksums.md5 == resource.md5() and checksums.sha256 == resource.sha256() and checksums.size == resource.size:
            comment_text = u'%s\n!!(green)Чексумы полностью совпадают!!' % comment_text
        else:
            comment_text = u'%s\n!!(red)Чексумы различаются, нужно эскалировать в разработку!!' % comment_text

        self._create_startrek_comment(comment_text)
        self.set_completed()

    @staticmethod
    def _create_hash_text(md5, sha256, size):
        return u'MD5: %s\nSHA256: %s\nFile size: %s' % (md5, sha256, size)


class ForceHandleBaseOperation(SupportOperation):
    """
    Операция чанками обрабатывает ресурсы. Позволяет обрабатывать паралельно файлы (например удаление или восстановление)
    Нужно в случае, если ресурсов в папке очень много (миллионы) или стандартные операции не справляются
    """
    folder_dao = FolderDAO()

    @property
    def manager(self):
        # циклический импорт
        from mpfs.core.operations import manager
        return manager

    @property
    def path(self):
        return self.data['path']

    def get_resources_count(self):
        if 'resources_count' not in self.data:
            self.data['resources_count'] = self.folder_dao.get_resource_count(self.uid, self.path)
            log.info('Find %s files' % self.data['resources_count'])

        return self.data['resources_count']

    def active_operations(self):
        return self.data.get('active_operations', [])

    def add_active_operations(self, oid):
        if 'active_operations' not in self.data:
            self.data['active_operations'] = []

        log.info('Create new operation %s' % oid)
        self.data['active_operations'].append(oid)

    def failed_operations(self):
        return self.data.get('failed_operations', [])

    def remove_operation(self, oid):
        self.data['active_operations'].remove(oid)

    def add_operations_in_failed(self, oid):
        if 'failed_operations' not in self.data:
            self.data['failed_operations'] = []
        self.data['failed_operations'].append(oid)

    def set_flag(self, flag_name, value=True):
        self.data[flag_name] = value

    def get_flag(self, flag_name):
        return self.data.get(flag_name, False)

    def is_all_files_in_queue(self):
        return self.get_flag('all_files_in_queue')

    def is_all_folders_in_queue(self):
        return self.get_flag('all_folders_in_queue')

    def is_handle_all_files(self):
        return self.get_flag('handle_all_files')

    def is_handle_all_folders(self):
        return self.get_flag('handle_all_folders')

    def _process(self, *args, **kwargs):
        super(ForceHandleBaseOperation, self)._process(*args, **kwargs)
        resources_count = self.get_resources_count()

        for oid in self.active_operations():
            log.debug('Process active operation %s' % oid)
            operation = self.manager.get_operation(self.uid, oid)

            # Если операция зависла, то фейлим ее принудительно
            if operation.is_executing() and \
               operation.dtime < datetime.utcnow() - timedelta(seconds=SUPPORT_FORCE_HANDLE_BULK_OPERATION_TIMEOUT):

                log.warn('Operation %s lose' % oid)
                operation.set_failed({'message': 'Operation updated too long ago (%s). Failing %s.' % (operation.dtime, oid)})

            if operation.is_completed():
                self.remove_operation(oid)
                if any(c[tags.STATUS] != codes.COMPLETED for c in operation.data[tags.PROTOCOL]):
                    log.debug('Operation %s completed with errors' % oid)
                    self.add_operations_in_failed(oid)
                else:
                    log.debug('Operation %s completed' % oid)

                self.data['handle_resources_count'] = self.data.get('handle_resources_count', 0) + len(operation.data[tags.PROTOCOL])
                if self.data['handle_resources_count'] > 0:
                    percent_done = float(self.data['handle_resources_count']) / resources_count
                    total_time = (ctimestamp() - self.ctime) / percent_done
                    self.data['eta'] = int(self.ctime + total_time)

            if operation.is_failed():
                self.remove_operation(oid)
                log.debug('Operation %s failed' % oid)
                self.add_operations_in_failed(oid)

        if len(self.active_operations()) == 0:
            log.info('Empty active operations')
            if self.is_all_files_in_queue():
                self.data['handle_all_files'] = True
            if self.is_all_folders_in_queue():
                self.data['handle_all_folders'] = True

        resources_for_handle_count = SUPPORT_FORCE_HANDLE_CHUNK_COUNT * SUPPORT_FORCE_HANDLE_CHUNK_SIZE * 2
        if not self.is_all_files_in_queue():
            log.debug('Process next chunk files')
            resources = self.folder_dao.get_files_determined(self.uid, self.path, resources_for_handle_count)
            log.debug('Get %s files' % len(resources))
            if self.handle_resources(resources):
                self.data['all_files_in_queue'] = True
                log.info('All files in queue')

        # Если закончили обработку всех файлов, удаляем папки
        if self.is_handle_all_files() and not self.is_all_folders_in_queue():
            log.debug('Process next chunk folders')
            resources = self.folder_dao.get_folders_determined(self.uid, self.path, resources_for_handle_count)
            log.debug('Get %s folders' % len(resources))
            if self.handle_resources(resources, max_suboperations=1):
                self.data['all_folders_in_queue'] = True
                log.info('All folders in queue')

        self.save()

        if self.is_handle_all_files() and self.is_handle_all_folders():
            Bus().inspect(self.uid, Address.Make(self.uid, DISK_AREA_PATH).id)
            log.debug('Recalculate space OK')

            if self.check_correct():
                log.debug('Operation check correct')
                self._create_startrek_comment(self.get_success_text())
            else:
                log.debug('Operation check failed')
                self._create_startrek_comment(self.get_failed_text())

            self.set_completed()
        else:
            self.reenque(SUPPORT_FORCE_HANDLE_REENQUE_DELAY)

    def handle_resources(self, resources, max_suboperations = SUPPORT_FORCE_HANDLE_CHUNK_COUNT):
        next_fid = self.data.get('next_fid')
        if next_fid:
            try:
                offset = next(
                    i for i, resource in enumerate(resources)
                    if resource['fid'].__str__() == next_fid
                )
            except StopIteration:
                offset = 0
        else:
            offset = 0

        log.info('Resources offset %d' % offset)

        allow_count = min(
            SUPPORT_FORCE_HANDLE_CREATE_OPERATION_COUNT_ON_ONE_HANDLE,
            max_suboperations - len(self.active_operations())
        )
        log.info('Try create %d operations' % allow_count)

        for i in range(allow_count):
            start_file_index = offset + i * SUPPORT_FORCE_HANDLE_CHUNK_SIZE
            end_file_index = start_file_index + SUPPORT_FORCE_HANDLE_CHUNK_SIZE
            operation = self.create_operation(resources[start_file_index:end_file_index])
            self.add_active_operations(operation.id)

            try:
                self.data['next_fid'] = resources[end_file_index]['fid'].__str__()
            except IndexError:
                # обработали все ресурсы
                self.data['next_fid'] = None
                return True
        return False

    def create_operation(self, resources):
        raise NotImplementedError()

    def check_correct(self):
        raise NotImplementedError()

    def get_success_text(self):
        raise NotImplementedError()

    def get_failed_text(self):
        raise NotImplementedError()


class ForceRemoveFolderOperation(ForceHandleBaseOperation):
    subtype = 'force_remove_folder'

    def create_operation(self, resources):
        return self.manager.create_operation(
            self.uid,
            'support',
            'bulk_action',
            odata={
                'cmd': to_json(
                    [{
                        'action': 'rm',
                        'params': {'uid': self.uid, 'path': '%s:%s' % (self.uid, r['path'])}
                    } for r in resources if r['path'] != DISK_AREA_PATH]
                )
            }
        )

    def check_correct(self):
        try:
            Bus().get_resource(self.uid, self.path)
            return False
        except errors.ResourceNotFound:
            return True

    def get_success_text(self):
        return u'Папка %s у пользователя %s успешно удалена' % (self.path, self.uid)

    def get_failed_text(self):
        return u'Папка %s у пользователя %s могла быть удалена не полностью, лучше проверить' % (self.path, self.uid)


class ForceDropTrashOperation(ForceHandleBaseOperation):
    subtype = 'force_drop_trash'

    def _process(self, *args, **kwargs):
        if self.data.get('is_first_exec', True):
            Bus().set_trash_lock(self.uid, self.id)
        else:
            Bus().update_lock(Address.Make(self.uid, TRASH_AREA_PATH).id)
        super(ForceDropTrashOperation, self)._process(*args, **kwargs)

    def set_completed(self):
        try:
            Bus().unset_trash_lock(self.uid)
        except Exception, e:
            error_log.error(e.__str__())

        super(ForceDropTrashOperation, self).set_completed()

    def set_failed(self, error):
        try:
            Bus().unset_trash_lock(self.uid)
        except Exception, e:
            error_log.error(e.__str__())

        super(ForceDropTrashOperation, self).set_failed(error)

    def create_operation(self, resources):
        return self.manager.create_operation(
            self.uid,
            'support',
            'bulk_action',
            odata={
                'cmd': to_json(
                    [{
                        'action': 'rm',
                        'params': {'uid': self.uid, 'path': '%s:%s' % (self.uid, r['path']), 'lock': False},
                    } for r in resources if r['path'] != TRASH_AREA_PATH]
                )
            }
        )

    def check_correct(self):
        return len(Bus().tree(self.uid, '%s:%s' % (self.uid, TRASH_AREA_PATH), deep_level=1, sort='name', order=1)['list']) == 0

    def get_success_text(self):
        return u'Корзина пользователя %s успешно очищена' % self.uid

    def get_failed_text(self):
        return u'Корзина пользователя %s могла быть очищена не полностью, лучше проверить' % self.uid


class ForceRestoreTrashFolderOperation(ForceHandleBaseOperation):
    subtype = 'force_restore_trash_folder'

    def _process(self, *args, **kwargs):
        if self.data.get('is_first_exec', True):
            support_prohibited_cleaning_users.put(
                uid=self.uid,
                comment=self.data['startrek_issue'],
                moderator='api',
                check_limit=False
            )
        super(ForceRestoreTrashFolderOperation, self)._process(*args, **kwargs)

    def set_completed(self):
        try:
            support_prohibited_cleaning_users.remove(self.uid)
        except Exception, e:
            error_log.error(e.__str__())

        super(ForceRestoreTrashFolderOperation, self).set_completed()

    def set_failed(self, error):
        try:
            support_prohibited_cleaning_users.remove(self.uid)
        except Exception, e:
            error_log.error(e.__str__())

        super(ForceRestoreTrashFolderOperation, self).set_failed(error)

    def create_operation(self, resources):
        return self.manager.create_operation(
            self.uid,
            'support',
            'bulk_action',
            odata={
                'cmd': to_json(
                    [{
                        'action': 'trash_restore',
                        'params': {'uid': self.uid, 'force': True, 'path': '%s:%s' % (self.uid, r['path'])}
                    } for r in resources if r['path'] != TRASH_AREA_PATH]
                )
            }
        )

    def check_correct(self):
        if self.path == TRASH_AREA_PATH:
            return True
        else:
            try:
                Bus().get_resource(self.uid, self.path)
                return False
            except errors.ResourceNotFound:
                return True

    def get_success_text(self):
        return u'Папка %s пользователя %s успешно восстановлена из корзины' % (self.path, self.uid)

    def get_failed_text(self):
        return u'Папка %s пользователя %s могла быть не полностью восстановлена из корзины, лучше проверить' % (self.path, self.uid)


class ForceRestoreHiddenFolderOperation(ForceHandleBaseOperation):
    subtype = 'force_restore_hidden_folder'

    def create_operation(self, resources):
        return self.manager.create_operation(
            self.uid,
            'support',
            'bulk_action',
            odata={
                'cmd': to_json(
                    [{
                        'action': 'restore_deleted',
                        'params': {'uid': self.uid, 'path': '%s:%s' % (self.uid, r['path'])}
                    } for r in resources if r['path'] != HIDDEN_AREA_PATH]
                )
            }
        )

    def check_correct(self):
        # Тут может оказаться, что папка в hidden все еще существует, поэтому всегда возвращаем True
        return True

    def get_success_text(self):
        return u'Папка %s пользователя %s успешно восстановлена из hidden\'а' % (self.path, self.uid)

    def get_failed_text(self):
        return u'Папка %s пользователя %s могла быть не полностью восстановлена из hidden\'a, лучше проверить' % (self.path, self.uid)


class ReindexFacesAlbumOperation(SupportOperation):
    subtype = 'reindex_faces_album'

    def _process(self, *args, **kwargs):
        super(ReindexFacesAlbumOperation, self)._process(*args, **kwargs)

        if not self.data.get('in_queue', False):
            djfs_albums.reindex_faces({'uid': self.uid, 'reset_albums': True})

            self.data['in_queue'] = True
            self.save()

        if ctimestamp() - self.ctime > SUPPORT_REINDEX_ALBUMS_TIMEOUT:
            raise errors.MPFSTimeout()

        user = User(self.uid)
        state = user.get_faces_indexing_state()['faces_indexing_state']

        if state == 'reindexed':
            self._create_startrek_comment(u'Переиндексация альбомов лиц для пользователя %s завершена' % self.uid)
            self.set_completed()
        else:
            self.reenque(SUPPORT_REINDEX_ALBUMS_REENQUE_DELAY)
