# -*- coding: utf-8 -*-
"""
Содержит логику обработки заявок акции "back to school"

https://st.yandex-team.ru/CHEMODAN-36810
"""
import enum

from mpfs.engine.process import get_error_log
from mpfs.core.billing.client import Client
from mpfs.core.billing.product import Product
from mpfs.core.billing.processing.common import simple_create_service
from mpfs.common.errors.billing import BillingProductIsSingletone
from mpfs.core.services.clck_service import clck
from mpfs.core.social.publicator import Publicator
from mpfs.core.queue import mpfs_queue
from mpfs.common.util import to_json


class ApplicationStatus(enum.Enum):
    ok = 'ok'
    error = 'error'
    already_used = 'already_used'
    not_accepted = 'not_accepted'
    service_already_provided = 'service_already_provided'
    not_valid_url = 'not_valid_url'


class Back2SchoolApplication(object):
    def __init__(self, email, public_url, is_assessor_accepted, is_unique, not_valid_url):
        self.email = email
        self.public_url = public_url
        self.is_assessor_accepted = is_assessor_accepted
        self.is_unique = is_unique
        self.not_valid_url = not_valid_url

    def as_dict(self):
        return {
            'email': self.email,
            'public_url': self.public_url,
            'is_assessor_accepted': self.is_assessor_accepted,
            'is_unique': self.is_unique,
            'not_valid_url': self.not_valid_url,
        }

    def as_json(self):
        return to_json(self.as_dict())

    def get_uid(self):
        public_hash = clck.short_url_to_public_hash(self.public_url)
        _, symlink_addr = Publicator().parse_private_hash(public_hash)
        return symlink_addr.uid

    def __repr__(self):
        return "%s(**%s)" % (
            self.__class__.__name__,
            self.as_dict()
        )


class SendPromoEmail(object):
    EMAIL_TEMPLATES_MAP = {
        ApplicationStatus.ok: {
            'template_name': 'spacePack/backToSchool/storageSpaceAdded',
        },
        ApplicationStatus.not_accepted: {
            'template_name': 'spacePack/backToSchool/errorCheckFailed',
        },
        ApplicationStatus.already_used: {
            'template_name': 'spacePack/backToSchool/errorRequestDuplicated',
        },
        ApplicationStatus.service_already_provided: {
            'template_name': 'spacePack/backToSchool/errorRequestDuplicated',
        },
        ApplicationStatus.not_valid_url: {
            'template_name': 'spacePack/backToSchool/errorCorruptedLink',
        },
    }

    @classmethod
    def send(cls, email, status):
        try:
            extra_data = cls.EMAIL_TEMPLATES_MAP[status]
        except KeyError:
            return

        if not extra_data['template_name']:
            return

        data = {
            'sender_email': 'promo-support@disk.yandex.ru',
            'email_to': email,
            'template_args': None,
        }
        data.update(extra_data)
        mpfs_queue.put(data, 'send_email')


def process_application(raw_application):
    """Обработка заявок по акции"""
    status = None
    try:
        application = Back2SchoolApplication(**raw_application)
        if application.not_valid_url:
            status = ApplicationStatus.not_valid_url
        elif not application.is_unique:
            # Отправляет c email'а модерируемой рассылки письмо о том, что данный студенческий уже участвовал в акции. Конец.
            status = ApplicationStatus.already_used
        elif not application.is_assessor_accepted:
            # Отправляет с email'а модерируемой рассылки письмо о том, что заявка не прошла проверку. Конец.
            status = ApplicationStatus.not_accepted
        else:
            uid = application.get_uid()
            product = Product('back_to_school')
            client = Client(uid)
            try:
                simple_create_service(client=client, product=product, send_email=True)
                status = ApplicationStatus.ok
            except BillingProductIsSingletone:
                # Отправляет с email'а модерируемой рассыкли письмо о том, что у пользователя уже есть место за участие в акции. Конец.
                status = ApplicationStatus.service_already_provided
    except Exception:
        get_error_log().exception("Error during processing: %r" % raw_application)
        return ApplicationStatus.error
    else:
        SendPromoEmail.send(application.email, status)
        return status
