# -*- coding: utf-8 -*-
"""

MPFS
CORE

Общий пользователь

"""
import time
import traceback
import datetime
import uuid

from dateutil.relativedelta import relativedelta

import mpfs.engine.process
from mpfs.common.static.tags.experiment_names import REQUESTS_TO_COMMON_FOR_GROUPS_SET_FLAG, FORBIDDEN_PHOTOUNLIM
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.overdraft import OVERDRAFT_DATE_FIELD, LIGHT_OVERDRAFT_STATUS, \
    HARD_OVERDRAFT_STATUS, BLOCK_OVERDRAFT_STATUS, OVERDRAFT_STATUS_FIELD, NO_OVERDRAFT_STATUS, \
    OVERDRAFT_BLOCK_DATE_FIELD, OVERDRAFT_HARD_DATE_FIELD, OVERDRAFT_KEY, OVERDRAFT_RESET_COUNT_FIELD
from mpfs.common.util.video_unlim import COUNTRY_KEY, COUNTRY_FIELD
from mpfs.core.albums.dao.albums_info import AlbumsInfoDAO
from mpfs.core.billing.dao.overdraft import OverdraftDAO
from mpfs.core.billing.product.default_products_service import DefaultProductsService
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.promo_codes.logic.come_back_users_promo import provide_discount_if_needed
from mpfs.core.queue import mpfs_queue
from mpfs.core.support import comment
from mpfs.engine.process import get_global_tld

from mpfs.config import settings
from mpfs.common import errors
from mpfs.common.static.tags.billing import SID
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.core.billing import ServiceList, Service
from mpfs.core.services.passport_service import (
    passport,
    passport_dedicated,
)
from mpfs.core.address import Address
from mpfs.core.billing.client import Client
from mpfs.core.billing.market import Market
from mpfs.core.bus import Bus
from mpfs.core import billing, factory
from mpfs.core.billing.processing.common import simple_create_service, simple_delete_service
from mpfs.core.user.constants import *
from mpfs.core.user.dao.user import UserDAO
from mpfs.core.metastorage.control import user_index, disk, photounlim, disk_info
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.office.util import was_editor_used
from mpfs.core.organizations.dao.organizations import OrganizationDAO
from mpfs.core.user_activity_info.dao import UserActivityInfoDAO, UserActivityDAOItem
from mpfs.core.filesystem.quota import Quota
from mpfs.metastorage.postgres.schema import PlatformType


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
push_queue = PushQueue()

usrctl = mpfs.engine.process.usrctl()
user_activity_info_dao = UserActivityInfoDAO()
albums_info_dao = AlbumsInfoDAO()

ALLOWED_ONLINE_EDITOR_TYPE = ['microsoft_online', 'only_office']

FEATURE_TOGGLES_B2B_ADD_SPACE = settings.feature_toggles['b2b_add_space']
YCRID_PREFIXES_MAP = settings.user_activity_info['ycrid_prefixes_map']
PROMO_COME_BACK_USER_DISCOUNT_MINIMUM_DIFFERENCE_IN_MONTHS = settings.promo['come_back_user_discount']['minimum_difference_in_months']
USER_OVERDRAFT_RESTRICTIONS_THRESHOLD = settings.user['overdraft']['restrictions_threshold']
FEATURE_TOGGLES_CREATE_GEO_ALBUMS_FOR_NEW_USERS = settings.feature_toggles['create_geo_albums_for_new_users']


def log_smart_common_requests(message):
    SMART_COMMON_REQUESTS_PREFIX = 'smart common requests: '
    log.info(SMART_COMMON_REQUESTS_PREFIX + message)


class CommonUser(object):

    required_user_domains = None
    required_folders = None
    required_files = None
    predefined_content_folders = None
    predefined_content_files = None
    subscribe_passport = None
    type = None

    def __init__(self, uid, project='disk', info=None):
        self.uid      = uid
        self.project  = project
        self.locale   = DEFAULT_LOCALE
        self.reg_time = None
        self.b2b_key  = None
        self._is_mailish = None
        self.yateam_uid = None
        self._pdd = None
        self._is_domain_allowed_for_office = None
        self._unlimited_autouploading_enabled = None
        self._unlimited_photo_autouploading_enabled = None
        self._unlimited_photo_autouploading_allowed = None
        self._unlimited_video_autouploading_allowed = None
        self._user_index = None
        self._office_online_editor_type = None
        self._only_office_enabled = None
        self._online_editor_enabled = None
        self._office_selection_strategy = None
        self._advertising_enabled = None
        self._public_settings_enabled = None
        # Это поле обновляем через UserDAO, поэтому его изменения можно не заметить в рамках одного запроса при
        # повторном получении юзера из-за кеширования
        self._hancom_enabled = None

        if info:
            self._user_index = info
            self.locale = info.get('locale', DEFAULT_LOCALE)
            self.reg_time = info.get('reg_time', None)
            self.b2b_key = info.get('b2b_key', None)
            self._pdd = info.get('pdd', self._pdd)
            self.yateam_uid = info.get('yateam_uid', None)
            self._is_mailish = info.get('is_mailish', None)
            self._hancom_enabled = info.get('hancom_enabled', None)
            self._unlimited_autouploading_enabled = info.get('unlimited_autouploading_enabled', None)
            self._unlimited_photo_autouploading_enabled = info.get('unlimited_photo_autouploading_enabled', None)
            self._unlimited_photo_autouploading_allowed = info.get('unlimited_photo_autouploading_allowed', None)
            self._unlimited_video_autouploading_allowed = info.get('unlimited_video_autouploading_allowed', None)
            self._office_online_editor_type = info.get('office_online_editor_type', None)
            self._only_office_enabled = info.get('only_office_enabled', None)
            self._online_editor_enabled = info.get('online_editor_enabled', None)
            self._office_selection_strategy = info.get('office_selection_strategy', None)
            self._advertising_enabled = info.get('advertising_enabled', None)
            self._public_settings_enabled = info.get('public_settings_enabled', None)

    @classmethod
    def is_for(cls, type):
        return cls.type == type

    def get_user_info(self):
        """Получить userinfo из паспорта

        Результат кешируется сервисом паспорта
        """
        return passport.userinfo(self.uid)

    def is_pdd(self):
        """Проверить является ли юзер ПДД.

        .. notes::
            Проверка производится на основе значения `domain` в ответе Паспота.
            @fantasy утверждает, что `domain` у ПДД-юзера не меняется.

        :rtype: bool
        """
        if self.get_pdd_domain() is not None:
            return True
        return False

    def get_pdd_domain(self):
        if isinstance(self._pdd, dict):
            return self._pdd.get('domain')
        return None

    def is_domain_allowed_for_office(self):
        if self._is_domain_allowed_for_office is None:
            self._is_domain_allowed_for_office = False
            domain_or_domains = self.get_pdd_domain()
            if domain_or_domains:
                if self.is_b2b():
                    domain_or_domains = passport.get_all_domain_list(domain_or_domains)
                if was_editor_used(domain_or_domains):
                    self._is_domain_allowed_for_office = True
        return self._is_domain_allowed_for_office

    def update_pdd_domain(self, domain):
        usrctl.update(self.uid, self.type, self.get_supported_locale(), pdd={'domain': domain})

    @classmethod
    def NeedInit(cls, uid, project, db=True, data=True, sid=True):
        '''
        Проверка, нуждается ли пользователь в инициации
        '''

        def _user_exists():
            if db:
                if usrctl.check(uid):
                    return True
                else:
                    return False
            else:
                return True

        def _domains_exist():
            if data:
                for contr in cls.required_user_domains:
                    if not contr.check(uid):
                        return False
                return True
            else:
                return True

        def _folders_exist():
            if data:
                for path, storage in cls.required_folders:
                    response = storage.show(uid, path)
                    if response.value is None:
                        return False
                return True
            else:
                return True

        def _files_exist():
            if data:
                for file, storage in cls.required_files:
                    response = storage.show(uid, file[0])
                    if response.value is None:
                        return False
                return True
            else:
                return True

        def _subscribed():
            if not sid:
                return True

            userinfo = passport.userinfo(uid)
            has_disk = userinfo.get('has_disk', False)
            is_mailish = userinfo.get('is_mailish', False)

            if is_mailish:
                return True

            return has_disk

        return not (
            _user_exists() and
            _domains_exist() and
            _folders_exist() and
            _files_exist() and
            _subscribed()
        )

    @classmethod
    def Create(cls, uid, project='disk', **params):
        pass

    @staticmethod
    def _bind_to_market(uid, country):
        market = Market.from_country(country)
        Client(uid).bind_market(market)

    @classmethod
    def _init_geo_albums(cls, uid):
        if FEATURE_TOGGLES_CREATE_GEO_ALBUMS_FOR_NEW_USERS:
            albums_info_dao.create_base_revision(uid)

    @staticmethod
    def _save_country(uid, country):
        country_info = {'ctime': int(time.time()), COUNTRY_FIELD: country}
        disk_info.put(uid, COUNTRY_KEY, country_info)

    @classmethod
    def _create_initial_services(cls, uid, **params):
        pass

    @classmethod
    def _create_user_in_db(cls, uid, locale, shard=None, b2b_key=None, pdd=None, is_mailish=None, hancom_enabled=None,
                           lock_key=None, enable_quick_move=False, is_paid=False, faces_indexing_state=None,
                           office_selection_strategy=None, default_product_id=None):
        '''
        Создаем базу данных под юзера
        '''
        if usrctl.is_user_completely_initialized(uid):
            log.debug('updating repository for %s, locale %s, type STANDART' % (uid, locale))
            usrctl.update(uid, cls.type, locale, b2b_key, pdd, is_mailish, hancom_enabled)
            if hancom_enabled:
                log.info('Enabled Hancom for uid=%s' % uid)
            else:
                log.info('Disabled Hancom for uid=%s' % uid)
        else:
            log.debug('creating repository for %s, locale %s, type STANDART' % (uid, locale))
            usrctl.create(uid, cls.type, locale, shard, b2b_key, pdd, is_mailish, hancom_enabled, lock_key,
                          enable_quick_move, is_paid, faces_indexing_state,
                          office_selection_strategy=office_selection_strategy, default_product_id=default_product_id)
            if hancom_enabled:
                log.info('Enabled Hancom for uid=%s' % uid)
            if office_selection_strategy:
                log.info('setting office editor selection strategy for uid=%s strategy=%s' % (
                    uid, office_selection_strategy
                ))

    @classmethod
    def _create_domains(cls, uid):
        '''
        Создаем домены в базе
        '''
        for contr in cls.required_user_domains:
            if not contr.check(uid):
                try:
                    contr.create(uid)
                except errors.StorageDomainAlreadyExists:
                    log.info('Domain "%s" for user %s already exists, skipping' % (contr.name, uid))
                else:
                    log.debug('created %s domain for %s' % (contr.name, uid))

    @classmethod
    def _create_system_folders(cls, uid):
        '''
        Создаем системные каталоги в доменах
        '''
        disk_data_indexer = DiskDataIndexer()
        try:
            for path, storage in cls.required_folders:
                try:
                    storage.show_single(uid, path, None)
                except errors.StorageNotFound:
                    log.debug('creating folder %s:%s' % (uid, path))
                    try:
                        storage.make_folder(uid, path, {})
                    except errors.StorageFolderAlreadyExist:
                        log.info('Folder "%s" for user %s already exists, skipping' % (path, uid))
                    else:
                        folder = factory.get_resource(uid, path)
                        if hasattr(folder, 'file_id'):
                            folder.meta['file_id'] = folder.file_id
                            disk_data_indexer.push(folder, 'modify', operation='mkdir')
        finally:
            disk_data_indexer.flush_index_data()

    @classmethod
    def _create_system_files(cls, uid):
        '''
        Создаем системные файлы в доменах
        '''
        for file_data, storage in cls.required_files:
            file_name, file_value = file_data
            try:
                storage.show_single(uid, file_name, None)
            except errors.StorageNotFound:
                log.debug('creating file %s:%s' % (uid, file_name))
                storage.put(uid, file_name, file_value)

    @classmethod
    def _create_user_folders(cls, uid, locale):
        '''
        Создаем пользователькие каталоги в Диске
        '''
        if cls.predefined_content_folders is None:
            return

        fs = Bus(async=True)
        for folder in cls.predefined_content_folders.get(locale, []):
            response = disk.show(uid, folder)
            if response.value is None:
                address = Address.Make(uid, folder)
                log.debug('creating folder %s' % address.id)
                fs.mkdir(uid, address.id)

    @classmethod
    def _copy_predefined_files(cls, uid, locale):
        '''
        Складируем газеты и макулатуру пользователю в Диск
        '''
        if cls.predefined_content_files is None:
            return

        fs = Bus(async=True)
        for src_path, dst_path in cls.predefined_content_files.get(locale, {}).items():
            try:
                src_address = Address.Make(SHARE_UID, src_path)
                dst_address = Address.Make(uid, dst_path)
                original = fs.resource(SHARE_UID, src_address.id)
                md5, sha256, size = original.md5(), original.sha256(), original.get_size()
                log.debug('copying file %s' % dst_address.id)
                if settings.feature_toggles['hardlink_predefined_content']:
                    fs.hardlink_copy(uid, dst_address.id, md5, size, sha256, type='store')
                else:
                    fs.copy_resource(uid, src_address.id, dst_address.id, force=False, lock_source=False)
            except Exception:
                log.warning('cannot copy %s to %s' % (src_address.id, dst_address.id))
                error_log.error(traceback.format_exc())

    @classmethod
    def _release_lock(cls, uid):
        usrctl.release_lock(uid)

    @classmethod
    def _remove_own_lock_timestamp(cls, uid, lock_key):
        usrctl.remove_own_lock_timestamp(uid, lock_key)

    @classmethod
    def _post_process_user(cls, uid, userinfo, *args, **kwargs):
        pass

    @classmethod
    def passport_subscribe(cls, uid, sid='cloud'):
        if cls.subscribe_passport:
            try:
                passport_dedicated.subscribe(uid, sid)
            except (errors.PassportBadResult, errors.PassportNoResponse):
                if settings.feature_toggles['use_passport']:
                    raise

    @classmethod
    def passport_unsubscribe(cls, uid, sid='cloud'):
        if cls.subscribe_passport:
            try:
                passport_dedicated.unsubscribe(uid, sid)
            except (errors.PassportBadResult, errors.PassportNoResponse):
                if settings.feature_toggles['use_passport']:
                    raise

    @classmethod
    def Check(cls, uid, *args, **kwargs):
        db_info = usrctl.check(uid)
        if not db_info or db_info.get('type') != cls.type:
            raise errors.StorageInitUser()

    def get_supported_locale(self, ext_locale=None):
        '''
        Определение локали юзера
        '''
        ext_locale = self.locale if not ext_locale else ext_locale
        return DEFAULT_LOCALE if (ext_locale is None or ext_locale not in SUPPORTED_LOCALES) else ext_locale

    def get_default_folders(self):
        locale = self.get_supported_locale()
        result = {}
        for key, value in DEFAULT_FOLDERS.iteritems():
            result[key] = value.get(locale)
        return result

    def remove_user(self):
        usrctl.remove(self.uid)

    def check_karma(self, karma=None):
        '''
        Нечеловеческой силы метод проверки кармы пользователя
        '''
        karma = int(passport.userinfo(self.uid).get('karma'))
        if karma == 100:
            raise errors.UserKarmaBad()

    def set_block(self, blocked, comment_for_block=None):
        doc = {"$set": {'blocked': blocked}}
        spec = {'_id': self.uid}
        user_index.update(spec, doc)

        if comment_for_block:
            comment.create(
                self.uid,
                'system',
                'block_user' if blocked else 'unblock_user',
                data={
                    'comment': comment_for_block,
                    'address': 'account',
                }
            )

    def set_deleted(self):
        doc = {"$set": {'deleted': datetime.datetime.now()}}
        spec = {'_id': self.uid}
        user_index.update(spec, doc)

    def check_deleted(self):
        '''Возвращает дату удаления или None'''
        return usrctl.check(self.uid).get('deleted')

    def check_blocked(self):
        '''
        Проверить существование и блокировку в одном запросе
        '''
        user_record = user_index.check_user(self.uid)
        if user_record and user_record.get('blocked') == 1:
            raise errors.UserBlocked()

    def is_advertising_enabled(self, tld=None):
        """Включать ли рекламу для юзера."""
        if tld is None:
            tld = get_global_tld()

        if self.is_paid():
            return False
        if self.is_b2b():
            return False
        if tld in ('com', 'com.tr'):
            return False
        return True

    def is_b2b(self):
        return self.b2b_key is not None

    def is_mailish(self):
        """Проверить является ли пользователь `mailish`."""
        return bool(self._is_mailish)

    def is_hancom_enabled(self):
        """Проверить, включен ли для пользователья Hancom"""
        return self._hancom_enabled is True

    def get_online_editor(self):
        """microsoft - это редактор по умолчанию"""
        return self._office_online_editor_type

    def get_only_office_enabled(self):
        return self._only_office_enabled

    def get_office_selection_strategy(self):
        return self._office_selection_strategy

    def set_office_selection_strategy(self, value):
        UserDAO().set_office_selection_strategy(self.uid, value)
        self._office_selection_strategy = value

    def get_online_editor_enabled(self):
        return self._online_editor_enabled

    def get_advertising_enabled(self):
        return self._advertising_enabled

    def get_public_settings_enabled(self):
        return self._public_settings_enabled or self.only_disk_pro_without_ps_billing_and_b2b_enabled

    def is_unlimited_autouploading_enabled(self):
        """Проверить, доступна ли безлимитная автозагрузка с телефона"""
        return self._unlimited_autouploading_enabled is True

    def is_unlimited_photo_autouploading_allowed(self):
        if experiment_manager.is_feature_active(FORBIDDEN_PHOTOUNLIM):
            return bool(self.disk_pro_enabled or self.get_unlimited_photo_autouploading_allowed())
        return True

    def is_unlimited_video_autouploading_allowed(self):
        return bool(self.disk_pro_enabled or self.get_unlimited_video_autouploading_allowed())

    def get_unlimited_autouploading(self):
        """Проверить, доступна ли безлимитные автозагрузки с телефона"""
        return UserDAO().get_unlimited_autouploading(self.uid)

    def get_unlimited_photo_autouploading_allowed(self):
        return self._unlimited_photo_autouploading_allowed

    def get_unlimited_video_autouploading_allowed(self):
        return self._unlimited_video_autouploading_allowed

    def get_faces_indexing_state(self):
        """Получить статус переиндексации альбомов лиц"""
        return UserDAO().get_faces_indexing_state(self.uid)

    def is_unlimited_photo_autouploading_enabled(self):
        """Проверить, доступна ли безлимитная автозагрузка с телефона"""
        return self._unlimited_photo_autouploading_enabled is True

    def make_b2b(self, b2b_key):
        usrctl.update(self.uid, self.type, self.get_supported_locale(), b2b_key)
        if FEATURE_TOGGLES_B2B_ADD_SPACE:
            client = billing.Client(self.uid)
            simple_create_service(client, billing.Product('b2b_10gb'))

    def reset_b2b(self, noemail):
        """
        Сбрасывает флаг b2b_key у пользователя и удаляет услуги с продуктом b2b_*, если такие есть.
        """
        send_email = not bool(noemail)
        usrctl.reset_b2b(self.uid)
        client = billing.Client(self.uid)
        for pid in ('b2b_10gb', 'b2b_250gb', 'b2b_1tb'):
            try:
                b2b_services = ServiceList(client=client, product=billing.Product(pid).pid)
                for service in b2b_services:
                    simple_delete_service(client, Service(sid=service[SID]), send_email=send_email)
            except Exception:
                log.exception('Unable to delete pid=%s for uid=%s', pid, self.uid)

    def enable_hancom(self):
        UserDAO().enable_hancom(self.uid)
        log.info('Enabled Hancom for uid=%s' % self.uid)
        self._hancom_enabled = True

    def has_geo_albums(self):
        return albums_info_dao.get_revision(self.uid) is not None

    def disable_hancom(self):
        UserDAO().disable_hancom(self.uid)
        log.info('Disabled Hancom for uid=%s' % self.uid)
        self._hancom_enabled = None

    def enable_unlimited_autouploading(self):
        UserDAO().enable_unlimited_autouploading(self.uid)
        if not photounlim.check(self.uid):
            self._initialize_photounlim_storage()
        self._unlimited_autouploading_enabled = True

    def disable_unlimited_autouploading(self):
        UserDAO().disable_unlimited_autouploading(self.uid)
        self._unlimited_autouploading_enabled = None

    def set_online_editor(self, online_editor):
        if online_editor not in ALLOWED_ONLINE_EDITOR_TYPE:
            raise errors.OnlineEditorUnknow()
        UserDAO().set_online_editor(self.uid, online_editor)
        self._office_online_editor_type = online_editor

    def set_unlimited_autouploading(self, video_status=None, video_reason='by_user', photo_status=None):
        unlimited_autouploading = self.get_unlimited_autouploading()
        if unlimited_autouploading['unlimited_video_autouploading_enabled'] is None and video_status is None:
            video_status = self._unlimited_autouploading_enabled
        if unlimited_autouploading['unlimited_photo_autouploading_enabled'] is None and photo_status is None:
            photo_status = self._unlimited_autouploading_enabled

        if video_status or photo_status:
            if not photounlim.check(self.uid):
                self._initialize_photounlim_storage()

        return UserDAO().set_unlimited_autouploading(self.uid, video_status, video_reason, photo_status)

    def set_only_office_enabled(self, value):
        UserDAO().set_only_office_enabled(self.uid, value)
        self._only_office_enabled = value

    def set_online_editor_enabled(self, value):
        UserDAO().set_online_editor_enabled(self.uid, value)
        self._online_editor_enabled = value

    def set_advertising_enabled(self, value):
        UserDAO().set_advertising_enabled(self.uid, value)
        self._advertising_enabled = value

    def set_public_settings_enabled(self, value):
        UserDAO().set_public_settings_enabled(self.uid, value)
        self._public_settings_enabled = value

    def set_unlimited_photo_autouploading_allowed(self, value):
        UserDAO().set_unlimited_photo_autouploading_allowed(self.uid, value)
        self._unlimited_photo_autouploading_allowed = value

    def set_unlimited_video_autouploading_allowed(self, value):
        UserDAO().set_unlimited_video_autouploading_allowed(self.uid, value)
        self._unlimited_video_autouploading_allowed = value

    def is_yateam(self):
        return self.yateam_uid is not None

    def make_yateam(self, yateam_uid):
        usrctl.make_yateam(self.uid, yateam_uid)
        self.yateam_uid = yateam_uid

    def reset_yateam(self):
        usrctl.reset_yateam(self.uid)
        self.yateam_uid = None

    def is_blocked(self):
        '''
        Заблокирован ли пользователь
        '''
        try:
            blocked = int(usrctl.check(self.uid)['blocked'])
        except Exception:
            blocked = 0

        return blocked

    @property
    def is_standart(self):
        '''
        Проверка, обычный ли это пользователь
        '''
        return self.type == 'standart'

    def is_free(self):
        return not self.is_paid()

    def is_paid(self):
        return False

    def set_is_paid(self, is_paid):
        pass

    def is_b2b_paid(self):
        if not self.is_b2b():
            return False
        organization = OrganizationDAO().find_by_id(self.b2b_key)
        if organization is not None:
            return organization.is_paid
        return False

    def is_all_files_versioning_enabled(self):
        return False

    def info(self):
        """
        Выдача полной информации о пользователе:
        """
        result = {
            'version': mpfs.engine.process.usrctl().version(self.uid),
            'reg_time': self.reg_time,
            'db': mpfs.engine.process.usrctl().info(self.uid)
        }

        if self.is_b2b():
            result['is_b2b'] = True
            result['b2b_key'] = self.b2b_key
            result['b2b_paid'] = self.is_b2b_paid()

        if self.is_pdd():
            result['pdd'] = {'domain': self.get_pdd_domain()}

        return result

    def set_state(self, key, value, namespace=None):
        pass

    def remove_state(self, key, namespace=None):
        pass

    def set_setting(self, key, value, namespace=None):
        pass

    def remove_setting(self, key, namespace=None):
        pass

    def public_info(self, raise_if_does_not_exist=False):
        """Возвращает публичные данные пользователя.

        Если передать ключ `raise_if_does_not_exist`, то будет вызвано исключение в случае,
        если пользователь отсутствует в паспорте.
        """
        return passport.public_userinfo(self.uid, raise_if_does_not_exist=raise_if_does_not_exist)

    def is_missing_or_blocked_in_passport(self):
        userinfo = passport.userinfo(uid=self.uid)
        if not userinfo.get('uid') or not userinfo.get('is_enabled'):
            return True
        return False

    def is_in_overdraft_for_restrictions(self, additional_space=0):
        return Quota().free(uid=self.uid) + additional_space <= -USER_OVERDRAFT_RESTRICTIONS_THRESHOLD

    def is_blocked_for_overdraft(self, additional_space=0):
        OVERDRAFT_HARD_RESTRICTION_DAYS = settings.user['overdraft']['hard_restriction_days']
        OVERDRAFT_BLOCK_RESTRICTION_DAYS = settings.user['overdraft']['block_restriction_days']
        overdraft_data = self.get_overdraft_info()
        if overdraft_data:
            overdraft_date = datetime.datetime.strptime(overdraft_data[OVERDRAFT_DATE_FIELD], '%Y-%m-%d').date()
            if self.is_in_overdraft_for_restrictions():
                return overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS) + \
                       datetime.timedelta(OVERDRAFT_BLOCK_RESTRICTION_DAYS) < datetime.date.today()

        return False

    def is_in_hard_overdraft(self, additional_space=0):
        OVERDRAFT_HARD_RESTRICTION_DAYS = settings.user['overdraft']['hard_restriction_days']
        overdraft_data = self.get_overdraft_info()
        if overdraft_data:
            overdraft_date = datetime.datetime.strptime(overdraft_data[OVERDRAFT_DATE_FIELD], '%Y-%m-%d').date()
            if self.is_in_overdraft_for_restrictions():
                return overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS) <= datetime.datetime.now().date()
            return False
        return Quota().free(uid=self.uid) + additional_space <= -USER_OVERDRAFT_RESTRICTIONS_THRESHOLD

    def is_overdrawn(self):
        """
        Определение овердрафта у пользователя
        """
        return Quota().free(uid=self.uid) <= 0

    def get_overdraft_info(self):
        """
        Информация об овердрафте
        :return:
        """
        OVERDRAFT_HARD_RESTRICTION_DAYS = settings.user['overdraft']['hard_restriction_days']
        OVERDRAFT_BLOCK_RESTRICTION_DAYS = settings.user['overdraft']['block_restriction_days']
        if self.is_in_overdraft_for_restrictions():
            overdraft_info = OverdraftDAO().get(self.uid)

            if overdraft_info:
                overdraft_date = overdraft_info.overdraft_date
                overdraft_data = {}
                overdraft_data[OVERDRAFT_DATE_FIELD] = overdraft_date.isoformat()
                overdraft_data[OVERDRAFT_STATUS_FIELD] = self.get_overdraft_status(overdraft_info.overdraft_date)
                overdraft_data[OVERDRAFT_HARD_DATE_FIELD] = (overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS)).isoformat()
                overdraft_data[OVERDRAFT_BLOCK_DATE_FIELD] = (overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS) + datetime.timedelta(OVERDRAFT_BLOCK_RESTRICTION_DAYS)).isoformat()
                overdraft_disk_info = disk_info.find_one_by_field(self.uid, {'key': OVERDRAFT_KEY})
                if overdraft_disk_info:
                    overdraft_data[OVERDRAFT_RESET_COUNT_FIELD] = overdraft_disk_info['data'][OVERDRAFT_RESET_COUNT_FIELD]

                return overdraft_data
        return {}

    def set_overdraft_info(self, date=None, force=False):
        if not date:
            date = datetime.date.today()
        overdraft_info = OverdraftDAO().get(self.uid)
        if not overdraft_info or force:
            OverdraftDAO().update_or_create(self.uid, date)

        overdraft_disk_info = disk_info.find_one_by_field(self.uid, {'key': OVERDRAFT_KEY})
        if not overdraft_disk_info:
            overdraft_info = {'ctime': int(time.time()), OVERDRAFT_RESET_COUNT_FIELD: 0}
            disk_info.put(self.uid, OVERDRAFT_KEY, overdraft_info)

    def get_overdraft_status(self, overdraft_date):
        OVERDRAFT_HARD_RESTRICTION_DAYS = settings.user['overdraft']['hard_restriction_days']
        OVERDRAFT_BLOCK_RESTRICTION_DAYS = settings.user['overdraft']['block_restriction_days']
        today = datetime.date.today()
        if isinstance(overdraft_date, basestring):
            overdraft_date = datetime.datetime.strptime(overdraft_date, '%Y-%m-%d').date()
        elif isinstance(overdraft_date, datetime.datetime):
            overdraft_date = overdraft_date.date()
        if not self.is_in_overdraft_for_restrictions():
            return NO_OVERDRAFT_STATUS
        elif today < overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS):
            return LIGHT_OVERDRAFT_STATUS
        elif today < overdraft_date + datetime.timedelta(OVERDRAFT_HARD_RESTRICTION_DAYS) + datetime.timedelta(OVERDRAFT_BLOCK_RESTRICTION_DAYS):
            return HARD_OVERDRAFT_STATUS
        return BLOCK_OVERDRAFT_STATUS

    def check_overdrawn(self):
        if self.is_overdrawn():
            raise errors.UserOverdrawn("Uid: %s" % self.uid)

    def _initialize_photounlim_storage(self):
        from mpfs.core import factory
        photounlim.create(self.uid)
        photounlim.make_folder(self.uid, PHOTOUNLIM_AREA_PATH, {})

        folder = factory.get_resource(self.uid, PHOTOUNLIM_AREA_PATH)
        disk_data_indexer = DiskDataIndexer()
        disk_data_indexer.push(folder, 'modify', operation='mkdir')
        disk_data_indexer.flush_index_data()

        log.debug('created %s domain for %s' % (photounlim.name, self.uid))

    @property
    def versioning_extended_period_enabled(self):
        return False

    @property
    def unlimited_video_autouploading_enabled(self):
        return False

    @property
    def has_mail360(self):
        return self.get_user_info().get('has_mail360')

    @property
    def priority_support_enabled(self):
        return False

    @property
    def advertising_enabled(self):
        return True

    @property
    def antifo_enabled(self):
        return True

    @property
    def disk_pro_enabled(self):
        return False

    @property
    def only_disk_pro_without_ps_billing_and_b2b_enabled(self):
        return False

    @property
    def faces_indexing_state(self):
        return None

    @property
    def desktop_folder_autosave_enabled(self):
        return False

    def get_activity_info(self):
        activity_info = user_activity_info_dao.find_by_uid(self.uid)
        return {dao_item.platform_type.value: {
            'first_activity': str(dao_item.first_activity),
            'last_activity': str(dao_item.last_activity)
        } for dao_item in activity_info}

    def update_activity_by_platform_type(self, platform_type):
        current_date = datetime.date.today()

        # сначала читаем текущую дату, чтобы не лочить строки апдейтом лишний раз
        all_current_activities = user_activity_info_dao.find_by_uid(self.uid)

        current_platform_last_activity = self.__get_activity_date_for_platform(all_current_activities, platform_type)
        if current_platform_last_activity is not None and current_date <= current_platform_last_activity:
            return

        dao_item = UserActivityDAOItem()
        dao_item.uid = self.uid
        dao_item.platform_type = platform_type
        dao_item.last_activity = dao_item.first_activity = current_date

        latest_activity_date = self.__get_latest_activity_date(all_current_activities)
        provide_discount_if_needed(self.uid, latest_activity_date, current_date)

        user_activity_info_dao.update_activity_dates(dao_item)

    def update_activity_by_ycrid(self, ycrid):
        from mpfs.common.util.activity_info_platform import ActivityInfoPlatformFactory
        platform = ActivityInfoPlatformFactory.build_by_ycrid(ycrid)
        if platform is None:
            return
        self.update_activity_by_platform_type(platform)

    def update_activity_for_mobile(self, ycrid):
        if not YcridParser.is_yandex_disk_mobile(ycrid):
            return
        self.update_activity_by_ycrid(ycrid)

    def __get_latest_activity_date(self, all_activities):
        if not all_activities:
            return None
        return max(x.last_activity for x in all_activities)

    def __get_activity_date_for_platform(self, all_activities, platform_type):
        if not all_activities:
            return None
        for activity in all_activities:
            if activity.platform_type == platform_type:
                return activity.last_activity

    def has_shared_folders(self):
        return True

    def set_has_shared_folders(self):
        if not experiment_manager.is_feature_active(REQUESTS_TO_COMMON_FOR_GROUPS_SET_FLAG):
            return

        log_smart_common_requests('setting flag for %s' % self.uid)

        value = uuid.uuid4().hex
        disk_info.put(self.uid, HAS_SHARED_FOLDERS_FIELD, value, None)
        from mpfs.core.social.share.entity import SharedEntity
        SharedEntity.set_has_shared(self.uid, True)

    def get_default_product_id(self):
        user_index_data = self._user_index
        if not user_index_data:
            user_index_data = usrctl.check(self.uid)

        return user_index_data.get('default_product_id')
