# -*- coding: utf-8 -*-
import datetime

from mpfs.dao.base import BaseDAO, BaseDAOItem
from mpfs.dao.fields import UidField, DateTimeField, StringField
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import SQL_ACQUIRE_MIGRATION_LOCK, SQL_RELEASE_PG_MIGRATION_LOCK, \
    SQL_CHECK_PG_MIGRATION_LOCK, SQL_SET_UNTIL_PG_MIGRATION_LOCK
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter
from mpfs.metastorage.postgres.schema import migration_lock


class PgMigrationLockDAOItem(BaseDAOItem):
    """Структура, описывающая лок миграции
    """
    uid = UidField(mongo_path='uid', pg_path=migration_lock.c.uid)
    lock_until = DateTimeField(mongo_path='lock_until', pg_path=migration_lock.c.lock_until)
    owner_mark = StringField(mongo_path='owner_mark', pg_path=migration_lock.c.owner_mark)


class PgMigrationLockDao(BaseDAO):
    dao_item_cls = PgMigrationLockDAOItem
    lock_time = datetime.timedelta(minutes=5)

    def __init__(self, session=None):
        super(PgMigrationLockDao, self).__init__(session)
        self.pg_query_executer = PGQueryExecuter()

    def acquire(self, uid, shard_id=None, until=None, owner_mark=None):
        """
        Поставить лок миграции пользователю
        :param uid:
        :param shard_id:
        :param until: datetime.datetime object
        :param owner_mark: str
        """
        shard_id = shard_id or self.pg_query_executer.get_shard_id(uid)
        until = until or (datetime.datetime.now() + self.lock_time)
        session = Session.create_from_shard_id(shard_id)
        cursor = session.execute(
            SQL_ACQUIRE_MIGRATION_LOCK,
            {'uid': uid, 'until': until, 'owner_mark': owner_mark},
        )

        for pg_data in cursor:
            if not pg_data['inserted']:
                raise RuntimeError('user already locked for migration')

    def release(self, uid, shard_id=None, owner_mark=None):
        shard_id = shard_id or self.pg_query_executer.get_shard_id(uid)
        session = Session.create_from_shard_id(shard_id)
        cursor = session.execute(
            SQL_RELEASE_PG_MIGRATION_LOCK,
            {'uid': uid, 'owner_mark': owner_mark},
        )

        for pg_data in cursor:
            if not pg_data['deleted']:
                raise RuntimeError('can not delete lock. Lock owner is different or lock expired')

    def set_until(self, uid, shard_id=None, until=None, owner_mark=None):
        """
        Обновить ttl лока миграции пользователю
        :param uid:
        :param shard_id:
        :param until: datetime.datetime object
        :param owner_mark: str
        """
        shard_id = shard_id or self.pg_query_executer.get_shard_id(uid)
        until = until or (datetime.datetime.now() + self.lock_time)
        session = Session.create_from_shard_id(shard_id)
        cursor = session.execute(
            SQL_SET_UNTIL_PG_MIGRATION_LOCK,
            {'uid': uid, 'until': until, 'owner_mark': owner_mark},
        )

        for pg_data in cursor:
            if not pg_data['updated']:
                raise RuntimeError('can not update lock. Lock owner is different or lock expired')

    def is_locked(self, uid, shard_id=None):
        shard_id = shard_id or self.pg_query_executer.get_shard_id(uid)
        session = Session.create_from_shard_id(shard_id)
        return session.execute(SQL_CHECK_PG_MIGRATION_LOCK, {'uid': uid, 'now': datetime.datetime.now()}).fetchone()[0]

