# -*- coding: utf-8 -*-

from mpfs.core.filesystem.dao.disk_info.disk_info import DiskInfoDAOItem
from mpfs.dao.base import BaseDAO, PostgresBaseDAOImplementation
import mpfs.common.errors as errors
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.exceptions import DatabaseConstraintError

from mpfs.metastorage.postgres.queries import (
    SQL_SELECT_SETTING_BY_UID,
    SQL_SELECT_SETTING_BY_ID_FOR_UPDATE,
    SQL_INSERT_SETTING,
    SQL_UPDATE_SETTING,
)


class SettingsDAO(BaseDAO):
    dao_item_cls = DiskInfoDAOItem

    def __init__(self):
        super(SettingsDAO, self).__init__()
        self._pg_impl = PostgresSettingsDAOImplementation(self.dao_item_cls)

    def get_keys(self, uid):
        return self._pg_impl.get_keys(uid)

    def remove_key(self, _id, uid, version, key):
        return self._pg_impl.remove_key(_id, uid, version, key)

    def set_key(self, _id, uid, path, version, key, value, type='file'):
        return self._pg_impl.set_key(_id, uid, path, version, key, value, type)


class PostgresSettingsDAOImplementation(PostgresBaseDAOImplementation):
    def get_keys(self, uid):
        uid = self.dao_item_cls.get_field_pg_representation('uid', uid)
        session = Session.create_from_uid(uid)
        cursor = session.execute(SQL_SELECT_SETTING_BY_UID, {'uid': uid})
        return [self.doc_to_item(i) for i in cursor]

    def remove_key(self, _id, uid, version, key):
        uid = self.dao_item_cls.get_field_pg_representation('uid', uid)
        _id = self.dao_item_cls.get_field_pg_representation('id', _id)
        session = Session.create_from_uid(uid)
        with session.begin():
            setting_cursor = session.execute(
                SQL_SELECT_SETTING_BY_ID_FOR_UPDATE, {
                    'id': _id,
                    'uid': uid,
                })
            setting_doc = setting_cursor.fetchone()
            if not setting_doc:
                raise errors.SettingStateNotFound()
            setting_item = self.doc_to_item(setting_doc)
            data = dict(setting_item.data)
            try:
                del data[key]
            except KeyError:
                raise errors.SettingStateNotFound()

            # исторически в поле data хранится массив пар
            data = [[k, v] for k, v in data.items()]

            cursor = session.execute(SQL_UPDATE_SETTING, {
                'id': _id,
                'uid': uid,
                'version': self.dao_item_cls.get_field_pg_representation('version', version),
                'data': self.dao_item_cls.get_field_pg_representation('data', data),
            })
            doc = cursor.fetchone()
            return self.doc_to_item(doc)

    def set_key(self, _id, uid, path, version, key, value, type='file'):
        uid = self.dao_item_cls.get_field_pg_representation('uid', uid)
        pg_id = self.dao_item_cls.get_field_pg_representation('id', _id)
        session = Session.create_from_uid(uid)
        for retry in range(2):
            # 'select for update' и update должны быть в одной транзакции
            # если 'select for update' ничего не вернул, делаем insert,
            # ловим `DatabaseConstraintError` и если поймали
            # делаем update в новой транзакции с блокированием записи
            with session.begin():
                setting_cursor = session.execute(
                    SQL_SELECT_SETTING_BY_ID_FOR_UPDATE, {
                        'id': pg_id,
                        'uid': uid,
                    })
                setting_doc = setting_cursor.fetchone()
                if not setting_doc:
                    try:
                        item = self._insert_key(_id, uid, path, version, key, value, type)
                        return item
                    except DatabaseConstraintError:
                        continue
                setting_item = self.doc_to_item(setting_doc)
                data = dict(setting_item.data)
                data[key] = value

                # исторически в поле data хранится массив пар
                data = [[k, v] for k, v in data.items()]

                cursor = session.execute(SQL_UPDATE_SETTING, {
                    'id': pg_id,
                    'uid': uid,
                    'version': self.dao_item_cls.get_field_pg_representation('version', version),
                    'data': self.dao_item_cls.get_field_pg_representation('data', data),
                })
                doc = cursor.fetchone()
                return self.doc_to_item(doc)

    def _insert_key(self, _id, uid, path, version, key, value, type='file'):
        data = [[key, value]]
        session = Session.create_from_uid(uid)
        cursor = session.execute(SQL_INSERT_SETTING, {
            'id': self.dao_item_cls.get_field_pg_representation('id', _id),
            'uid': self.dao_item_cls.get_field_pg_representation('uid', uid),
            'path': self.dao_item_cls.get_field_pg_representation('path', path),
            'version': self.dao_item_cls.get_field_pg_representation('version', version),
            'type': self.dao_item_cls.get_field_pg_representation('type', type),
            'data': self.dao_item_cls.get_field_pg_representation('data', data),
        })
        doc = cursor.fetchone()
        return self.doc_to_item(doc)
