# -*- coding: utf-8 -*-
import time
import datetime

from pymongo.errors import DuplicateKeyError

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util import ctimestamp
from mpfs.config import settings
from mpfs.core.albums.static import FacesIndexingState
from mpfs.dao.base import BaseDAO, BaseDAOItem, MongoBaseDAOImplementation, PostgresBaseDAOImplementation
from mpfs.dao.fields import (
    IntegerAsBoolField, DateTimeField, IntegerField, StringField,
    BoolField, UidField, StringArrayField, MSKDateTimeField, JsonField, UuidField
)
from mpfs.dao.session import Session
from mpfs.metastorage.mongo.util import shard_key
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter
from mpfs.metastorage.postgres.schema import user_index
from mpfs.metastorage.postgres.queries import (
    SQL_CHANGE_HANCOM_STATUS,
    SQL_CHANGE_UNLIMITED_AUTOUPLOADING_STATUS,
    SQL_SEARCH_USERS_BY_B2B_KEY,
    SQL_SEARCH_USERS_BY_YATEAM_UID,
    SQL_SEARCH_YATEAM_USERS,
    SQL_UPSERT_USER_INDEX,
    SQL_UPSERT_USER_INDEX_WITH_LOCK,
    SQL_RELEASE_USER_INIT_LOCK,
    SQL_REMOVE_OWN_USER_INIT_LOCK_TIMESTAMP,
    SQL_INCREMENT_VERSION_AND_SET_FORCE_SNAPSHOT_VERSION,
    SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_ALLOWED,
    SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_STATUS,
    SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_ALLOWED,
    SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_STATUS,
    SQL_GET_UNLIMITED_AUTOUPLOADING_STATUS,
    SQL_SET_IS_PAID,
    SQL_SET_ONLINE_EDITOR,
    SQL_CHANGE_ONLY_OFFICE_STATUS,
    SQL_CHANGE_ONLINE_EDITOR_STATUS,
    SQL_CHANGE_ADVERTISING_STATUS,
    SQL_CHANGE_PUBLIC_SETTINGS_STATUS,
    SQL_SET_OFFICE_SELECTION_STRATEGY, SQL_GET_FACES_INDEXING_STATE)

dbctl = mpfs.engine.process.dbctl()

YATEAM_ENABLE_FETCH_YATEAM_USERS_FROM_POSTGRES = settings.yateam['enable_fetch_yateam_users_from_postgres']
SYSTEM_USER_INIT_LOCK_TTL = settings.system['user_init_lock_ttl']


class UserDAOItem(BaseDAOItem):
    mongo_collection_name = 'user_index'
    postgres_table_obj = user_index
    uid_field_name = '_id'
    is_sharded = True

    uid = UidField(mongo_path='_id', pg_path=user_index.c.uid)

    version = IntegerField(mongo_path='version', pg_path=user_index.c.version, default_value=None)
    blocked = IntegerAsBoolField(mongo_path='blocked', pg_path=user_index.c.blocked,
                                 default_value=False)
    deleted = MSKDateTimeField(mongo_path='deleted', pg_path=user_index.c.deleted, default_value=None)

    # TODO Why default is None?
    user_type = StringField(mongo_path='type', pg_path=user_index.c.user_type, default_value=None)

    reg_time = DateTimeField(mongo_path='reg_time', pg_path=user_index.c.reg_time)
    # TODO Why default is None?
    # TODO Why not locale_type?
    locale = StringField(mongo_path='locale', pg_path=user_index.c.locale, default_value=None)

    shard_key = IntegerField(mongo_path='shard_key', pg_path=user_index.c.shard_key)
    b2b_key = StringField(mongo_path='b2b_key', pg_path=user_index.c.b2b_key, default_value=None)
    pdd = JsonField(mongo_path='pdd', pg_path=user_index.c.pdd, default_value=None)
    is_mailish = BoolField(mongo_path='is_mailish', pg_path=user_index.c.is_mailish, default_value=False)
    hancom_enabled = BoolField(mongo_path='hancom_enabled', pg_path=user_index.c.hancom_enabled, default_value=False)
    unlimited_autouploading_enabled = BoolField(mongo_path='unlimited_autouploading_enabled',
                                                pg_path=user_index.c.unlimited_autouploading_enabled,
                                                default_value=False)
    unlimited_photo_autouploading_enabled = BoolField(mongo_path='unlimited_photo_autouploading_enabled',
                                                pg_path=user_index.c.unlimited_photo_autouploading_enabled,
                                                default_value=False)
    unlimited_video_autouploading_enabled = BoolField(mongo_path='unlimited_video_autouploading_enabled',
                                                pg_path=user_index.c.unlimited_video_autouploading_enabled,
                                                default_value=False)
    unlimited_video_autouploading_reason = StringField(mongo_path='unlimited_video_autouploading_reason',
                                                       pg_path=user_index.c.unlimited_video_autouploading_reason,
                                                       default_value=None)
    unlimited_photo_autouploading_allowed = BoolField(mongo_path='unlimited_photo_autouploading_allowed',
                                                      pg_path=user_index.c.unlimited_photo_autouploading_allowed,
                                                      default_value=False)
    unlimited_video_autouploading_allowed = BoolField(mongo_path='unlimited_video_autouploading_allowed',
                                                      pg_path=user_index.c.unlimited_video_autouploading_allowed,
                                                      default_value=False)
    office_selection_strategy = StringField(mongo_path='office_selection_strategy',
                                            pg_path=user_index.c.office_selection_strategy,
                                            default_value=None)
    lock_key = UuidField(mongo_path='lock_key', pg_path=user_index.c.lock_key, default_value=None)
    lock_timestamp = MSKDateTimeField(mongo_path='lock_timestamp', pg_path=user_index.c.lock_timestamp, default_value=None)

    yateam_uid = UidField(mongo_path='yateam_uid', pg_path=user_index.c.yateam_uid, default_value=None)
    last_quick_move_version = IntegerField(mongo_path='last_quick_move_version',
                                           pg_path=user_index.c.last_quick_move_version, default_value=None)
    is_reindexed_for_quick_move = BoolField(mongo_path='is_reindexed_for_quick_move',
                                            pg_path=user_index.c.is_reindexed_for_quick_move, default_value=None)
    force_snapshot_version = IntegerField(mongo_path='force_snapshot_version',
                                          pg_path=user_index.c.force_snapshot_version, default_value=None)

    collections = StringArrayField(mongo_path='collections', pg_path=user_index.c.collections)
    is_paid = BoolField(mongo_path='is_paid', pg_path=user_index.c.is_paid, default_value=False)

    office_online_editor_type = StringField(mongo_path='office_online_editor_type', pg_path=user_index.c.office_online_editor_type, default_value=None)
    only_office_enabled = BoolField(mongo_path='only_office_enabled', pg_path=user_index.c.only_office_enabled, default_value=None)
    online_editor_enabled = BoolField(mongo_path='online_editor_enabled', pg_path=user_index.c.online_editor_enabled, default_value=None)

    faces_indexing_state = StringField(mongo_path='faces_indexing_state', pg_path=user_index.c.faces_indexing_state, default_value=None)
    faces_indexing_state_time = DateTimeField(mongo_path='faces_indexing_state_time', pg_path=user_index.c.faces_indexing_state_time, default_value=None)

    advertising_enabled = BoolField(mongo_path='advertising_enabled', pg_path=user_index.c.advertising_enabled, default_value=None)
    public_settings_enabled = BoolField(mongo_path='public_settings_enabled', pg_path=user_index.c.public_settings_enabled, default_value=None)

    default_product_id = StringField(mongo_path='default_product_id',
                                     pg_path=user_index.c.default_product_id,
                                     default_value=None)

    exclude_keys_after_conversion_to_mongo = {
        'b2b_key': None,
        'pdd': None,
        'yateam_uid': None,
        'deleted': None,
        'blocked': False,
        'hancom_enabled': False,
        'unlimited_autouploading_enabled': False,
        'lock_key': None,
        'lock_timestamp': None,
        'last_quick_move_version': None,
        'is_reindexed_for_quick_move': None,
    }


class UserDAO(BaseDAO):
    dao_item_cls = UserDAOItem

    def __init__(self):
        super(UserDAO, self).__init__()
        self._mongo_impl = MongoUserDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresUserDAOImplementation(self.dao_item_cls)
        self._usrctl = mpfs.engine.process.usrctl()

    def fetch_yateam_users(self):
        for user in self._mongo_impl.fetch_yateam_users():
            if not self._usrctl.is_user_in_postgres(user.uid):
                yield user
        for user in self._pg_impl.fetch_yateam_users():
            if self._usrctl.is_user_in_postgres(user.uid):
                yield user

    def get_users_by_yateam_uid(self, yateam_uid):
        """Выполнить поиск по всем шардам и СУБД пользователей, у которых заданный yateam_uid.

        ..notes:: Возвращаем пользователя только если он живет в той СУБД где мы его нашли.

        ..warning:: Нет гарантий, что на разных шардах нет двух разных уидов
            с одинаковым yateam_uid. Нет гарантий, что на двух разных СУБД нет уидов
            с одинаковым yateam_uid.
        """
        for user in self._mongo_impl.get_users_by_yateam_uid(yateam_uid):
            if not self._usrctl.is_user_in_postgres(user.uid):
                yield user
        for user in self._pg_impl.get_users_by_yateam_uid(yateam_uid):
            if self._usrctl.is_user_in_postgres(user.uid):
                yield user

    def get_users_by_b2b_key(self, b2b_key):
        """Выполнить поиск по всем шардам и СУБД пользователей, у которых заданный b2b_key.

        ..notes:: Возвращаем пользователя только если он живет в той СУБД где мы его нашли.
        """
        for user in self._mongo_impl.get_users_by_b2b_key(b2b_key):
            if not self._usrctl.is_user_in_postgres(user.uid):
                yield user
        for user in self._pg_impl.get_users_by_b2b_key(b2b_key):
            if self._usrctl.is_user_in_postgres(user.uid):
                yield user

    def enable_hancom(self, uid):
        """Проставляет флаг hancom_enabled"""
        return self._get_impl(uid).enable_hancom(uid)

    def disable_hancom(self, uid):
        """Убирает флаг hancom_enabled"""
        return self._get_impl(uid).disable_hancom(uid)

    def enable_unlimited_autouploading(self, uid):
        """Проставляет флаг unlimited_autouploading_enabled"""
        return self._get_impl(uid).enable_unlimited_autouploading(uid)

    def disable_unlimited_autouploading(self, uid):
        """Убирает флаг unlimited_autouploading_enabled"""
        return self._get_impl(uid).disable_unlimited_autouploading(uid)

    def set_unlimited_autouploading(self, uid, video_status=None, video_reason='by_user', photo_status=None):
        """Выставляет флаги  unlimited_autouploading__enabled"""
        return self._get_impl(uid).set_unlimited_autouploading(uid, video_status, video_reason, photo_status)

    def set_unlimited_video_autouploading_allowed(self, uid, video_allowed):
        """Выставляет флаг unlimited_video_autouploading_allowed"""
        return self._get_impl(uid).set_unlimited_video_autouploading_allowed(uid, video_allowed)

    def set_unlimited_photo_autouploading_allowed(self, uid, photo_allowed):
        """Выставляет флаг unlimited_photo_autouploading_allowed"""
        return self._get_impl(uid).set_unlimited_photo_autouploading_allowed(uid, photo_allowed)

    def set_only_office_enabled(self, uid, value):
        return self._get_impl(uid).set_only_office_enabled(uid, value)

    def set_online_editor_enabled(self, uid, value):
        return self._get_impl(uid).set_online_editor_enabled(uid, value)

    def set_advertising_enabled(self, uid, value):
        return self._get_impl(uid).set_advertising_enabled(uid, value)

    def set_public_settings_enabled(self, uid, value):
        return self._get_impl(uid).set_public_settings_enabled(uid, value)

    def get_unlimited_autouploading(self, uid):
        """Выставляет флаги  unlimited_autouploading__enabled"""
        return self._get_impl(uid).get_unlimited_autouploading(uid)

    def get_faces_indexing_state(self, uid):
        return self._get_impl(uid).get_faces_indexing_state(uid)

    def create_user(self, uid, type, locale, b2b_key, pdd, is_mailish, hancom_enabled, lock_key=None,
                    enable_quick_move=False, is_paid=False, faces_indexing_state=None,
                    office_selection_strategy=None, default_product_id=None):

        set_on_insert = {
            '_id': uid,
            'collections': [],
            'shard_key': shard_key(uid),
            'reg_time': int(time.time()),
            'faces_indexing_state': faces_indexing_state,
            'faces_indexing_state_time': ctimestamp() if faces_indexing_state == FacesIndexingState.REINDEXED else None
        }

        if enable_quick_move and mpfs.engine.process.usrctl().is_user_in_postgres(uid):
            set_on_insert['is_reindexed_for_quick_move'] = True

        set_always = {}

        set_always['is_paid'] = is_paid
        if locale is not None:
            set_always['locale'] = locale
        if type is not None:
            set_always['type'] = type
        if b2b_key is not None:
            set_always['b2b_key'] = b2b_key
        if isinstance(pdd, dict):
            set_always['pdd'] = pdd
        if isinstance(is_mailish, bool):
            set_always['is_mailish'] = is_mailish
        if isinstance(hancom_enabled, bool):
            set_always['hancom_enabled'] = hancom_enabled
        if isinstance(office_selection_strategy, basestring):
            set_always['office_selection_strategy'] = office_selection_strategy
        if isinstance(default_product_id, basestring):
            set_always['default_product_id'] = default_product_id

        if lock_key:
            lock_timestamp = datetime.datetime.now()
            ttl_threshold = lock_timestamp - datetime.timedelta(seconds=SYSTEM_USER_INIT_LOCK_TTL)
            set_always['lock_key'] = lock_key
            set_always['lock_timestamp'] = lock_timestamp

            self._get_impl(uid).create_user_with_lock(uid, set_on_insert, set_always, ttl_threshold)
        else:
            self._get_impl(uid).create_user(uid, set_on_insert, set_always)

        # ресетим кэш вручную, т.к. пишем в базу не через user_index_mpfs_collection
        from mpfs.metastorage.mongo.collections.base import user_index
        user_index.reset()

    def release_lock(self, uid):
        self._get_impl(uid).release_lock(uid)

    def remove_own_lock_timestamp(self, uid, lock_key):
        self._get_impl(uid).remove_own_lock_timestamp(uid, lock_key)

    def force_user_to_get_snapshot(self, uid):
        self._get_impl(uid).force_user_to_get_snapshot(uid)

    def set_is_paid(self, uid, is_paid):
        self._get_impl(uid).set_is_paid(uid, is_paid)

    def set_online_editor(self, uid, online_editor):
        self._get_impl(uid).set_online_editor(uid, online_editor)

    def set_office_selection_strategy(self, uid, office_selection_strategy):
        self._get_impl(uid).set_office_selection_strategy(uid, office_selection_strategy)


class MongoUserDAOImplementation(MongoBaseDAOImplementation):
    def __init__(self, dao_item_cls):
        super(MongoUserDAOImplementation, self).__init__(dao_item_cls)
        self._usrctl = mpfs.engine.process.usrctl()

    def fetch_yateam_users(self):
        query = {'yateam_uid': {'$exists': True}}
        for shard in dbctl.mapper.rspool.get_all_shards_names():
            res = self.find_on_shard(query, shard_name=shard)
            for mongo_data in res:
                user = UserDAOItem.create_from_mongo_dict(mongo_data)
                info = self._usrctl.info(user.uid)
                if info['shard'] == shard:
                    yield user

    def get_users_by_yateam_uid(self, yateam_uid):
        query = {'yateam_uid': yateam_uid}
        for shard in dbctl.mapper.rspool.get_all_shards_names():
            res = self.find_on_shard(query, shard_name=shard)
            for mongo_data in res:
                user = UserDAOItem.create_from_mongo_dict(mongo_data)
                info = self._usrctl.info(user.uid)
                if info['shard'] == shard:
                    yield user

    def get_users_by_b2b_key(self, b2b_key):
        query = {'b2b_key': b2b_key}
        for shard in dbctl.mapper.rspool.get_all_shards_names():
            res = self.find_on_shard(query, shard_name=shard)
            for mongo_data in res:
                user = UserDAOItem.create_from_mongo_dict(mongo_data)
                info = self._usrctl.info(user.uid)
                if info['shard'] == shard:
                    yield user

    def enable_hancom(self, uid):
        spec = {'_id': uid}
        doc = {'$set': {'hancom_enabled': True}}
        self.update(spec, doc)

    def disable_hancom(self, uid):
        spec = {'_id': uid}
        doc = {'$unset': {'hancom_enabled': ''}}
        self.update(spec, doc)

    def enable_unlimited_autouploading(self, uid):
        spec = {'_id': uid}
        doc = {'$set': {'unlimited_autouploading_enabled': True}}
        self.update(spec, doc)

    def disable_unlimited_autouploading(self, uid):
        spec = {'_id': uid}
        doc = {'$unset': {'unlimited_autouploading_enabled': ''}}
        self.update(spec, doc)

    def get_unlimited_autouploading(self, *args, **kwargs):
        # заглушка нужна пока не оторвём походы в монгу полностью
        return {
            'unlimited_video_autouploading_enabled': None,
            'unlimited_video_autouploading_reason': 'by_user',
            'unlimited_photo_autouploading_enabled': None}

    def create_user(self, uid, set_on_insert, set_always):
        fsync_safe_w = dbctl.fsync_safe_w('sharded')
        update = {'$setOnInsert': set_on_insert}
        # Добавляем $set только если set_always не пуст
        if set_always:
            update['$set'] = set_always
        self.update({'_id': uid}, update, upsert=True, **fsync_safe_w)

    def create_user_with_lock(self, uid, set_on_insert, set_always, ttl_threshold):
        # нельзя опираться на то, что при upsert-е зарейзится DuplicateKeyError
        # т.к. в мпфс обертке монгоколлекции при этой ошибке запросы ретраятся без параметра upsert
        # поэтому insert и update по отдельности
        # подробнее https://st.yandex-team.ru/CHEMODAN-33935
        fsync_safe_w = dbctl.fsync_safe_w('sharded')
        try:
            self.insert(dict(set_on_insert, **set_always), **fsync_safe_w)
        except DuplicateKeyError:
            query = {
                '_id': uid,
                '$or': [
                    {'lock_key': {'$exists': False}},
                    {'lock_key': {'$exists': True}, '$or': [
                        {'lock_timestamp': {'$lt': ttl_threshold}},
                        {'lock_timestamp': {'$exists': False}}
                    ]}
                ]
            }
            update_result = self.update(query, {'$set': set_always}, **fsync_safe_w)
            if not update_result['updatedExisting']:
                raise errors.StorageInitUserImpossibleToLock()

    def release_lock(self, uid):
        fsync_safe_w = dbctl.fsync_safe_w('sharded')
        self.update({'_id': uid}, {'$unset': {'lock_key': '', 'lock_timestamp': ''}}, **fsync_safe_w)

    def remove_own_lock_timestamp(self, uid, lock_key):
        fsync_safe_w = dbctl.fsync_safe_w('sharded')
        self.update({'_id': uid, 'lock_key': lock_key}, {'$unset': {'lock_timestamp': ''}}, **fsync_safe_w)

    def force_user_to_get_snapshot(self, uid):
        pass

    def set_is_paid(self, uid, is_paid):
        pass


class PostgresUserDAOImplementation(PostgresBaseDAOImplementation):
    def fetch_yateam_users(self):
        pg_query_executer = PGQueryExecuter()
        for shard_id in pg_query_executer.get_all_shard_ids():
            session = Session.create_from_shard_id(shard_id)
            cursor = session.execute(SQL_SEARCH_YATEAM_USERS)
            for pg_data in cursor:
                user = UserDAOItem.create_from_pg_data(pg_data)
                if not pg_query_executer.is_user_in_postgres(user.uid):
                    # Исключаем пользователей, которые находятся в процессе миграции
                    continue
                if pg_query_executer.get_shard_id(user.uid) == shard_id:
                    yield user

    def get_users_by_yateam_uid(self, yateam_uid):
        for shard_id in PGQueryExecuter().get_all_shard_ids():
            session = Session.create_from_shard_id(shard_id)
            cursor = session.execute(SQL_SEARCH_USERS_BY_YATEAM_UID, {'yateam_uid': yateam_uid})
            for pg_data in cursor:
                user = UserDAOItem.create_from_pg_data(pg_data)
                if PGQueryExecuter().get_shard_id(user.uid) == shard_id:
                    yield user

    def get_users_by_b2b_key(self, b2b_key):
        for shard_id in PGQueryExecuter().get_all_shard_ids():
            session = Session.create_from_shard_id(shard_id)
            cursor = session.execute(SQL_SEARCH_USERS_BY_B2B_KEY, {'b2b_key': b2b_key})
            for pg_data in cursor:
                user = UserDAOItem.create_from_pg_data(pg_data)
                if PGQueryExecuter().get_shard_id(user.uid) == shard_id:
                    yield user

    def enable_hancom(self, uid):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_HANCOM_STATUS, {'uid': uid, 'hancom_status': True})

    def disable_hancom(self, uid):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_HANCOM_STATUS, {'uid': uid, 'hancom_status': False})

    def enable_unlimited_autouploading(self, uid):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_UNLIMITED_AUTOUPLOADING_STATUS,
                                  {'uid': uid, 'unlimited_autouploading_status': True})

    def disable_unlimited_autouploading(self, uid):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_UNLIMITED_AUTOUPLOADING_STATUS,
                                  {'uid': uid, 'unlimited_autouploading_status': False})

    def set_unlimited_autouploading(self, uid, video_status=None, video_reason='by_user', photo_status=None, ):
        session = Session.create_from_uid(uid)
        if video_status is not None:
            session.execute_and_close(SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_STATUS,
                                      {'uid': uid, 'unlimited_video_autoupload_status': video_status,
                                       'unlimited_video_autoupload_reason_text': video_reason})
        if photo_status is not None:
            session.execute_and_close(SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_STATUS,
                                      {'uid': uid, 'unlimited_photo_autoupload_status': photo_status})
        return self.get_unlimited_autouploading(uid)

    def set_unlimited_video_autouploading_allowed(self, uid, video_allowed):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_ALLOWED,
                                  {'uid': uid, 'unlimited_video_autoupload_allowed': video_allowed})

    def set_unlimited_photo_autouploading_allowed(self, uid, photo_allowed):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_ALLOWED,
                                  {'uid': uid, 'unlimited_photo_autoupload_allowed': photo_allowed})

    def set_only_office_enabled(self, uid, value):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_ONLY_OFFICE_STATUS,
                                  {'uid': uid, 'only_office_enabled': value})

    def set_online_editor_enabled(self, uid, value):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_ONLINE_EDITOR_STATUS,
                                  {'uid': uid, 'online_editor_enabled': value})

    def set_advertising_enabled(self, uid, value):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_ADVERTISING_STATUS,
                                  {'uid': uid, 'advertising_enabled': value})

    def set_public_settings_enabled(self, uid, value):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_CHANGE_PUBLIC_SETTINGS_STATUS,
                                  {'uid': uid, 'public_settings_enabled': value})

    def get_unlimited_autouploading(self, uid):
        session = Session.create_from_uid(uid)
        return dict(session.execute(SQL_GET_UNLIMITED_AUTOUPLOADING_STATUS, {'uid': uid}).fetchone())

    def get_faces_indexing_state(self, uid):
        session = Session.create_from_uid(uid)
        return dict(session.execute(SQL_GET_FACES_INDEXING_STATE, {'uid': uid}).fetchone())

    def create_user(self, uid, set_on_insert, set_always):
        full_data = dict(set_on_insert, **set_always)

        pg_representation = self.dao_item_cls.create_from_mongo_dict(full_data).get_postgres_representation()
        query_params = {c.name: v for c, v in pg_representation.iteritems()}

        fields_for_update = [UserDAOItem.convert_mongo_key_to_postgres(k) for k in set_always.iterkeys()]
        if not fields_for_update:
            fields_for_update = ['uid']
        fields_for_update_placeholder = ','.join('%s=:%s' % (f, f) for f in fields_for_update)

        session = Session.create_from_uid(uid)
        session.execute(SQL_UPSERT_USER_INDEX % fields_for_update_placeholder, query_params)

    def create_user_with_lock(self, uid, set_on_insert, set_always, ttl_threshold):
        full_data = dict(set_on_insert, **set_always)

        pg_representation = self.dao_item_cls.create_from_mongo_dict(full_data).get_postgres_representation()
        query_params = {c.name: v for c, v in pg_representation.iteritems()}
        # преобразуем ttl_threshold к тому же формату, что и lock_timestamp
        _, query_params['ttl_threshold'] = UserDAOItem.convert_mongo_value_to_postgres_for_key('lock_timestamp', ttl_threshold)

        fields_for_update = [UserDAOItem.convert_mongo_key_to_postgres(k) for k in set_always.iterkeys()]
        if not fields_for_update:
            fields_for_update = ['uid']
        fields_for_update_placeholder = ','.join('%s=:%s' % (f, f) for f in fields_for_update)

        session = Session.create_from_uid(uid)
        if not session.execute(SQL_UPSERT_USER_INDEX_WITH_LOCK % fields_for_update_placeholder, query_params).fetchone():
            raise errors.StorageInitUserImpossibleToLock()

    def release_lock(self, uid):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_RELEASE_USER_INIT_LOCK, {'uid': uid})

    def remove_own_lock_timestamp(self, uid, lock_key):
        session = Session.create_from_uid(uid)
        session.execute_and_close(SQL_REMOVE_OWN_USER_INIT_LOCK_TIMESTAMP, {'uid': uid, 'lock_key': lock_key})

    def force_user_to_get_snapshot(self, uid):
        session = Session.create_from_uid(uid)
        session.execute(SQL_INCREMENT_VERSION_AND_SET_FORCE_SNAPSHOT_VERSION, {'uid': self.get_field_repr('uid', uid)})

    def set_is_paid(self, uid, is_paid):
        session = Session.create_from_uid(uid)
        session.execute(SQL_SET_IS_PAID, {'uid': self.get_field_repr('uid', uid), 'is_paid': is_paid})

    def set_online_editor(self, uid, online_editor):
        session = Session.create_from_uid(uid)
        session.execute(SQL_SET_ONLINE_EDITOR, {'uid': self.get_field_repr('uid', uid), 'online_editor': online_editor})

    def set_office_selection_strategy(self, uid, office_selection_strategy):
        session = Session.create_from_uid(uid)
        session.execute(SQL_SET_OFFICE_SELECTION_STRATEGY,
                        {'uid': self.get_field_repr('uid', uid),
                         'office_selection_strategy': office_selection_strategy})
