# -*- coding: utf-8 -*-
"""

MPFS
CORE

Работа с инвайтами пользователя

"""
import time
import mpfs.engine.process

from mpfs.invite import manager
from mpfs.config import settings
from mpfs.invite import errors
from mpfs.common.errors import billing as billing_errors
from mpfs.common.util import mailer
from mpfs.common.static.tags.push import INVITE
from mpfs.core.user.settings import States
from mpfs.core.user.constants import *
from mpfs.core.user.base import User
from mpfs.core.metastorage.control import invite_mpfs_referrals as referrals_table
from mpfs.core.metastorage.control import invite_mpfs_sent
from mpfs.core.services.passport_service import Passport
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.billing import api as billing_api

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport = Passport()
push_queue = PushQueue()


def report(uid, hsh):
    '''
    Сводка по инвайтосостоянию юзера
    '''
    if hsh is not None:
        code = manager.get_code('mpfs', hsh)
        uid = code.meta.get('referral')
        if not uid:
            raise errors.InviteNotFound(hsh)
    
    states = States(uid)
    invite_data = states.list('invites')

    uinfo = passport.userinfo(uid)
    default = {
        'name': uinfo.get('public_name'),
        'login': uinfo.get('login'),
        'activated': 0,
        'bonus_used': 0,
        'bonus_unused': INVITES_REFERRAL_POOL
    }
    
    for key, value in default.iteritems():
        if key not in invite_data:
            invite_data[key] = value
    
    if 'referral' in invite_data:
        referral = invite_data.get('referral')
        referral_info = passport.userinfo(referral) 
        invite_data['referral'] = {
            'uid'       : referral,
            'username'  : referral_info['public_name'],
            'login'     : referral_info['login'],
            'email'     : referral_info['email']
        }
        
    return invite_data

def personal_hash(uid):
    '''
    Сгенерировать личный инвайт
    Если уже есть - отдать 
    '''
    states = States(uid)
    invite_data = states.list('invites')
    
    try:
        code = manager.get_code('mpfs', invite_data.get('personal_hash'))
    except Exception:
        code = manager.generate_eternal_code('mpfs', referral=uid)
        states.set('personal_hash', code.hash, namespace='invites')
    
    return code
   

def list_activated(uid):
    '''
    Выдать список активированных юзеров (имена)
    '''
    result = []
    for item in referrals_table.folder_content(uid):
        friend_uid = item['_id']
        friend_state = item['state']
        uinfo = passport.userinfo(friend_uid)
        result.append({
            'name'  : uinfo.get('public_name'),
            'login' : uinfo.get('login'),
            'state' : friend_state,
            'userid': uinfo.get('email'),
            'locale': uinfo.get('language'),
        })    
    return result


def list_sent(uid):
    '''
    Выдать список отправленных инвайтов
    '''
    result = []
    for item in invite_mpfs_sent.get_all(uid=uid):
        result.append({
            'type': item.get('p'),
            'userid': item.get('u'),
        })    
    return result
        
    
def process_friend_activation(uid, referral):
    '''
    Обработка активации друга
    '''
    if uid == referral:
        return
    
    # Апдейтим таблицу рефералов
    if not referrals_table.check(uid):
        referrals_table.create(uid)
    referrals_table.put(
        uid,
        {'ref': referral, 'state': 'wait', 'mtime': int(time.time())}
    )
    
    # Меняем флажки у реферала
    ref_states = States(referral)
    activated_amt = ref_states.list('invites').get('activated', 0) + 1
    ref_states.set('activated', activated_amt, namespace='invites')
    
    # Меняем флажки у нового пользователя
    user_states = States(uid)
    user_states.bulk_set({'referral': referral, 'state': 'wait'}, namespace='invites')

     # Пушим рефераллу нотифай
    user_info = passport.userinfo(uid)
    _push_notify_to_referral(
        uid           = referral,
        user_info     = user_info,
        activated_amt = activated_amt,
        used_bonus    = ref_states.list('invites').get('bonus_used', 0),
        state         = 'wait'
    )


def process_device_installation(uid, device_type, device_id):
    # Смотрим, пришел ли пользователь по инвайту
    try:
        referral_record = referrals_table.show(uid)
        referral_uid = referral_record.get('ref')
    except AttributeError:
        return
    
    # Если устройство не десктоп - ничего не делаем
    if device_type != 'desktop':
        return
    
    # Если уже обработан - ничего не делаем
    if report(uid, None).get('state') == 'ok':
        return

    user_obj  = User(uid)
    user_info = passport.userinfo(uid)

    referral_obj = User(referral_uid)
    referral_info = passport.userinfo(referral_uid)

    ref_vars  = referral_obj.states.list('invites')
    used_invites_bonus = ref_vars.get('bonus_used', 0)
    
    def mark_suspicious():
        referrals_table.update_where(
            uid,
            {'mtime': int(time.time()), 'state': 'suspicious'},
            referral_record,
        )
        user_obj.states.set('state', 'suspicious', namespace='invites')
        
        # Пушим рефераллу нотифай
        _push_notify_to_referral(
            uid           = referral_uid,
            user_info     = user_info,
            activated_amt = ref_vars.get('activated', 0),
            used_bonus    = used_invites_bonus,
            state         = 'suspicious'
        )
        return
    
    # Если у реферала тот же ID - записываем, что мы подозрительно
    if referral_obj.devices.match(device_id, type=device_type):
        mark_suspicious()
        return
    
    # Если у друзей реферала есть тот же ID - записываем, что подозрительно
    # https://jira.yandex-team.ru/browse/CHEMODAN-8225
    for item in referrals_table.folder_content(referral_uid):
        another_friend_uid = item['_id']
        if another_friend_uid == uid:
            continue
        
        another_user_obj = User(another_friend_uid)
        if another_user_obj.devices.match(device_id, type=device_type):
            mark_suspicious()
            return
        
    # Увеличиваем место себе
    if settings.billing['enabled']:
        try:
            billing_api.service_create(uid, 'invited', line='bonus')
        except billing_errors.BillingProductIsSingletone:
            pass
    else:
        user_obj.enlarge_space(INVITES_USER_BONUS, reason='invited')
    user_obj.states.set('state', 'ok', namespace='invites')
    
    # Высылаем письмо себе
    if user_info['email']:
        params = {
            'friendName' : referral_info['public_name'],
            'username': user_info['username'],
            'login'   : user_info['login'],
            'locale'  : user_info['language'],
            'space'   : user_obj.info().get('space'),
        }
        mailer.send(user_info['email'], 'referral/award', params)
        
    # Проверяем, можно ли увеличить место тому, кто нас привел
    if INVITES_REFERRAL_BONUS > (INVITES_REFERRAL_POOL - used_invites_bonus):
        # Апдейтим таблицу рефералов, ставим, что нас реферал не получил место
        referrals_table.update_where(
            uid,
            {'mtime': int(time.time()), 'state': 'nospace'},
            referral_record,
        )
        
        # Пушим рефераллу нотифай
        _push_notify_to_referral(
            uid           = referral_uid,
            user_info     = user_info,
            activated_amt = ref_vars.get('activated', 0),
            used_bonus    = used_invites_bonus,
            state         = 'nospace'
        )
            
        return
        
    # Увеличиваем место тому, кто нас привел
    if settings.billing['enabled']:
        billing_api.service_create(referral_uid, 'referral', line='bonus')    
    else:
        referral_obj.enlarge_space(INVITES_REFERRAL_BONUS, reason='referral')
    used_invites_bonus = used_invites_bonus + INVITES_REFERRAL_BONUS
    referral_obj.states.bulk_set({
            'bonus_used'  : used_invites_bonus,
            'bonus_unused': INVITES_REFERRAL_POOL - used_invites_bonus,
        }, namespace='invites')
    
    # Апдейтим таблицу рефералов, указываем, что за нас реферал получил место
    referrals_table.update_where(
        uid,
        {'mtime': int(time.time()), 'state': 'ok'},
        referral_record,
    )
        
    # Рассылаем письмо рефералу
    params = {
        'friendName': user_info['public_name'],
        'username'  : referral_info['username'],
        'login'     : referral_info['login'],
        'locale'    : referral_info['language'],
    }
    if referral_info['email']:
        mailer.send(referral_info['email'], 'referral/accepted', params)    

    # Пушим рефераллу нотифай
    _push_notify_to_referral(
        uid           = referral_uid,
        user_info     = user_info,
        activated_amt = ref_vars.get('activated', 0),
        used_bonus    = used_invites_bonus,
        state         = 'ok'
    )
    
    
def _push_notify_to_referral(**kwargs):
    '''
    Пушим рефералу в ксиву
    '''
    data = {
        'class': INVITE,
        'uid': kwargs['uid'],
        'xiva_data': {
            "root": {
                "tag" : "invite",
                "parameters": {
                    'activated'   : kwargs['activated_amt'],
                    "name"        : kwargs['user_info']['public_name'],
                    'userid'      : kwargs['user_info']['email'],
                    'bonus_used'  : kwargs['used_bonus'],
                    'bonus_unused': INVITES_REFERRAL_POOL - kwargs['used_bonus'],
                    'state'       : kwargs['state'],
                }
            }
        }
    }
    
    push_queue.put(data)
        
def invite_friend(uid, provider, userid, params):
    '''
    Пригласить друга, выслав ему свой вечный инвайт
    '''
    if provider == 'email':
        try:
            userinfo = passport.userinfo(login=userid)
        except Exception:
            userinfo = {}
        if (userinfo and
                userinfo['has_disk'] and
                settings.feature_toggles['disallow_invite_disk_users']):
            raise errors.UserAlreadyHaveDisk


    code = personal_hash(uid)
    
    user_info = passport.userinfo(uid)
    params.update({
        'code'      : code,
        'friendName': user_info['public_name'],
    })
    
    if 'locale' not in params:
        params['locale'] = user_info['language']
        
    from mpfs.core.operations import manager
    return manager.create_operation(
        uid, 'invites', 'send',
        odata=dict( 
            provider = provider,
            userid   = userid,
            template = 'referral/offer',
            params   = params,
        )
    )            

def list_contacts(uid):
    '''
    Постановка задачи на получение списка контактов со статусами инвайтированности
    '''
    from mpfs.core.operations import manager
    return manager.create_operation(
        uid, 'invites', 'list_contacts',
        {}
    )
