# -*- coding: utf-8 -*-
"""

MPFS
CORE

Настройки и состояния пользователя

"""
import operator
from itertools import ifilter

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.core.metastorage.control import disk_info as control
from mpfs.core.user.constants import *
from mpfs.core.queue import mpfs_queue
from mpfs.common.util import mailer
from mpfs.core.services.passport_service import Passport
from mpfs.common.util import ctimestamp, hashed
from mpfs.core.billing import api as billing_api
from mpfs.metastorage.mongo.util import generate_version_number, name_for_key
from mpfs.core.user.dao.setting import SettingsDAO


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport = Passport()


def _create_storage_path(uid, path):
    """Создание каталогов для хранения"""
    parts = path.split(os.path.sep)
    storage_path = ''

    for chunk in parts:
        storage_path = storage_path + os.path.sep + chunk
        response = control.value(uid, storage_path)
        if response.value is None:
            try:
                response = control.make_folder(uid, storage_path, {})
            except errors.StorageFolderAlreadyExist:
                pass

    return response.value


def _enlarge_space(uid, reason, space):
    """Единый метод увеличения места за состояние

    Нужен для переходного периода с Биллингом
    """
    billing_api._temporary_enlarge_space(uid, reason, line='bonus', space=space)


def _reduce_space(uid, reason, space):
    """Единый метод уменьшения места за состояние

    Нужен для переходного периода с Биллингом
    """
    billing_api._temporary_reduce_space(uid, reason, space=space)


class UserKeyValueData(object):
    """Абстрактный класс каких-то простых данных пользователя"""
    type = None

    def __init__(self, uid, project=None, allow_empty=False):
        if not project:
            project = DEFAULT_PROJECT
        self.data = {}
        self.uid = uid
        self.project = project
        key = os.path.sep + self.type + os.path.sep + project + os.path.sep
        self.storage_parent = key

        try:
            response = control.folder_content(uid, key)
        except errors.StorageInitUser, e:
            response = None
        except errors.StorageEmptyDiskInfo, e:
            if allow_empty:
                response = None
            else:
                raise e

        if not response or not response.value:
            self.need_storage = True
            self.version = None
        else:
            self.need_storage = False
            self.version = response.version
            for namespace, values in response.value.iteritems():
                self.data[namespace] = dict(values.data)

    def list(self, namespace=None):
        """Выдать значения по определенному неймспейсу"""
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        return self.data.get(namespace, {})

    def list_all(self):
        """Выдать все возможные неймспейсы и значения"""
        return self.data

    def get(self, key, namespace=DEFAULT_NAMESPACE):
        return self.data.get(namespace, {}).get(key)

    def set(self, key, value, namespace=None, pended=True):
        """Установить значение в определеном неймспейсе

        Если ключ входит в список SET_STATES_SETTINGS, то
        будет его обрабатывать в очереди
        """
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        stored_dict = self.data.get(namespace, {})

        action = 'update' if key in stored_dict else 'set'
        if action == 'update' and stored_dict.get(key) == value:
            return

        needs_async_post_process = SET_STATES_SETTINGS.get(self.project, {}).get(self.type, {}).get(namespace, [])
        deny_async_post_process = DENY_MANUAL_STATE_SETTINGS.get(self.project, {}).get(self.type, {}).get(namespace, [])

        if key in needs_async_post_process and action == 'set':
            if (key in deny_async_post_process and not pended) or (key not in deny_async_post_process):
                self.post_process(key, value, namespace, action, pended=pended)
        else:
            self.update_and_save(key, value, namespace)

    def bulk_set(self, new_data, namespace=None):
        """Массовое сохранение значений.

        Обрабатывает синхронно, в очередь ничего не ставит
        Нельзя применять на состояния, связанные с выдаче места
        """
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        stored_dict = self.data.get(namespace, {})

        for key, value in new_data.iteritems():
            stored_dict[key] = value

        self.data[namespace] = stored_dict
        self._save(namespace)

        for key, value in new_data.iteritems():
            log.info('%s %s:%s:%s:%s set %s' % (self.type, self.uid, self.project, namespace, key, value))

    def remove(self, key, namespace=None, pended=True):
        """Удалить значение из неймспейса

        Если ключ входит в список REMOVE_STATES_SETTINGS, то
        будет его обрабатывать в очереди
        """
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        stored_dict = self.data.get(namespace, {})

        needs_post_process = REMOVE_STATES_SETTINGS.get(self.project, {}).get(self.type, {}).get(namespace, [])

        if key in stored_dict and key in needs_post_process:
            self.post_process(key, None, namespace, action='remove', pended=pended)
        elif key in stored_dict:
            self.remove_and_save(key, namespace)
        else:
            raise errors.SettingStateNotFound()

    def clear(self):
        """Очистить все неймспейсы

        Данный метод ничего не изменяет в базе, только в памяти объекта
        """
        for namespace, data in self.data.items():
            del(self.data[namespace])
            log.info('%s %s:%s:%s clear namespace' % (self.type, self.uid, self.project, namespace))

    def update_and_save(self, key, value, namespace):
        stored_dict = self.data.get(namespace, {})
        stored_dict[key] = value
        self.data[namespace] = stored_dict
        self._save(namespace)
        log.info('%s %s:%s:%s:%s set %s' % (self.type, self.uid, self.project, namespace, key, value))

    def remove_and_save(self, key, namespace):
        stored_dict = self.data.get(namespace, {})
        if key in stored_dict:
            del(stored_dict[key])
            self.data[namespace] = stored_dict
            self._save(namespace)
            log.info('%s %s:%s:%s:%s removed' % (self.type, self.uid, self.project, namespace, key))
        else:
            log.info('%s %s:%s:%s:%s does not exist, remove ignored' % (self.type, self.uid, self.project, namespace,
                                                                        key))

    def _save(self, namespace):
        if self.need_storage:
            _create_storage_path(self.uid, self.storage_parent)

        stored_key = self.storage_parent + namespace
        stored_dict = self.data[namespace]
        stored_data = sorted(stored_dict.iteritems(), key=operator.itemgetter(0))
        response = control.put(self.uid, stored_key, stored_data, version=self.version)
        self.version = int(response.version)

    def post_process(self, key, value, namespace, action, pended=True):
        if pended:
            job = {
                'action': action,
                'uid': self.uid,
                'project': self.project,
                'type': self.type,
                'key': key,
                'namespace': namespace,
                'value': value,
            }
            log.info('send post process %s %s:%s:%s:%s' % (self.type, self.uid, self.project, namespace, key))
            mpfs_queue.put(job, 'post_process_user_var')
        else:
            self.immediately_post_process(key, value, namespace, action)

    def immediately_post_process(self, key, value, namespace, action):
        if action == 'remove':
            self.remove_and_save(key, namespace)
        else:
            self.update_and_save(key, value, namespace)


class Settings(object):
    """Класс настроек пользователя"""
    type = 'settings'
    dao = SettingsDAO()

    def __init__(self, uid, project=None, allow_empty=False):
        if not project:
            project = DEFAULT_PROJECT
        self.data = {}
        self.uid = uid
        self.project = project
        self.storage_parent = '/%s/%s/' % (self.type, project)
        self.allow_empty = allow_empty
        self._fetch()

    def list(self, namespace=None):
        """Выдать значения по определенному неймспейсу"""
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        return self.data.get(namespace, {})

    def list_all(self):
        """Выдать все возможные неймспейсы и значения"""
        return self.data

    def get(self, key, namespace=DEFAULT_NAMESPACE):
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        return self.data.get(namespace, {}).get(key)

    def set(self, key, value, namespace=DEFAULT_NAMESPACE, pended=True):
        self._update(key, value, namespace)
        log.info('%s %s:%s:%s:%s set %s' % (
            self.type, self.uid, self.project, namespace, key, value
        ))

    def remove(self, key, namespace=DEFAULT_NAMESPACE, pended=True):
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        self._update(key, None, namespace, remove=True)
        log.info('%s %s:%s:%s:%s remove' % (
            self.type, self.uid, self.project, namespace, key
        ))

    def _fetch(self):
        self.data = {}
        rows = self.dao.get_keys(self.uid)
        if not self.allow_empty and not rows:
            raise errors.StorageEmptyDiskInfo

        for row in ifilter(lambda x: x.path.startswith(self.storage_parent), rows):
            namespace = name_for_key(row.path)
            self.data[namespace] = {}
            for k, v in row.data:
                self.data[namespace][k] = v

    def _update(self, key, value, namespace=DEFAULT_NAMESPACE, remove=False):
        if namespace is None:
            namespace = DEFAULT_NAMESPACE
        _id, parent, path = self._get_ids(namespace)
        new_version = generate_version_number()
        if remove:
            setting = self.dao.remove_key(_id, self.uid, new_version, key)
        else:
            setting = self.dao.set_key(_id, self.uid, path, new_version, key, value)
        self.data[namespace] = dict(setting.data)

    @staticmethod
    def _normalize_path(path):
        return '/' + '/'.join(filter(None, path.split('/')))

    def _get_ids(self, namespace):
        uid = str(self.uid)
        path = self._normalize_path(self.storage_parent + namespace)
        _id = hashed('%s:%s' % (uid, path))
        parent = None
        return _id, parent, path


class States(UserKeyValueData):
    """Класс состояний пользователя"""
    type = 'states'

    def immediately_post_process(self, state, value, namespace, action):
        """Пост-бработка установки пользовательских состояний"""
        possible_actions = {
            'set': self._post_process_set,
            'remove': self._post_process_remove
        }

        method = possible_actions.get(action)
        if method:
            method(state, value, namespace)

    def _post_process_set(self, state, value, namespace):
        reason = INCREMENT_REASONS[state]
        states = SPACE_INCREMENTS[reason]['states']
        border = SPACE_INCREMENTS[reason]['border']

        if border and ctimestamp() >= border:
            return

        from mpfs.core.user.base import User
        user = User(self.uid, self.project)

        set_list = self.list(namespace)

        """
        if reason == 'upload' and not filter(lambda s: s in set_list, SPACE_INCREMENTS['client']['states']):
            increment_value = SPACE_INCREMENTS['client']['value']
            increment_name  = SPACE_INCREMENTS['client']['states'][0]
            self.update_and_save(increment_name, 1, namespace)
            _enlarge_space(user.uid, increment_name, increment_value)
        """

        set_list = self.list(namespace)
        increment_value = SPACE_INCREMENTS[reason]['value']
        if not filter(lambda s: s in set_list, states):
            _enlarge_space(user.uid, state, increment_value)

        self.update_and_save(state, value, namespace)

        if state == 'mobile_installed':
            user.passport_subscribe_mobile()

        if state == 'desktop_installed':
            user.passport_subscribe_desktop()

        if state == 'photostream_used':
            userinfo = passport.userinfo(self.uid)
            params = {
                'username': userinfo['username'],
                'login': userinfo['login'],
                'locale': userinfo['language'],
            }
            mailer.send(userinfo['email'], 'spacePack/camera', params)

    def _post_process_remove(self, state, value, namespace):
        reason = INCREMENT_REASONS[state]
        states = SPACE_INCREMENTS[reason]['states']
        set_list = self.list(namespace)

        from mpfs.core.user.base import User
        user = User(self.uid, self.project)

        # Раньше мы вадавали отдельную услугу за инсталяицю моб. устройства
        # сейчас этого не делаем и услуги может не быть
        try:
            decrement_value = SPACE_INCREMENTS[reason]['value']
            if filter(lambda s: s in set_list, states):
                _reduce_space(user.uid, state, decrement_value)
        except Exception:
            pass

        self.remove_and_save(state, namespace)
