# -*- coding: utf-8 -*-
"""

MPFS
CORE

Общий пользователь

"""
import datetime
import time
import traceback
import uuid

from itertools import izip, repeat

import mpfs.engine.process
from mpfs.common.static.tags import experiment_names
from mpfs.common.static.tags.experiment_names import (
    REQUESTS_TO_COMMON_FOR_GROUPS_USE_FLAG
)
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.overdraft import OVERDRAFT_STATUS_FIELD, LIGHT_OVERDRAFT_STATUS, HARD_OVERDRAFT_STATUS, \
    BLOCK_OVERDRAFT_STATUS

from mpfs.config import settings
from mpfs.core.albums.static import FacesIndexingState
from mpfs.core.billing import ProductCard
from mpfs.core.billing.constants import PRODUCT_INITIAL_10GB_ID
from mpfs.core.billing.processing.repair import recalculate_limit_by_services
from mpfs.core.billing.product.default_products_service import DefaultProductsService
from mpfs.core.filesystem.quota import Quota
from mpfs.core.office.static import OfficeSelectionStrategyConst
from mpfs.core.office.util import get_editor, check_user_country_match
from mpfs.core.services.passport_service import passport
from mpfs.common import errors
from mpfs.common.static.tags.billing import CLIENT
from mpfs.common.util import filter_uid_by_percentage, CacheMixin
from mpfs.core.email.logic import send_email_async_by_uid
from mpfs.core.metastorage.control import link_data, disk, trash, disk_info, hidden_data
from mpfs.core.filesystem.base import Filesystem
from mpfs.core.user.dao.user import UserDAO
from mpfs.core.user.settings import Settings, States
from mpfs.core.user.devices import Devices
from mpfs.core.user.constants import *
from mpfs.core.queue import mpfs_queue
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.user.common import CommonUser
from mpfs.core import billing
from mpfs.core.billing.processing.common import simple_create_service
from mpfs.core.billing.service import ServiceList
from mpfs.core.billing.product.catalog import PRIMARY_PRODUCTS_IDS, ROSTELECOM_UNLIM_PRODUCTS_IDS, \
    PARTNERS_PAID_PRODUCTS_IDS, PS_BILLING_PRODUCT_IDS
from mpfs.core.user.events import EnlargeSpaceEvent, ReduceSpaceEvent
from mpfs.core.billing.client import Client
from mpfs.core.organizations.logic import update_organization_async

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
push_queue = PushQueue()

usrctl = mpfs.engine.process.usrctl()
dbctl = mpfs.engine.process.dbctl()

FEATURE_TOGGLES_B2B_ADD_SPACE = settings.feature_toggles['b2b_add_space']
FEATURE_TOGGLES_REGISTER_HANCOM_USERS = settings.feature_toggles['register_hancom_users']
OFFICE_HANCOM_USER_PERCENTAGE = settings.office['hancom_user_percentage']
FEATURE_TOGGLES_ADD_YANDEX_PLUS_SERVICE_ON_USER_INIT = settings.feature_toggles['add_yandex_plus_service_on_user_init']
USER_INIT_LOCK_PERCENTAGE = settings.feature_toggles['user_init_lock_percentage']
PAID_PRODUCTS_IDS = (PRIMARY_PRODUCTS_IDS | ROSTELECOM_UNLIM_PRODUCTS_IDS | PARTNERS_PAID_PRODUCTS_IDS |
                     # https://st.yandex-team.ru/CHEMODAN-72577
                     {'force_yandex_pro'})
PROMO_USER_INIT_DISCOUNT_2018_START_TS = settings.promo['user_init_discount_2018']['start_ts']
PROMO_USER_INIT_DISCOUNT_2018_END_TS = settings.promo['user_init_discount_2018']['end_ts']
PROMO_USER_INIT_DISCOUNT_2018_DISCOUNT_TEMPLATE_ID = settings.promo['user_init_discount_2018']['discount_template_id']
PROMO_USER_INIT_DISCOUNT_2018_ENABLED_UIDS_SURELY = settings.promo['user_init_discount_2018']['enabled_uids_surely']
PROMO_USER_INIT_DISCOUNT_2018_UID_HASH_SEED = settings.promo['user_init_discount_2018']['uid_hash_seed']
POSTGRES_QUICK_MOVE_ENABLE_QUICK_MOVE_FOR_NEW_USERS = settings.postgres['quick_move']['enable_quick_move_for_new_users']
USER_DEFAULT_COUNTRY = settings.user['default_country']

FACES_ALLOWED_COUNTRIES = {'BY', 'KZ', 'RU', 'UA'}  # sync with /disk/djfs/core/src/main/java/ru/yandex/chemodan/app/djfs/core/album/FacesAlbumManager.java:ALLOWED_COUNTRIES


class StandartUser(CacheMixin, CommonUser):

    required_user_domains = (
        disk,
        disk_info,
        trash,
        hidden_data,
        link_data,
    )

    required_folders = (
        list(izip(REQUIRED_CONTENT_FOLDERS, repeat(disk, len(REQUIRED_CONTENT_FOLDERS)))) +
        list(izip(REQUIRED_TRASH_FOLDERS, repeat(trash, len(REQUIRED_TRASH_FOLDERS)))) +
        list(izip(REQUIRED_FOLDERS_IN_HIDDEN, repeat(hidden_data, len(REQUIRED_FOLDERS_IN_HIDDEN))))
    )

    required_files = list(izip(REQUIRED_DISK_INFO_FILES, repeat(disk_info, len(REQUIRED_DISK_INFO_FILES))))

    predefined_content_folders = PREDEFINED_CONTENT_FOLDERS_2014
    predefined_content_files = PREDEFINED_CONTENT_FILES_2014
    subscribe_passport = True
    type = 'standart'

    def __init__(self, uid, project='disk', info=None):
        super(StandartUser, self).__init__(uid, project, info)

        self._states = None
        self._settings = None
        self._devices = None
        self.fs = Filesystem()

        self.has_attach = None
        if info:
            self.has_attach = 'attach_data' in info.get('collections')

    @property
    def states(self):
        if self._states is None:
            self._states = States(self.uid, self.project)
        return self._states

    @property
    def settings(self):
        if self._settings is None:
            self._settings = Settings(self.uid, self.project)
        return self._settings

    @property
    def devices(self):
        if self._devices is None:
            self._devices = Devices(self.uid, self.project)
        return self._devices

    @classmethod
    def Create(cls, uid, project='disk', **params):
        """Инициализация пользователя"""
        userinfo = passport.userinfo(uid)
        already_subscribed = userinfo.get('has_disk', False)
        pdd = None
        if passport.is_from_pdd(uid=uid):
            pdd = {'domain': userinfo['pdd_domain']}

        is_mailish = userinfo.get('is_mailish', False)

        if not is_mailish:
            try:
                # Подписываемся и получаем данные от Паспорта
                cls.passport_subscribe(uid)
            except errors.PassportPasswordNeeded:
                raise
            except Exception:
                log.warning('user creation failed, rollbacking %s' % uid)
                cls.passport_unsubscribe(uid)
                raise

        if filter_uid_by_percentage(uid, USER_INIT_LOCK_PERCENTAGE):
            lock_key = uuid.uuid4().hex
        else:
            lock_key = None

        try:
            passport.reset()
            userinfo = passport.userinfo(uid)

            db_info = usrctl.check(uid)
            new_user_creation = False

            locale = None
            if not db_info or db_info.get('type') != cls.type:
                new_user_creation = True
            elif 'locale' not in db_info:
                log.error('User exists but locale is missing uid=%s' % uid)
            else:
                locale = db_info['locale']

            # Вычисляем локаль пользователя
            if locale:
                pass
            elif params.get('locale') in SUPPORTED_LOCALES:
                locale = params.get('locale')
            elif userinfo.get('language') in SUPPORTED_LOCALES:
                locale = userinfo.get('language')
            else:
                locale = DEFAULT_LOCALE

            # Добавляем услугу за Яндекс.Плюс, если пользователь подписан
            if FEATURE_TOGGLES_ADD_YANDEX_PLUS_SERVICE_ON_USER_INIT:
                if userinfo.get('has_plus', False):
                    params['add_services'] = params.get('add_services', []) + ['yandex_plus_10gb']

            # Костыль, для SSO организаций добавляем место в диске, т.к. пс-биллинг пока не знает про такие организации
            sso_organizations = dict()
            for sso_oragnization in settings.billing['sso_organization_space']:
                sso_oragnization_chunks = sso_oragnization.split(':')
                if len(sso_oragnization_chunks) != 2:
                    error_log.error('Incorrect sso_organization_space setting key format, get %s', sso_oragnization)
                    continue

                domid, space = sso_oragnization_chunks
                if not domid.isdigit() or not space.isdigit():
                    error_log.error('Incorrect sso_organization_space setting key format, get %s', sso_oragnization)
                    continue

                sso_organizations[domid] = long(sso_oragnization_chunks[1])

            if userinfo.get('domain_id', '') in sso_organizations and userinfo.get('sso_user', ''):
                product = 'yandex_b2b_mail_pro:' + str(sso_organizations[userinfo.get('domain_id', '')])
                params['add_services'] = params.get('add_services', []) + [product]
            # Костыль закончен

            country = userinfo.get('country', USER_DEFAULT_COUNTRY)

            if country is None or country.upper() in FACES_ALLOWED_COUNTRIES:
                if experiment_manager.is_feature_active('index_faces_for_new_users'):
                    faces_indexing_state = FacesIndexingState.REINDEXED
                else:
                    faces_indexing_state = None
            else:
                faces_indexing_state = FacesIndexingState.COUNTRY_RESTRICTION

            # заводим юзера в базе
            shard = params.get('shard')
            b2b_key = params.get('b2b_key')
            hancom_enabled = None
            if (FEATURE_TOGGLES_REGISTER_HANCOM_USERS
                and filter_uid_by_percentage(uid, OFFICE_HANCOM_USER_PERCENTAGE)):
                hancom_enabled = True
            office_selection_strategy = None
            if (experiment_manager.is_feature_active(experiment_names.ONLYOFFICE_ON_USERINIT) and
                    check_user_country_match(uid, 'RU')):
                office_selection_strategy = OfficeSelectionStrategyConst.FORCE_OO

            default_product_id = DefaultProductsService.get_product_id()

            cls._create_user_in_db(uid, locale, shard, b2b_key, pdd, is_mailish, hancom_enabled, lock_key=lock_key,
                                   enable_quick_move=POSTGRES_QUICK_MOVE_ENABLE_QUICK_MOVE_FOR_NEW_USERS, is_paid=False,
                                   faces_indexing_state=faces_indexing_state,
                                   office_selection_strategy=office_selection_strategy,
                                   default_product_id=default_product_id)
            cls._create_domains(uid)
            cls._create_system_folders(uid)
            cls._create_system_files(uid)
            cls._create_initial_services(uid, **params)
            cls._create_user_folders(uid, locale)
            cls._bind_to_market(uid, country)
            cls._save_country(uid, country)
            cls._init_geo_albums(uid)

            if not already_subscribed:
                # FIXME
                # тк для mailish пользователей мы не можем подписать его на наш sid в паспорте
                # (нужны доработки в паспорте), то здесь возможен баг для таких пользователей
                # а именно, если вдруг каким макаром они удалят предопределенные файлы, и вызовется
                # снова создание/инициализация, то мы их пересоздадим ему вновь
                # чтобы этого избежать нужна доработка со стороны паспорта, чтобы мы могли их подписывать
                # (тем самым это как бы флаг, что файлы мы ему клали и все прошло ок), либо заводить флаг в базе
                cls._copy_predefined_files(uid, locale)

            cls._post_process_user(uid, userinfo, locale)

            # Запускаем пост-процессинг реферальной программы
            if 'referral' in params:
                job = {'uid': uid, 'referral': params.get('referral')}
                mpfs_queue.put(job, 'post_process_user_init')
                log.info('send post init %s, params %s' % (uid, params))

            if new_user_creation and not params.get('noemail') and not already_subscribed and not is_mailish:
                cls.send_welcome_mail(uid, locale, params.get('source'))

            if b2b_key:
                update_organization_async(b2b_key)

            if lock_key:
                cls._release_lock(uid)  # удаляем даже чужой лок, т.к. все успешно дошло до конца
        except errors.StorageInitUserImpossibleToLock:
            if not already_subscribed:
                cls.passport_unsubscribe(uid)
            log.warning('another user init in progress for user %s' % uid)
            raise
        except Exception:
            log.warning('user creation failed, rollbacking %s' % uid)
            if not already_subscribed:
                cls.passport_unsubscribe(uid)
            error_log.error(traceback.format_exc())
            if lock_key:
                try:
                    cls._remove_own_lock_timestamp(uid, lock_key)
                except Exception:
                    error_log.exception('Removing lock timestamp for user %s failed' % uid)
            raise

    @classmethod
    def _create_initial_services(cls, uid, **params):
        """Create initial services if needed"""
        products = []
        is_share = uid == mpfs.engine.process.share_user()

        if not is_share:
            # add default virtual product space amount
            user = cls(uid)
            product = DefaultProductsService.get_by_product_id(user.get_default_product_id())
            user.enlarge_space(product.attributes.amount, reason=product.pid)

        if params.get('add_services'):
            products.extend(params.get('add_services'))

        client = billing.Client(uid)
        for p in products:
            try:
                product, attributes = billing.Product.parse(p)
                simple_create_service(client, product, attributes=attributes)
            except errors.billing.BillingProductIsSingletone:
                log.info('Product %s for user %s already exist, skipping' % (p, uid))

        recalculate_limit_by_services(uid)

    @classmethod
    def _post_process_user(cls, uid, userinfo, locale):
        """Установка параметров зависящих от типа пользователя"""
        user = cls(uid)

        # Выдаем гигабайты за яндексовость и создаем NDA папку
        if userinfo['has_staff']:
            user.set_state('yandex_staff', 1)
            from mpfs.core.yateam.logic import update_user_yateam_status_async
            update_user_yateam_status_async(uid)

        # Добавляем турецким юзерам 1ГБ за лояльность
        # https://jira.yandex-team.ru/browse/CHEMODAN-5208
        if locale == 'tr':
            try:
                reg_date = userinfo.get('reg_date')
                turkish = userinfo.get('country') == 'tr'
                if turkish and reg_date is not None and reg_date <= TURKISH_BORDER_DATE:
                    user.enlarge_space(TURKISH_SPACE_BONUS, reason=TURKISH_REASON)
            except Exception:
                log.warning('failed while trying to increase turkish space %s' % uid)
                error_log.error(traceback.format_exc())

    def info(self):
        """Выдать полную информации о пользователе: настройки, состояния, разное"""
        common = super(StandartUser, self).info()

        result = {
            'settings': self.settings.list_all(),
            'states': self.states.list_all(),
            'devices': self.devices.list_all(),
            'reg_time': self.reg_time,
            'locale': self.locale,
        }

        if self.project == 'disk':
            if 'invites' in result['states']:
                del result['states']['invites']

            unlimited_autouploding = {}
            for field in ('unlimited_video_autouploading_enabled', 'unlimited_photo_autouploading_enabled',
                          'unlimited_video_autouploading_reason', 'unlimited_photo_autouploading_allowed',
                          'unlimited_video_autouploading_allowed'):
                unlimited_autouploding[field] = self._user_index.get(field)

            unlimited_video_autouploading_enabled = int(self.is_unlimited_autouploading_enabled())
            if unlimited_autouploding['unlimited_video_autouploading_enabled'] is not None:
                unlimited_video_autouploading_enabled = int(unlimited_autouploding['unlimited_video_autouploading_enabled'])

            unlimited_photo_autouploading_enabled = int(self.is_unlimited_autouploading_enabled())
            if unlimited_autouploding['unlimited_photo_autouploading_enabled'] is not None:
                unlimited_photo_autouploading_enabled = int(unlimited_autouploding['unlimited_photo_autouploading_enabled'])

            unlimited_photo_autouploading_allowed = int(self.is_unlimited_photo_autouploading_allowed())
            if unlimited_autouploding['unlimited_photo_autouploading_allowed'] is not None:
                unlimited_photo_autouploading_allowed = int(unlimited_autouploding['unlimited_photo_autouploading_allowed'])

            unlimited_video_autouploading_allowed = int(self.is_unlimited_video_autouploading_allowed())
            if unlimited_autouploding['unlimited_video_autouploading_allowed'] is not None:
                unlimited_video_autouploading_allowed = int(unlimited_autouploding['unlimited_video_autouploading_allowed'])

            user_editor = get_editor(self.uid)
            result.update({
                'space': self.fs.quota.report(self.uid),
                'blocked': self.is_blocked(),
                'paid': int(self.is_paid()),
                'is_mailish': int(self.is_mailish()),
                'advertising_enabled': int(self.is_advertising_enabled()),
                'unlimited_autoupload_enabled': int(self.is_unlimited_autouploading_enabled()),
                'unlimited_video_autoupload_enabled': unlimited_video_autouploading_enabled,
                'unlimited_video_autoupload_reason': unlimited_autouploding['unlimited_video_autouploading_reason'],
                'unlimited_video_autoupload_allowed': unlimited_video_autouploading_allowed,
                'unlimited_photo_autoupload_enabled': unlimited_photo_autouploading_enabled,
                'unlimited_photo_autoupload_allowed': unlimited_photo_autouploading_allowed,
                'is_overdrawn': int(self.is_overdrawn()),
                'photoslice_albums_enabled': 1,
                'office_online_editor_type': self.get_online_editor(),
                'office_online_editor_name': user_editor.type_label if user_editor else None,
                'faces_indexing_state': self.faces_indexing_state,
                'office_selection_strategy': self.get_office_selection_strategy(),
                'user_last_mso_usage': self.get_last_mso_usage()
            })
            result['default_folders'] = self.get_default_folders()

        if self.check_deleted():
            result['deleted'] = self.check_deleted()

        if self.has_attach is not None:
            result['has_attach'] = str(int(self.has_attach))

        result['trash_autoclean_period'] = settings.system['system']['trash_autoclean_period']
        result.update(common)
        if self.is_in_overdraft_for_restrictions() and experiment_manager.is_feature_active('new_overdraft_strategy'):  # если пользователь - овердрафтник
            result.update(self.get_overdraft_info())

        passport_user_info = passport.userinfo(self.uid)
        if passport_user_info.get('pdd_org_id'):
            result['pdd_org_id'] = passport_user_info['pdd_org_id']

        return result

    def is_paid(self):
        """Есть ли у пользователя активные платные подписки.

        Кеширует список билинговых услуг!
        :rtype bool
        """
        if self._user_index:
            is_paid = self._user_index.get('is_paid')
            if is_paid is not None:
                return is_paid

        cache_key = ('service_list', self.uid)
        service_list = self.cache_get(cache_key)
        if service_list is None:
            client = Client(self.uid)
            from mpfs.core.metastorage.decorators import secondary_preferred_mode
            with secondary_preferred_mode():
                service_list = ServiceList(client=client)
            self.cache_set(cache_key, service_list)

        for s in service_list:
            if s['pid'] in PAID_PRODUCTS_IDS:
                return True
        return False

    def set_is_paid(self, is_paid):
        # служебный метод, проставляет в disk.user_index, чтобы каждый раз не приходилось ходить в common за услугами
        UserDAO().set_is_paid(self.uid, is_paid)

    def is_all_files_versioning_enabled(self):
        """Значение настройки 'версионировать все файлы пользователя'

        Напрямую пользоваться нельзя. Используй `ResourceVersionManager.is_versionable`
        """
        setting = self.settings.get('allFilesVersioningEnabled', namespace='verstka')
        if not setting:
            return False
        return str(setting) == '1'

    def awaps_info(self):
        """Выдача информации о пользователе для awaps

         - Настроена ли автозагрузка
         - Есть ли общие папки
         - Есть ли публичные ссылки
         - Все ли 20 инвайтов реализованы (может ли он получить ещё место за инвайты)
         - пользовался ли скриншотилкой
         - есть ли платные пакеты
        """
        photostream_used = int(bool(self.fs.chksysdir(self.uid, 'photostream', self.get_supported_locale())))

        from mpfs.core.social.share import ShareProcessor
        shared_folders = ShareProcessor().list_all_folders(self.uid)

        public_links = int(bool(link_data.links_exist(self.uid)))

        from mpfs.core.user import invites
        all_invites_activated = int(len(invites.list_activated(self.uid)) >= 20)

        screenshot = int(bool(self.fs.chksysdir(self.uid, 'screenshots', self.get_supported_locale())))

        result = {
            'photostream_used': photostream_used,
            'shared_folders': int(bool(shared_folders['owned'] or shared_folders['joined'])),
            'public_links': public_links,
            'all_invites_activated': all_invites_activated,
            'screenshot': screenshot,
            'paid_services': int(self.is_paid()),
        }
        return result

    def enlarge_space(self, added, reason=None):
        """Увеличение лимита места"""
        limit = self.fs.quota.limit(uid=self.uid)
        new_limit = int(limit) + added
        self.fs.quota.change_limit_on_delta(added, uid=self.uid)

        EnlargeSpaceEvent(uid=self.uid, old_limit=limit, new_limit=new_limit, reason=reason).send()

        # https://jira.yandex-team.ru/browse/CHEMODAN-7291
        # сохраняем причину выдачи
        #space_subscribes.put_one({
        #    'uid'    : self.uid,
        #    'reason' : reason,
        #    'date'   : int(time.time()),
        #    'add'    : added,
        #    'old'    : limit,
        #    'new'    : new_limit,
        #})

    def reduce_space(self, reduced, reason=None):
        """Уменьшение лимита места"""
        limit = self.fs.quota.limit(uid=self.uid)
        new_limit = int(limit) - reduced
        self.fs.quota.change_limit_on_delta(-reduced, uid=self.uid)

        ReduceSpaceEvent(uid=self.uid, old_limit=limit, new_limit=new_limit, reason=reason).send()

        # удаляем причину выдачи
        #space_subscribes.remove(uid=self.uid, reason=reason)

    def passport_subscribe_mobile(self):
        if not passport.userinfo(self.uid).get('has_mobile_disk'):
            try:
                self.passport_subscribe(self.uid, sid='mobilecloud')
            except Exception:
                error_log.warning(traceback.format_exc())

    def passport_subscribe_desktop(self):
        if not passport.userinfo(self.uid).get('has_desktop_disk'):
            try:
                self.passport_subscribe(self.uid, sid='cloudsoft')
            except Exception:
                error_log.warning(traceback.format_exc())

    def passport_subscribe_undeletable(self):
        if not passport.userinfo(self.uid).get('undeletable'):
            try:
                self.passport_subscribe(self.uid, sid='cloudpaid')
            except Exception:
                error_log.warning(traceback.format_exc())

    def passport_subscribe_migrant(self):
        self.passport_subscribe(self.uid, sid='cloudmigrant')

    def set_state(self, key, value, namespace=None):
        """Установка состояния"""
        self.states.set(key, value, namespace)

    def remove_state(self, key, namespace=None):
        """Удаление состояния"""
        self.states.remove(key, namespace)

    def set_setting(self, key, value, namespace=None):
        """Установка настройки"""
        self.settings.set(key, value, namespace)

    def remove_setting(self, key, namespace=None):
        """Удаление настройки"""
        self.settings.remove(key, namespace)

    def install_device(self, type, id, info):
        """Инсталляция устройства"""
        job = {
            'uid': self.uid,
            'id': id,
            'type': type,
            'info': info
        }
        mpfs_queue.put(job, 'process_device_install')
        log.info('send device install %s, params %s' % (self.uid, job))

    def uninstall_device(self, type, id):
        """Деинсталляция устройства"""
        self.devices.uninstall(type, id)

    def service_list(self):
        """Возвращает список всех услуг пользователя."""
        return ServiceList(**{CLIENT: Client(self.uid)})

    def public_info(self, raise_if_does_not_exist=False):
        """Возвращает публичные данные пользователя"""
        result = super(StandartUser, self).public_info(raise_if_does_not_exist=raise_if_does_not_exist)
        result['paid'] = int(self.is_paid())
        result['advertising_enabled'] = int(self.is_advertising_enabled())
        return result

    @staticmethod
    def get_welcome_campaign_name(registration_source):
        return settings.welcome_email_campaigns['reg_source_to_campaign_map'].get(
            registration_source, settings.welcome_email_campaigns['default'])

    @classmethod
    def send_welcome_mail(cls,uid, locale, registration_source):
        campaign_name = cls.get_welcome_campaign_name(registration_source)
        send_email_async_by_uid(
            uid,
            campaign_name,
            template_args={'products': ProductCard.get_current_products_for(uid, skip_third_tariff=True)},
            locale=locale,
        )

    @property
    def versioning_extended_period_enabled(self):
        return self._is_priority

    @property
    def priority_support_enabled(self):
        return self._is_priority

    @property
    def advertising_enabled(self):
        if self._is_priority:
            return False
        if self.get_advertising_enabled() is False:
            return False
        return True

    @property
    def antifo_enabled(self):
        return not self._is_priority

    @property
    def disk_pro_enabled(self):
        return self._is_priority

    @property
    def only_disk_pro_without_ps_billing_enabled(self):
        cache_key = ('_is_only_disk_pro_without_ps_billing_enabled', self.uid)
        is_priority = self.cache_get(cache_key)
        if is_priority is None:
            is_priority = self.is_yateam() or self.settings.get(IS_PAID_WITHOUT_PS_BILLING) or self.is_b2b_paid()
            self.cache_set(cache_key, is_priority)
        return is_priority

    @property
    def only_disk_pro_without_ps_billing_and_b2b_enabled(self):
        cache_key = ('_is_only_disk_pro_without_ps_billing_and_b2b_enabled', self.uid)
        is_priority = self.cache_get(cache_key)
        if is_priority is None:
            is_priority = self.is_yateam() or self.settings.get(IS_PAID_WITHOUT_PS_BILLING)
            self.cache_set(cache_key, is_priority)
        return is_priority

    @property
    def desktop_folder_autosave_enabled(self):
        return self._is_priority

    @property
    def unlimited_video_autouploading_enabled(self):
        return self._is_priority

    @property
    def faces_indexing_state(self):
        return self._user_index.get('faces_indexing_state')

    @property
    def _is_priority(self):
        cache_key = ('_is_priority', self.uid)
        is_priority = self.cache_get(cache_key)
        if is_priority is None:
            is_priority = self.is_yateam() or self.is_paid() or self.is_b2b_paid()
            self.cache_set(cache_key, is_priority)
        return is_priority

    @classmethod
    def can_init_user(cls, uid):
        user_info = passport.userinfo(uid)
        return (user_info.get('has_portal_login') or
                user_info.get('pdd') or
                user_info.get('is_mailish'))

    def get_last_mso_usage(self):
        mso_usage_info = disk_info.find_one_by_field(self.uid, {'key': MSO_USAGE_KEY})
        if mso_usage_info:
            return mso_usage_info['data'][MSO_USAGE_FIELD]

    def set_last_mso_usage(self, last_mso_usage):
        last_mso_usage_info = {'ctime': int(time.time()), MSO_USAGE_FIELD: last_mso_usage}
        disk_info.put(self.uid, MSO_USAGE_KEY, last_mso_usage_info)

    def has_shared_folders(self):
        from mpfs.core.social.share.entity import SharedEntity
        has_shared_folders = SharedEntity.has_shared(self.uid)

        if not experiment_manager.is_feature_active(REQUESTS_TO_COMMON_FOR_GROUPS_USE_FLAG):
            if not has_shared_folders:
                self.check_has_shared_folder_in_cache()
            return True

        return has_shared_folders

    def check_has_shared_folder_in_cache(self):
        from mpfs.core.social.share import LinkToGroup
        from mpfs.core.social.share import Group
        has_groups = Group.has_data_in_cache_for(self.uid)
        has_group_links = LinkToGroup.has_data_in_cache_for(self.uid)

        # проверяем только если хоть что-то подгружали в кэш
        if has_groups is None and has_group_links is None:
            return

        if has_groups or has_group_links:
            from mpfs.core.user.common import log_smart_common_requests
            log_smart_common_requests(
                'not using flag (has_shared_folders=%s) for uid=%s; match with real state=%s' % (
                False, self.uid, False
            ))
