# -*- coding: utf-8 -*-
import re
import click
import ujson


RESOURCE_ID_RE = re.compile(r'^\d+:\w+')
ADDRESS_RE = re.compile(r'^\d+:/(disk|trash)/')


def _format_resource(resource):
    if resource:
        return {
            'address': resource.address.id,
            'resource_id': resource.resource_id.serialize()
        }
    else:
        return {}


def _format_version_chain(version_chain):
    return version_chain.dao_item.get_mongo_representation()


def _format_version(version):
    return version.dao_item.get_mongo_representation()


@click.command('get_versioned_resources', help=u"Получить все версии пользователя")
@click.argument('uid')
@click.option("--versions", is_flag=True, help=u"Вывести версии")
def get_versioned_resources(uid, versions):
    import mpfs.engine.process
    try:
        mpfs.engine.process.setup_admin_script()
    except RuntimeError:
        pass
    from mpfs.core.versioning.logic.version_manager import ResourceVersionManager
    from mpfs.core.factory import get_resource_by_resource_id
    from mpfs.common.errors import ResourceNotFound

    result = []
    for vc in ResourceVersionManager.fetch_all_version_chains(uid):
        resource_id = vc.resource_id
        try:
            resource = get_resource_by_resource_id(resource_id.uid, resource_id)
        except ResourceNotFound:
            resource = None

        version_info = {
            'resource': _format_resource(resource),
            'version_link': _format_version_chain(vc),
        }
        if versions:
            version_info['versions'] = [_format_version(v) for v in vc.iterate_over_all_versions()]
        result.append(version_info)
    click.echo(ujson.dumps(result))


@click.command('get_versions', help=u"Получить версии ресурса")
@click.argument('addr_or_resource_id')
def get_versions(addr_or_resource_id):
    import mpfs.engine.process
    try:
        mpfs.engine.process.setup_admin_script()
    except RuntimeError:
        pass
    from mpfs.core.address import ResourceId, Address
    from mpfs.core.versioning.logic.version_chain import VersionChain
    from mpfs.core.factory import get_resource, get_resource_by_resource_id
    from mpfs.common.errors import ResourceNotFound

    if ADDRESS_RE.search(addr_or_resource_id):
        addr = Address(addr_or_resource_id)
        resource = get_resource(addr.uid, addr)
        version_chain = VersionChain.get_by_resource_id(resource.resource_id)
    elif RESOURCE_ID_RE.search(addr_or_resource_id):
        resource_id = ResourceId.parse(addr_or_resource_id)
        version_chain = VersionChain.get_by_resource_id(resource_id)
        try:
            resource = get_resource_by_resource_id(resource_id.uid, resource_id)
        except ResourceNotFound:
            resource = None
    else:
        raise NotImplementedError()
    result = {
        'resource': _format_resource(resource),
        'version_link': _format_version_chain(version_chain),
        'versions': [_format_version(v) for v in version_chain.iterate_over_all_versions()],
    }
    click.echo(ujson.dumps(result))


@click.command('put_tasks_to_remove_expired_versions', help=u"Поставить таски чистки версий")
def put_tasks_to_remove_expired_versions():
    import mpfs.engine.process
    mpfs.engine.process.setup_admin_script()
    from mpfs.core.versioning.logic.cleaner import VersionsCleanerManager
    VersionsCleanerManager.run()


@click.group()
def versioning():
    pass


versioning.add_command(get_versioned_resources)
versioning.add_command(get_versions)
versioning.add_command(put_tasks_to_remove_expired_versions)
