# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.dao.base import (
    BaseDAO,
    MongoBaseDAOImplementation,
    PostgresBaseDAOImplementation,
    BulkInsertReqGenerator,
    Session,
)


class BaseVersioningDAO(BaseDAO):
    def save(self, item):
        return self._get_impl_by_uid(item.uid).save(item)

    def delete(self, item):
        return self._get_impl_by_uid(item.uid).delete(item)

    def count_by_uid(self, uid):
        return self._get_impl_by_uid(uid).count_by_uid(uid)

    def remove_by_uid(self, uid):
        return self._get_impl_by_uid(uid).remove_by_uid(uid)

    def fetch_by_uid(self, uid):
        return self._get_impl_by_uid(uid).fetch_by_uid(uid)

    def bulk_insert(self, uid, items):
        if not items:
            return
        for item in items:
            if item.uid is None:
                raise ValueError("%r" % item)
            if uid != item.uid:
                raise ValueError("Can save only for one uid. %s %s" % (uid, item.uid))
        return self._get_impl_by_uid(uid).bulk_insert(uid, items)


class MongoBaseVersioningDAOImplementation(MongoBaseDAOImplementation):
    def save(self, item):
        coll = self.get_collection_by_uid(item.uid)
        coll.update({'uid': item.uid, '_id': item.id}, item.get_mongo_representation(), upsert=True)

    def delete(self, item):
        coll = self.get_collection_by_uid(item.uid)
        coll.remove({'uid': item.uid, '_id': item.id})

    def remove_by_uid(self, uid):
        coll = self.get_collection_by_uid(uid)
        coll.remove({'uid': uid})

    def count_by_uid(self, uid):
        coll = self.get_collection_by_uid(uid)
        return coll.find({'uid': uid}).count()

    def fetch_by_uid(self, uid):
        coll = self.get_collection_by_uid(uid)
        cursor = coll.find({'uid': uid})
        for doc in cursor:
            yield self.doc_to_item(doc)

    def bulk_insert(self, uid, items):
        coll = self.get_collection_by_uid(uid)
        coll.insert([d.get_mongo_representation() for d in items])


class PostgresBaseVersioningDAOImplementation(PostgresBaseDAOImplementation):
    def save(self, item):
        raise NotImplementedError()

    def delete(self, item):
        raise NotImplementedError()

    def count_by_uid(self, uid):
        raise NotImplementedError()

    def remove_by_uid(self, uid):
        raise NotImplementedError()

    def fetch_by_uid(self, uid):
        raise NotImplementedError()

    def bulk_insert(self, uid, items):
        raise NotImplementedError()
