# -*- coding: utf-8 -*-
from pymongo import DESCENDING
from mpfs.dao.base import (
    BaseDAOItem,
    Session,
    BulkInsertReqGenerator
)
from mpfs.core.versioning.dao.base import (
    BaseVersioningDAO,
    MongoBaseVersioningDAOImplementation,
    PostgresBaseVersioningDAOImplementation
)
from mpfs.dao.fields import (
    Md5Field,
    StringField,
    UidField,
    UuidField,
    MSKDateTimeField,
    FileIdField,
)
from mpfs.metastorage.postgres.schema import version_links
from mpfs.metastorage.postgres import versioning_queries


class VersionLinkDAOItem(BaseDAOItem):
    mongo_collection_name = 'version_links'
    postgres_table_obj = version_links
    uid_field_name = 'uid'
    is_sharded = True

    id = UuidField(mongo_path='_id', pg_path=version_links.c.id)
    uid = UidField(mongo_path='uid', pg_path=version_links.c.uid)

    file_id = FileIdField(mongo_path='file_id', pg_path=version_links.c.file_id)
    """file_id ресурса, к которому привязан version_link."""
    disk_path = StringField(mongo_path='disk_path', pg_path=version_links.c.disk_path, default_value=None)
    """Путь в диске до удаления. Нужен для механизма "призрачных" ссылок."""
    disk_path_hash = Md5Field(mongo_path='disk_path_hash', pg_path=version_links.c.disk_path_hash, default_value=None)
    """Хеш от путя path. Нужен для построения индекса по этому полю для механизма "призрачных" ссылок."""
    date_created = MSKDateTimeField(mongo_path='date_created', pg_path=version_links.c.date_created, default_value=None)


class VersionLinkDAO(BaseVersioningDAO):
    dao_item_cls = VersionLinkDAOItem

    def __init__(self):
        super(VersionLinkDAO, self).__init__()
        self._mongo_impl = MongoVersionLinkDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresVersionLinkDAOImplementation(self.dao_item_cls)

    def get_by_id(self, uid, version_link_id):
        return self._get_impl_by_uid(uid).get_by_id(uid, version_link_id)

    def get_by_resource_id(self, resource_id):
        return self._get_impl_by_uid(resource_id.uid).get_by_resource_id(resource_id)

    def get_by_uid_and_path_hash(self, uid, path_hash):
        return self._get_impl_by_uid(uid).get_by_uid_and_path_hash(uid, path_hash)

    def delete_version_links_without_versions(self, shard, batch_size):
        return self._get_impl_by_shard_endpoint(shard).delete_version_links_without_versions(shard, batch_size)


class MongoVersionLinkDAOImplementation(MongoBaseVersioningDAOImplementation):
    def get_by_id(self, uid, version_link_id):
        coll = self.get_collection_by_uid(uid)
        doc = coll.find_one({'uid': uid, '_id': version_link_id})
        return self.doc_to_item(doc)

    def get_by_resource_id(self, resource_id):
        coll = self.get_collection_by_uid(resource_id.uid)
        doc = coll.find_one({'uid': resource_id.uid, 'file_id': resource_id.file_id})
        return self.doc_to_item(doc)

    def get_by_uid_and_path_hash(self, uid, disk_path_hash):
        coll = self.get_collection_by_uid(uid)
        spec = {
            'uid': uid,
            'disk_path_hash': disk_path_hash
        }
        doc = coll.find_one(spec, sort=[('date_created', DESCENDING)])
        return self.doc_to_item(doc)

    def delete_version_links_without_versions(self, shard_endpoint, batch_size):
        return 0


class PostgresVersionLinkDAOImplementation(PostgresBaseVersioningDAOImplementation):
    def get_by_id(self, uid, version_link_id):
        # на момент написания нужно было только для монги (для миграции mongo2pg)
        raise NotImplementedError()

    def get_by_resource_id(self, resource_id):
        uid = self.get_field_repr('uid', resource_id.uid)
        file_id = self.get_field_repr('file_id', resource_id.file_id)

        session = Session.create_from_uid(uid)
        result_proxy = session.execute(
            versioning_queries.SQL_VERSIONING_GET_VERSION_LINK_BY_FILE_ID,
            {'uid': uid, 'file_id': file_id}
        )
        return self.fetch_one_item(result_proxy)

    def get_by_uid_and_path_hash(self, uid, disk_path_hash):
        uid = self.get_field_repr('uid', uid)
        disk_path_hash = self.get_field_repr('disk_path_hash', disk_path_hash)

        session = Session.create_from_uid(uid)
        result_proxy = session.execute(
            versioning_queries.SQL_VERSIONING_GET_VERSION_LINK_BY_PATH_HASH,
            {'uid': uid, 'disk_path_hash': disk_path_hash}
        )
        return self.fetch_one_item(result_proxy)

    def save(self, item):
        params = {c.name: v for c, v in item.get_postgres_representation().iteritems()}
        session = Session.create_from_uid(params['uid'])
        session.execute(versioning_queries.SQL_VERSIONING_SAVE_VERSION_LINK, params)

    def delete(self, item):
        uid = self.get_field_repr('uid', item.uid)
        _id = self.get_field_repr('id', item.id)

        session = Session.create_from_uid(uid)
        session.execute(versioning_queries.SQL_VERSIONING_DELETE_VERSION_LINK, {'uid': uid, 'id': _id})

    def count_by_uid(self, uid):
        uid = self.get_field_repr('uid', uid)
        session = Session.create_from_uid(uid)
        result_proxy = session.execute(versioning_queries.SQL_VERSIONING_COUNT_VERSION_LINKS_BY_UID, {'uid': uid})
        return result_proxy.fetchone()[0]

    def remove_by_uid(self, uid):
        uid = self.get_field_repr('uid', uid)
        session = Session.create_from_uid(uid)
        session.execute(versioning_queries.SQL_VERSIONING_DELETE_VERSION_LINKS_BY_UID, {'uid': uid})

    def fetch_by_uid(self, uid):
        uid = self.get_field_repr('uid', uid)
        session = Session.create_from_uid(uid)
        result_proxy = session.execute(versioning_queries.SQL_VERSIONING_GET_VERSION_LINKS_BY_UID, {'uid': uid})
        for doc in result_proxy:
            yield self.doc_to_item(doc)

    def bulk_insert(self, uid, items):
        uid = self.get_field_repr('uid', uid)
        session = Session.create_from_uid(uid)
        version_links_gen = BulkInsertReqGenerator(version_links, items)
        session.execute(
            version_links_gen.generate_tmpl(),
            version_links_gen.generate_values()
        )

    def delete_version_links_without_versions(self, shard_endpoint, batch_size):
        session = Session.create_from_shard_endpoint(shard_endpoint)
        session.execute(versioning_queries.SQL_VERSIONING_DELETE_VERSION_LINKS_WITHOUT_VERSIONS,
                        {'batch_size': batch_size})
