# -*- coding: utf-8 -*-

from mpfs.common import errors
from mpfs.common.errors.share import GroupNoPermit
from mpfs.core.address import Address, ResourceId
from mpfs.core.factory import get_resource_by_resource_id, get_resource
from mpfs.core.filesystem.resources.share import SharedResource
from mpfs.core.services.disk_service import zaberun
from mpfs.core.social.share.constants import SharedFolderRights
from mpfs.core.versioning.dao.version_data import VersionType
from mpfs.core.versioning.iteration_keys import VersioningIterationKey
from mpfs.core.versioning.logic.version_manager import ResourceVersionManager


def _generate_download_url_for(version, name):
    if version.type in (VersionType.binary, VersionType.current):
        return zaberun.generate_file_url(
            version.uid_created,
            version.file_stid,
            name,
            fsize=version.size,
            hid=version.dao_item.hid,
            md5=version.dao_item.md5,
            owner_uid=version.dao_item.uid,
        )
    return None


def versioning_get_checkpoints(req):
    """Получить только основные версии (несвёрнутые).

    Query string аргументы:
        * uid [обязательный]
        * resource_id [обязательный]
        * iteration_key - ключ для получения следующей пачки версий. Пустое значение - 1-ая пачка
    """
    resource_id = ResourceId.parse(req.resource_id)
    resource = get_resource_by_resource_id(req.uid, resource_id, enable_collections=('user_data',))
    iteration_key = VersioningIterationKey.parse(req.iteration_key)

    next_iteration_key, versions = ResourceVersionManager.get_checkpoint_versions(resource, iteration_key=iteration_key)

    return next_iteration_key, [(version, _generate_download_url_for(version, resource.name))
                                for version in versions]


def versioning_get_folded(req):
    """Получить свёрнутрые версии.

    Query string аргументы:
        * uid [обязательный]
        * resource_id [обязательный]
        * iteration_key [обязательный]
    """
    resource_id = ResourceId.parse(req.resource_id)
    resource = get_resource_by_resource_id(req.uid, resource_id, enable_collections=('user_data',))
    iteration_key = VersioningIterationKey.parse(req.iteration_key)
    if not iteration_key:
        raise errors.BadRequestError('Iteration key required')

    next_iteration_key, versions = ResourceVersionManager.get_folded_versions(resource, iteration_key)

    return next_iteration_key, [(version, _generate_download_url_for(version, resource.name))
                                for version in versions]


def versioning_restore(req):
    """Восстановить версию файла

    Query string аргументы:
        * uid [обязательный]
        * resource_id [обязательный]
        * version_id [обязательный] - id версии, которую надо восстановить
    """
    resource_id = ResourceId.parse(req.resource_id)
    resource = get_resource_by_resource_id(req.uid, resource_id, enable_collections=('user_data',))
    if (isinstance(resource, SharedResource) and
            not SharedFolderRights.is_rw(resource.meta['group']['rights'])):
        raise GroupNoPermit()
    return ResourceVersionManager.restore_version(req.uid, resource, req.version_id)


def versioning_save(req):
    """Сохранить версию как отдельный файл

    Query string аргументы:
        * uid [обязательный]
        * resource_id [обязательный]
        * version_id [обязательный] - id версии, которую сохранить как файл
    """
    resource_id = ResourceId.parse(req.resource_id)
    resource = get_resource_by_resource_id(req.uid, resource_id, enable_collections=('user_data',))
    if (isinstance(resource, SharedResource) and
            not SharedFolderRights.is_rw(resource.meta['group']['rights'])):
        raise GroupNoPermit()
    return ResourceVersionManager.save_version(req.uid, resource, req.version_id)


def versioning_get_version(req):
    """Получить одну версию

    Query string аргументы:
        * uid [обязательный]
        * resource_id или path [один из]
        * version_id [обязательный] - id версии, которую сохранить как файл
    """
    if req.path:
        address = Address.Make(req.uid, req.path)
        resource = get_resource(req.uid, address)
    elif req.resource_id:
        resource_id = ResourceId.parse(req.resource_id)
        resource = get_resource_by_resource_id(req.uid, resource_id, enable_collections=('user_data',))
    else:
        raise errors.BadRequestError()
    version = ResourceVersionManager.get_version(resource, req.version_id)

    return version, _generate_download_url_for(version, resource.name)
