# -*- coding: utf-8 -*-
import dateutil.parser

from mpfs.common.util import from_json, to_json
from mpfs.common.util.crypt import AesCbcCryptAgent, DEFAULT_SECRET


class VersioningIterationKey(object):
    """Итератор по версиям"""
    DEFAULT_PAGE_SIZE = 40
    _crypt_agent = AesCbcCryptAgent(DEFAULT_SECRET)

    def __init__(self, version_dt=None, limit=None, checkpoint_version_id=None):
        self.version_dt = version_dt
        self.checkpoint_version_id = checkpoint_version_id
        self.limit = limit or self.DEFAULT_PAGE_SIZE
        if self.limit <= 0:
            raise ValueError()

    def is_folded_iteration(self):
        return self.checkpoint_version_id is not None

    def get_next(self, next_item_dt):
        return self.__class__(version_dt=next_item_dt, limit=self.limit, checkpoint_version_id=self.checkpoint_version_id)

    @classmethod
    def first_page(cls, limit=None):
        return cls(limit=limit)

    @classmethod
    def parse(cls, raw_iteration_key):
        if not raw_iteration_key:
            return None
        raw_json = cls._crypt_agent.decrypt(raw_iteration_key)
        data = from_json(raw_json)
        if data['version_dt'] is not None:
            data['version_dt'] = dateutil.parser.parse(data['version_dt'])
        return cls(
            version_dt=data['version_dt'],
            limit=data['limit'],
            checkpoint_version_id=data['checkpoint_version_id']
        )

    def serialize(self):
        data = {
            'version_dt': self.version_dt.isoformat() if self.version_dt else None,
            'limit': self.limit,
            'checkpoint_version_id': self.checkpoint_version_id,
        }
        raw_json = to_json(data)
        raw_iteration_key = self._crypt_agent.encrypt(raw_json)
        return raw_iteration_key
