# -*- coding: utf-8 -*-
import datetime
import itertools
import uuid
import time

from mpfs.engine.process import get_cloud_req_id
from mpfs.common import errors
from mpfs.common.util import chunks2
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.core.versioning.iteration_keys import VersioningIterationKey
from mpfs.core.versioning.dao.version_data import VersionDataDAO, VersionDataDAOItem, VersionType
from mpfs.core.versioning.errors import NotBinaryVersion, VersioningTimeout
from mpfs.dao.fields import AntiVirusStatusField
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.resources.disk import MPFSFile


class VersionManager(object):
    version_data_dao = VersionDataDAO()

    @classmethod
    def fetch_expired_versions_on_shard(cls, shard_endpoint, limit):
        for dao_item in cls.version_data_dao.fetch_expired_versions_on_shard(shard_endpoint, limit):
            yield Version(dao_item)

    @classmethod
    def bulk_remove_versions_on_shard(cls, shard_endpoint, versions, put_stids_on_cleaning=True):
        if put_stids_on_cleaning:
            cls.put_versions_stids_to_deleted_stids(versions)
        cls.version_data_dao.bulk_delete_on_shard(shard_endpoint, [v.dao_item for v in versions])

    @classmethod
    def put_versions_stids_to_deleted_stids(cls, versions):
        """Вытащить stid-ы из версий и поставить их на чистку"""
        deleted_stids = []
        for version in versions:
            dao_item = version.dao_item
            if dao_item.type != VersionType.binary:
                continue
            deleted_stids.append(
                DeletedStid(stid=dao_item.file_stid, size=dao_item.size, stid_source=DeletedStidSources.VERSIONING)
            )
            for stid in (dao_item.preview_stid, dao_item.digest_stid):
                if stid:
                    deleted_stids.append(DeletedStid(stid=stid, stid_source=DeletedStidSources.VERSIONING))
        if deleted_stids:
            DeletedStid.controller.bulk_create(deleted_stids, get_size_from_storage=True)

    @classmethod
    def bulk_remove_versions(cls, versions, put_stids_on_cleaning=True):
        """Удалить пачку версий

        Позволяет удалять версии принадлежащие разным пользователям
        """
        if not versions:
            return
        versions.sort(key=lambda v: v.dao_item.uid)
        for uid, uid_versions in itertools.groupby(versions, key=lambda v: v.dao_item.uid):
            for versions_batch in chunks2(uid_versions, chunk_size=1000):
                if put_stids_on_cleaning:
                    cls.put_versions_stids_to_deleted_stids(versions_batch)
                cls.version_data_dao.bulk_delete(uid, [v.dao_item for v in versions_batch])


class Version(object):
    _dao = VersionDataDAO()

    def __init__(self, dao_item):
        if not isinstance(dao_item, VersionDataDAOItem):
            raise TypeError('`VersionDataDAOItem` expected. Got: %r' % dao_item)
        self.dao_item = dao_item

    @property
    def date_created(self):
        """Дата порождения версии

        Когда файл загрузили - это и будет дата порождения версии.
        Актуально только для бинарных версий, т.к. у остальных date_created ~= record_date_created
        Пример:
            Файл загрузили 2016 году и перезаписали в 2018. Тогда у версии будет:
            `date_created` - 2016 год
            `record_date_created` - 2018 год
        """
        return self.dao_item.date_created

    @property
    def record_date_created(self):
        """Когда был создан объект версии"""
        return self.dao_item.record_date_created

    @property
    def id(self):
        return self.dao_item.id

    @property
    def file_stid(self):
        return self.dao_item.file_stid

    @property
    def platform_created(self):
        return self.dao_item.platform_created

    @property
    def size(self):
        return self.dao_item.size

    @property
    def md5(self):
        return self.dao_item.md5

    @property
    def sha256(self):
        return self.dao_item.sha256

    @property
    def type(self):
        return self.dao_item.type

    @property
    def uid_created(self):
        return self.dao_item.uid_created

    @property
    def is_checkpoint(self):
        return self.dao_item.is_checkpoint

    def reset_checkpoint(self):
        self._dao.reset_checkpoint(self.dao_item.uid, self.dao_item.id)

    def save(self):
        self._dao.save(self.dao_item)

    def get_iteration_key_for_folded_items(self):
        if not self.dao_item.is_checkpoint or self.dao_item.folded_counter == 0:
            return
        return VersioningIterationKey(
            version_dt=self.date_created,
            checkpoint_version_id=self.id
        )

    def copy(self):
        new_dao_item = self.dao_item.copy()
        new_dao_item.id = uuid.uuid4().hex
        new_dao_item.record_date_created = datetime.datetime.now()
        new_dao_item.parent_version_id = None
        new_dao_item.uid = None
        new_dao_item.version_link_id = None
        new_dao_item.is_checkpoint = True
        new_dao_item.folded_counter = 0
        return self.__class__(new_dao_item)

    def can_be_restored(self):
        return self.type == VersionType.binary

    def get_resource_fields(self):
        if not self.can_be_restored():
            raise NotBinaryVersion()
        return {
            'source_uid': self.dao_item.uid_created,
            'source_platform': self.dao_item.platform_created,
            'md5': self.dao_item.md5,
            'sha256': self.dao_item.sha256,
            'size': self.dao_item.size,
            'hid': self.dao_item.hid,
            'file_mid': self.dao_item.file_stid,
            'pmid': self.dao_item.preview_stid,
            'digest_mid': self.dao_item.digest_stid,
            'video_info': self.dao_item.video_info,
            'etime': int(time.mktime(self.dao_item.exif_time.timetuple())) if self.dao_item.exif_time else None,
            'drweb': AntiVirusStatusField.convert_av_scan_status_to_num(self.dao_item.antivirus_status)
                        if self.dao_item.antivirus_status else None,
            'width': self.dao_item.width,
            'height': self.dao_item.height,
            'angle': self.dao_item.angle,
        }

    @classmethod
    def create_binary(cls, resource, date_to_remove):
        if not isinstance(resource, MPFSFile):
            raise errors.NotFile('Expected "MPFSFile". Got %r' % resource)

        dao_item = cls._fill_common_fields_on_create(date_to_remove)
        dao_item.type = VersionType.binary
        dao_item.uid_created = resource.source_uid
        dao_item.platform_created = resource.source_platform
        dao_item.date_created = datetime.datetime.fromtimestamp(int(resource.mtime))
        checksums = resource.get_checksums()
        dao_item.hid = checksums.hid
        dao_item.size = checksums.size
        dao_item.md5 = checksums.md5
        dao_item.sha256 = checksums.sha256
        dao_item.file_stid = resource.file_mid()
        dao_item.preview_stid = resource.preview_mid()
        dao_item.digest_stid = resource.digest_mid()
        dao_item.video_info = resource.get_video_info()
        dao_item.width = resource.get_width()
        dao_item.height = resource.get_height()
        dao_item.angle = resource.get_angle()
        # эти поля надо руками конвертить
        etime_ts = resource.meta.get('etime')
        dao_item.exif_time = datetime.datetime.fromtimestamp(int(etime_ts)) if etime_ts else None
        dao_item.antivirus_status = AntiVirusStatusField.convert_num_to_av_scan_status(resource.drweb())
        return cls(dao_item)

    @classmethod
    def create_fake(cls, uid_created, version_type, date_created, date_to_remove):
        dao_item = cls._fill_common_fields_on_create(date_to_remove)
        dao_item.type = version_type
        dao_item.date_created = date_created
        dao_item.uid_created = uid_created
        dao_item.platform_created = YcridParser.get_platform(get_cloud_req_id())
        dao_item.hid = None
        dao_item.size = None
        dao_item.md5 = None
        dao_item.sha256 = None
        dao_item.file_stid = None
        dao_item.preview_stid = None
        dao_item.digest_stid = None
        dao_item.video_info = None
        dao_item.exif_time = None
        dao_item.antivirus_status = None
        dao_item.width = None
        dao_item.height = None
        dao_item.angle = None
        return cls(dao_item)

    @classmethod
    def _fill_common_fields_on_create(cls, date_to_remove):
        dao_item = VersionDataDAOItem()
        dao_item.id = uuid.uuid4().hex
        dao_item.record_date_created = datetime.datetime.now()
        dao_item.date_to_remove = date_to_remove
        # эти поля могут меняться при сохранении в БД
        dao_item.uid = None
        dao_item.version_link_id = None
        dao_item.parent_version_id = None
        dao_item.is_checkpoint = True
        dao_item.folded_counter = 0
        return dao_item

    def __repr__(self):
        return '<%(class)s(%(id)s, %(uid)s, %(type)s, %(date_created)s, %(version_link_id)s, %(is_checkpoint)s)>' % {
            'class': self.__class__.__name__,
            'id': self.dao_item.id,
            'uid': self.dao_item.uid,
            'type': self.dao_item.type,
            'date_created': self.dao_item.date_created,
            'version_link_id': self.dao_item.version_link_id,
            'is_checkpoint': self.dao_item.is_checkpoint
        }
