# -*- coding: utf-8 -*-
import datetime
import uuid
import hashlib

from mpfs.common.util.rps_limiter import InMemoryRPSLimiter
from mpfs.core.address import ResourceId
from mpfs.engine.process import get_default_log
from mpfs.common.util import chunks2, pairwise
from mpfs.core.versioning.errors import (
    VersionLinkNotFound, VersionNotFound,
    VersionWrongTimes, VersionLinkIDsMissmatch
)
from mpfs.core.versioning.logic.version import Version, VersionManager
from mpfs.core.versioning.iteration_keys import VersioningIterationKey
from mpfs.core.versioning.dao.version_links import VersionLinkDAOItem, VersionLinkDAO
from mpfs.core.versioning.dao.version_data import VersionDataDAO, VersionType

default_log = get_default_log()


class VersionChain(object):
    iterate_versions_batch_size = 2000
    remove_versions_batch_size = 20
    remove_versions_rps_limiter = InMemoryRPSLimiter(50)
    checkpoint_dt_delta = datetime.timedelta(minutes=10)
    one_ms_dt_delta = datetime.timedelta(microseconds=1000)
    allow_to_add_dt_delta = datetime.timedelta(minutes=-1)
    version_link_dao = VersionLinkDAO()
    version_data_dao = VersionDataDAO()

    def __init__(self, version_link_dao_item):
        if not isinstance(version_link_dao_item, VersionLinkDAOItem):
            raise TypeError('`VersionLinkDAOItem` expected. Got: %r' % version_link_dao_item)
        self.dao_item = version_link_dao_item
        self._is_new = False

    @property
    def is_new(self):
        return self._is_new

    @property
    def resource_id(self):
        return ResourceId(self.dao_item.uid, self.dao_item.file_id)

    @classmethod
    def get_by_phantom_address(cls, uid, address):
        path_hash = cls._path_to_hash(address.path)
        version_link_dao_item = cls.version_link_dao.get_by_uid_and_path_hash(address.uid, path_hash)
        if not version_link_dao_item:
            raise VersionLinkNotFound()
        return cls(version_link_dao_item)

    def set_phantom_path(self, disk_path):
        self.dao_item.disk_path = disk_path
        self.dao_item.disk_path_hash = self._path_to_hash(disk_path)
        self.version_link_dao.save(self.dao_item)

    def reset_phantom_path(self):
        self.dao_item.disk_path = None
        self.dao_item.disk_path_hash = None
        self.version_link_dao.save(self.dao_item)

    @classmethod
    def ensure(cls, resource_id):
        try:
            version_chain = cls.get_by_resource_id(resource_id)
        except VersionLinkNotFound:
            dao_item = VersionLinkDAOItem()
            dao_item.id = uuid.uuid4().hex
            dao_item.uid = resource_id.uid
            dao_item.file_id = resource_id.file_id
            dao_item.date_created = datetime.datetime.now()
            dao_item.disk_path = None
            dao_item.disk_path_hash = None
            cls.version_link_dao.save(dao_item)
            version_chain = cls(dao_item)
            version_chain._is_new = True
        return version_chain

    @classmethod
    def get_by_resource_id(cls, resource_id):
        version_link_dao_item = cls.version_link_dao.get_by_resource_id(resource_id)
        if version_link_dao_item is None:
            raise VersionLinkNotFound()
        return cls(version_link_dao_item)

    def remove(self):
        """Удалить version_link и все связанные версии"""
        for versions_batch in chunks2(self.iterate_over_all_versions(),
                                      chunk_size=self.remove_versions_batch_size):
            self.remove_versions_rps_limiter.block_until_allowed(self.remove_versions_batch_size)
            self.remove_versions(versions_batch)
        self.version_link_dao.delete(self.dao_item)

    def remove_versions(self, versions, put_stids_on_cleaning=True):
        """Удалить пачку связанных с version_link версий"""
        if not versions:
            return
        self_id = self.dao_item.id
        for version in versions:
            if version.dao_item.version_link_id != self_id:
                raise VersionLinkIDsMissmatch('Attemp to remove not binded version')
        VersionManager.bulk_remove_versions(versions, put_stids_on_cleaning=put_stids_on_cleaning)

    def append_version(self, version):
        """Добавить одну версию"""
        self.append_versions([version])

    def append_versions(self, versions):
        """Добавить новые версии (будут в начале выдачи)

        Версии должны идти по возрастанию
        """
        if not versions:
            return

        try:
            latest_version = self.get_latest_version()
        except VersionNotFound:
            self._link_versions_list(versions)
        else:
            self._link_versions_list([latest_version] + versions)
            if latest_version.dao_item.is_checkpoint is False:
                latest_version.reset_checkpoint()
        self.version_data_dao.bulk_insert(self.dao_item.uid, [v.dao_item for v in versions])

    def appendleft_versions(self, versions, ignore_dt_missmatch_versions=False):
        """Добавить ранние версии (будут в конце выдачи)

        Версии должны идти по возрастанию
        :param ignore_dt_missmatch_versions: если добавляется версии более новые чем хвостовая, то откинуть их.
                                             Нужно для безопасного move версий.
        """
        if not versions:
            return

        try:
            earliest_version = self.get_earliest_version()
        except VersionNotFound:
            earliest_version = None

        if ignore_dt_missmatch_versions and earliest_version:
            bound_dt = earliest_version.date_created
            versions = filter(lambda v: v.date_created < bound_dt, versions)
            if not versions:
                return

        self._link_versions_list(versions)

        if earliest_version:
            self._link_prev_version(earliest_version, versions[-1])
            earliest_version.save()
        self.version_data_dao.bulk_insert(self.dao_item.uid, [v.dao_item for v in versions])

    def truncate(self, limit):
        """
        Удаление старых версий

        Работает по следующей логике:
        1. сначала удаляет обычные версии (не чекпойнты) начиная с самых старых версий
        2. если не осталось обычных версий, то начинает удалять чекпойнты с самых старых версий

        :param limit: сколько версий оставить
        :return: None
        """
        if limit < 0:
            limit = 0
        total_versions = self.count_versions()
        if total_versions <= limit:
            return

        num_versions_to_remove = total_versions - limit
        skipped_checkpoints = 0
        batch_size = 1000
        # сначала удаляем все обычные версии (не чекпойнты)
        while True:
            versions_to_remove = []
            version_dao_items = self.version_data_dao.get_all_ascending(self.dao_item.uid, self.dao_item.id, skipped_checkpoints, batch_size)
            for version_dao_item in version_dao_items:
                version = self._convert_to_version(version_dao_item)
                if version.is_checkpoint:
                    if versions_to_remove:
                        self.remove_versions(versions_to_remove)
                        versions_to_remove = []
                    else:
                        skipped_checkpoints += 1

                    if version.dao_item.folded_counter != 0:
                        # у чекпойнта ставим правильный счетчик
                        version.dao_item.folded_counter = 0
                        version.save()
                else:
                    versions_to_remove.append(version)
                    num_versions_to_remove -= 1

                if num_versions_to_remove <= 0:
                    # почистили нужное кол-во версий - выход
                    if versions_to_remove:
                        self.remove_versions(versions_to_remove)
                    return

            if versions_to_remove:
                self.remove_versions(versions_to_remove)

            if len(version_dao_items) < batch_size:
                # прошлись по всем версиям до конца
                # остались одни чекпойнты - чистим их (см. ниже)
                break

        # все обчные версии удалены, начинаем удалять чекпойнты с конца
        while True:
            versions_to_remove = []
            version_dao_items = self.version_data_dao.get_all_ascending(self.dao_item.uid, self.dao_item.id, 0, batch_size)
            for version_dao_item in version_dao_items:
                version = self._convert_to_version(version_dao_item)
                versions_to_remove.append(version)
                num_versions_to_remove -= 1

                if num_versions_to_remove <= 0:
                    # почистили нужное кол-во версий - выход
                    if versions_to_remove:
                        self.remove_versions(versions_to_remove)
                    return

            if versions_to_remove:
                self.remove_versions(versions_to_remove)

            if len(version_dao_items) < batch_size:
                # последняя итерация - выходим
                return

    def count_versions(self):
        return self.version_data_dao.count_version_link_versions(self.dao_item.uid, self.dao_item.id)

    def count_versions_greater_than_dt(self, border_dt):
        return self.version_data_dao.count_version_link_versions_greater_than_dt(self.dao_item.uid, self.dao_item.id, border_dt)

    def get_version_by_id(self, version_id):
        version_dao_item = self.version_data_dao.get_by_id(self.dao_item.uid, self.dao_item.id, version_id)
        return self._convert_to_version(version_dao_item)

    def get_earliest_version(self):
        version_dao_item = self.version_data_dao.get_earliest_version(self.dao_item.uid, self.dao_item.id)
        return self._convert_to_version(version_dao_item)

    def get_latest_version(self):
        version_dao_item = self.version_data_dao.get_latest_version(self.dao_item.uid, self.dao_item.id)
        return self._convert_to_version(version_dao_item)

    def iterate_over_all_versions(self):
        """Проитерироваться по всем версиям"""
        iteration_key = VersioningIterationKey.first_page(limit=self.iterate_versions_batch_size)
        while True:
            iteration_key, versions = self.get_all_versions(iteration_key)
            for version in versions:
                yield version
            if not iteration_key:
                break

    def get_all_versions(self, iteration_key):
        """Получить все версии

        :rtype: tuple(VersioningIterationKey, list(Version))
        """
        return self._get_versions(iteration_key, False)

    def get_checkpoint_versions(self, iteration_key):
        """Получить только основные (checkpoint) версии

        :rtype: tuple(VersioningIterationKey, list(Version))
        """
        return self._get_versions(iteration_key, True)

    def get_folded_versions(self, iteration_key):
        """Получить свернутые версии (между двумя чекпойнтами)

        :rtype: tuple(VersioningIterationKey, list(Version))
        """
        if not iteration_key.is_folded_iteration():
            VersionNotFound()
        next_iteration_key, versions = self.get_all_versions(iteration_key)
        only_folded_versions = []
        for version in versions:
            if version.dao_item.is_checkpoint:
                if version.dao_item.id == iteration_key.checkpoint_version_id:
                    continue
                else:
                    next_iteration_key = None
                    break
            only_folded_versions.append(version)
        return next_iteration_key, only_folded_versions

    def _get_versions(self, iteration_key, only_checkpoints):
        if only_checkpoints:
            dao_items_getter = self.version_data_dao.get_checkpoints
        else:
            dao_items_getter = self.version_data_dao.get_all
        version_data_dao_items = dao_items_getter(
            self.dao_item.uid, self.dao_item.id,
            iteration_key.version_dt, iteration_key.limit
        )

        if len(version_data_dao_items) < iteration_key.limit:
            next_iteration_key = None
        else:
            last_item = version_data_dao_items[-1]
            next_iteration_key = iteration_key.get_next(last_item.date_created)
        return next_iteration_key, [Version(v) for v in version_data_dao_items]

    def _convert_to_version(self, version_dao_item):
        if version_dao_item:
            return Version(version_dao_item)
        raise VersionNotFound()

    @staticmethod
    def _path_to_hash(path):
        if isinstance(path, unicode):
            path = path.encode('utf-8')
        return hashlib.md5(path).hexdigest()

    def _link_versions_list(self, versions):
        self._bind_version(versions[0])
        for prev_ver, ver in pairwise(versions):
            self._bind_version(ver)
            self._link_prev_version(ver, prev_ver)

    def _bind_version(self, version):
        version.dao_item.uid = self.dao_item.uid
        version.dao_item.version_link_id = self.dao_item.id

    def _link_prev_version(self, version, prev_version):
        if not(version.dao_item.version_link_id == prev_version.dao_item.version_link_id == self.dao_item.id):
            raise VersionLinkIDsMissmatch()

        # проверяем, что версии упорядочены по времени
        versions_delta = version.dao_item.date_created - prev_version.dao_item.date_created
        if versions_delta <= self.one_ms_dt_delta:
            if versions_delta < self.allow_to_add_dt_delta:
                msg = "Version date_created missmatch. %s (cur) <= %s (prev) (%s). Cur: %s Prev: %s" % (
                    version.dao_item.date_created,
                    prev_version.dao_item.date_created,
                    versions_delta,
                    version, prev_version,
                )
                raise VersionWrongTimes(msg)
            else:
                # если небольшая разница, то делаем такой хак, чтобы версии шли одна за другой.
                version.dao_item.date_created = prev_version.dao_item.date_created + self.one_ms_dt_delta

        version.dao_item.is_checkpoint = True
        version.dao_item.parent_version_id = prev_version.dao_item.id
        if versions_delta > self.checkpoint_dt_delta:
            # новый чекпойнт
            version.dao_item.folded_counter = 0
        else:
            # цепочка продолжается
            version.dao_item.folded_counter = prev_version.dao_item.folded_counter + 1
            # старый чекпойт больше не чекпойнт
            prev_version.dao_item.is_checkpoint = False
            # prev_version.dao_item.folded_counter = 0

    def __repr__(self):
        return '<%(class)s(%(id)s, %(uid)s:%(file_id)s, %(date_created)s)>' % {
            'class': self.__class__.__name__,
            'id': self.dao_item.id,
            'uid': self.dao_item.uid,
            'file_id': self.dao_item.file_id,
            'date_created': self.dao_item.date_created,
        }
