# -*- coding: utf-8 -*-
import datetime

from mpfs.core.db_rps_limiter import InDBRPSLimiter
from mpfs.engine.process import get_cloud_req_id, get_error_log, get_default_log
from mpfs.core.address import ResourceId
from mpfs.config import settings
from mpfs.common.errors import ResourceNotFound
from mpfs.common.util import chunks2, to_json
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.filesystem.resources.disk import MPFSFile
from mpfs.core.versioning.errors import VersionLinkNotFound, NotBinaryVersion
from mpfs.core.versioning.dao.version_data import VersionType
from mpfs.core.versioning.dao.version_links import VersionLinkDAO
from mpfs.core.versioning.logic.version import Version
from mpfs.core.versioning.logic.version_chain import VersionChain
from mpfs.core.versioning.iteration_keys import VersioningIterationKey
from mpfs.core.factory import get_resource_by_resource_id
from mpfs.core.user.constants import VERSIONING_SAVE_VERSION_SUFFIX
from mpfs.core.queue import mpfs_queue

default_log = get_default_log()
error_log = get_error_log()

VERSIONING_TOGGLES_GLOBAL = settings.versioning['toggles']['global']
VERSIONING_SUPPRESS_HOOKS_EXCEPTION = settings.versioning['suppress_hooks_exception']
VERSIONING_SAVE_VERSION_DT_FORMAT = '%d.%m.%Y %H:%M:%S'


def async_remove_versions_by_full_index(full_index, owner_uid=None):
    """Из full_index генерит таски на удаление версий"""
    resource_ids = []
    for raw_item in full_index.itervalues():
        if raw_item.get('type') != 'file':
            continue
        try:
            resource_id = ResourceId(owner_uid or raw_item['uid'], raw_item['meta']['file_id'])
        except Exception:
            pass
        else:
            resource_ids.append(resource_id)
    ResourceVersionManager.async_bulk_remove_versions(resource_ids)


class ResourceVersionManager(object):
    """Менеджер работы с версиями. Определяется для ресурса, с версиями которого собираемся работать."""
    version_live_time = datetime.timedelta(days=90)
    version_visibility_period = {
        'common': datetime.timedelta(days=14),
        'extended': datetime.timedelta(days=90),
    }

    # временной лимит на кол-во создаваемых версий на пользователя
    limiter_per_user = InDBRPSLimiter('create_versions')
    not_limited_extensions = {'docx', 'doc', 'xls', 'xlsx', 'odt', 'odp', 'ods'}

    # временной лимит на кол-во создаваемых версий на файл
    limit_delta = datetime.timedelta(minutes=1)
    limit_value = 10

    truncate_limit = 4400
    truncate_activation_limit = truncate_limit + 50

    @classmethod
    def add_binary_version(cls, resource):
        """Добавляет бинарную версию."""
        if not cls.is_versionable(resource):
            return

        try:
            version = Version.create_binary(resource, cls._get_version_remove_date(resource))
            version_chain = VersionChain.ensure(resource.resource_id)
            if (not cls._is_limit_exceed_per_user(resource) and
                    not cls._is_limit_exceed_per_file(version_chain)):
                version_chain.append_version(version)
            cls._put_truncate_task_if_required(version_chain)
            return version
        except Exception:
            cls._suppress_except_block()

    @classmethod
    def trash_append(cls, modify_uid, disk_resource, trash_resource):
        """Хук версионирования при удалении в корзину.

        Добавляет 2 версии:
            * Бинарную версию удаляемого ресурса
            * "Удалено"
        """
        if not cls.is_versionable(disk_resource):
            return

        try:
            version_chain = VersionChain.ensure(disk_resource.resource_id)
            date_to_remove = cls._get_version_remove_date(disk_resource)
            binary_version = Version.create_binary(disk_resource, date_to_remove)
            trashed_version = Version.create_fake(
                modify_uid,
                VersionType.trashed,
                datetime.datetime.fromtimestamp(int(trash_resource.mtime)),
                date_to_remove
            )

            if (not cls._is_limit_exceed_per_user(disk_resource, versions_num=2) and
                    not cls._is_limit_exceed_per_file(version_chain)):
                version_chain.append_versions([binary_version, trashed_version])

            disk_path = trash_resource.original_id
            if disk_path:
                version_chain.set_phantom_path(disk_path)
            cls._put_truncate_task_if_required(version_chain)
        except Exception:
            cls._suppress_except_block()

    @classmethod
    def restore_from_trash(cls, modify_uid, restored_resource):
        """Хук версионирования при восстановлении из корзины."""
        if not cls.is_versionable(restored_resource):
            return

        try:
            version_chain = VersionChain.ensure(restored_resource.resource_id)
            restored_version = Version.create_fake(
                modify_uid,
                VersionType.restored,
                datetime.datetime.fromtimestamp(int(restored_resource.mtime)),
                cls._get_version_remove_date(restored_resource)
            )
            if (not cls._is_limit_exceed_per_user(restored_resource) and
                    not cls._is_limit_exceed_per_file(version_chain)):
                version_chain.append_version(restored_version)
            version_chain.reset_phantom_path()
            cls._put_truncate_task_if_required(version_chain)
        except Exception:
            cls._suppress_except_block()

    @classmethod
    def bind_versions_from_trash(cls, disk_resource):
        """Хук версионирования при загрузке файла

        Подвязвает к ресурсу версии через механизм "призрачных" путей.
        Если найден version_link для соотвествующего path, то копируются версии файла из корзины
        """
        if not cls.is_versionable(disk_resource):
            return

        try:
            try:
                trashed_version_chain = VersionChain.get_by_phantom_address(disk_resource.owner_uid, disk_resource.address)
            except (ResourceNotFound, VersionLinkNotFound):
                return

            version_chain = VersionChain.ensure(disk_resource.resource_id)
            copied_versions_iter = (v.copy() for v in trashed_version_chain.iterate_over_all_versions())
            for versions_batch in chunks2(copied_versions_iter, chunk_size=100):
                if cls._is_limit_exceed_per_user(disk_resource, versions_num=len(versions_batch)):
                    break

                # получаем версии по убыванию, а добавлять надо по возрастанию
                versions_batch.sort(key=lambda x: x.date_created)
                version_chain.appendleft_versions(versions_batch)
            cls._put_truncate_task_if_required(version_chain)
        except Exception:
            cls._suppress_except_block()

    @classmethod
    def is_versionable(cls, resource):
        """Является ли ресурс версионируемым"""
        if not isinstance(resource, MPFSFile):
            return False
        if not VERSIONING_TOGGLES_GLOBAL:
            return False
        return True

    @classmethod
    def get_checkpoint_versions(cls, resource, iteration_key=None):
        return cls._get_versions(VersionChain.get_checkpoint_versions, resource, iteration_key=iteration_key)

    @classmethod
    def get_all_versions(cls, resource, iteration_key=None):
        return cls._get_versions(VersionChain.get_all_versions, resource, iteration_key=iteration_key)

    @classmethod
    def get_folded_versions(cls, resource, iteration_key):
        version_chain = VersionChain.get_by_resource_id(resource.resource_id)
        result = version_chain.get_folded_versions(iteration_key)
        return cls._filter_not_visible_versions(resource, result)

    @classmethod
    def get_version(cls, resource, version_id):
        version_chain = VersionChain.get_by_resource_id(resource.resource_id)
        return version_chain.get_version_by_id(version_id)

    @classmethod
    def restore_version(cls, modify_uid, resource, version_id):
        """Применить версию к ресурсу"""
        version_chain = VersionChain.get_by_resource_id(resource.resource_id)
        version_to_restore = version_chain.get_version_by_id(version_id)
        if not version_to_restore.can_be_restored():
            raise NotBinaryVersion()

        LockHelper.check(resource)
        resource_fields = version_to_restore.get_resource_fields()
        resource_fields.pop('hid', None)
        resource_fields['source_uid'] = modify_uid
        resource_fields['source_platform'] = YcridParser.get_platform(get_cloud_req_id())

        cur_version = Version.create_binary(resource, cls._get_version_remove_date(resource))
        version_chain.append_version(cur_version)

        changes = {}
        deleted = []
        for k, v in resource_fields.iteritems():
            if v is None:
                deleted.append(k)
            else:
                changes[k] = v

        resource.setprop(changes, deleted)
        return resource

    @classmethod
    def save_version(cls, modify_uid, resource, version_id):
        """Сохранить версию как отдельный файл"""
        version_chain = VersionChain.get_by_resource_id(resource.resource_id)
        version = version_chain.get_version_by_id(version_id)
        if not version.can_be_restored():
            raise NotBinaryVersion()

        LockHelper.check(resource)
        from mpfs.core.bus import Bus
        from mpfs.core.user.base import User
        fs = Bus()
        locale = User(modify_uid).get_supported_locale()
        formated_dt = version.dao_item.date_created.strftime(VERSIONING_SAVE_VERSION_DT_FORMAT)
        version_suffix = VERSIONING_SAVE_VERSION_SUFFIX[locale] % formated_dt
        tgt_address = resource.visible_address.add_suffix(version_suffix)
        tgt_address = fs.autosuffix_address(tgt_address)

        resource_fields = version.get_resource_fields()
        resource_fields.pop('hid', None)
        resource_fields['source_uid'] = modify_uid
        resource_fields['source_platform'] = YcridParser.get_platform(get_cloud_req_id())
        resource_fields['mimetype'] = resource.mimetype
        for k, v in resource_fields.items():
            if v is None:
                resource_fields.pop(k)
        return fs.mkfile(modify_uid, tgt_address.id, notify=True, data=resource_fields)

    @classmethod
    def move_versions(cls, src_resource_id, dst_resource_id):
        """Переместить версии от одного ресурса другому"""
        if not VERSIONING_TOGGLES_GLOBAL:
            return

        try:
            try:
                src_version_chain = VersionChain.get_by_resource_id(src_resource_id)
            except VersionLinkNotFound:
                return

            dst_version_chain = VersionChain.ensure(dst_resource_id)
            for versions_batch in chunks2(src_version_chain.iterate_over_all_versions(), chunk_size=1000):
                copied_versions = [v.copy() for v in versions_batch]
                # append требует отсортированные по возрастанию date_created версии
                copied_versions.sort(key=lambda x: x.date_created)
                dst_version_chain.appendleft_versions(copied_versions, ignore_dt_missmatch_versions=True)
                src_version_chain.remove_versions(versions_batch, put_stids_on_cleaning=False)
            src_version_chain.remove()
        except Exception:
            cls._suppress_except_block()

    @classmethod
    def async_bulk_remove_versions(cls, resource_ids):
        if not resource_ids:
            return
        if not VERSIONING_TOGGLES_GLOBAL:
            return
        for batch in chunks2(resource_ids, chunk_size=150000):
            mpfs_queue.put({'raw_resource_id': [r.serialize() for r in batch]}, 'remove_versions')

    @classmethod
    def async_remove_versions(cls, resource_id):
        if not VERSIONING_TOGGLES_GLOBAL:
            return
        cls.async_bulk_remove_versions([resource_id])

    @classmethod
    def remove_versions(cls, resource_id, check_resource_not_exist=True):
        if not VERSIONING_TOGGLES_GLOBAL:
            return
        try:
            version_chain = VersionChain.get_by_resource_id(resource_id)
        except VersionLinkNotFound:
            return

        if check_resource_not_exist:
            try:
                get_resource_by_resource_id(
                    resource_id.uid,
                    resource_id,
                    enable_collections=('user_data', 'trash')
                )
            except ResourceNotFound:
                pass
            else:
                default_log.warn('Resource "%s" exists. Dont remove related versions', resource_id.serialize())
                return

        version_chain.remove()

    @classmethod
    def fetch_all_version_chains(cls, uid):
        for version_link_dao_item in VersionLinkDAO().fetch_by_uid(uid):
            yield VersionChain(version_link_dao_item)

    @classmethod
    def _get_versions(cls, versions_getting_method, resource, iteration_key=None):
        if iteration_key:
            version_chain = VersionChain.get_by_resource_id(resource.resource_id)
            result = versions_getting_method(version_chain, iteration_key)
        else:
            # для первой страницы добавляем текущую версию (сам файл)
            current_version = Version.create_binary(resource, cls._get_version_remove_date(resource))
            current_version.dao_item.is_checkpoint = False
            current_version.dao_item.type = VersionType.current
            try:
                version_chain = VersionChain.get_by_resource_id(resource.resource_id)
            except VersionLinkNotFound:
                result = None, [current_version]
            else:
                iteration_key = VersioningIterationKey.first_page()
                next_iteration_key, versions = versions_getting_method(version_chain, iteration_key)
                result = next_iteration_key, [current_version] + versions

        return cls._filter_not_visible_versions(resource, result)

    @classmethod
    def _get_version_remove_date(cls, resource):
        return datetime.datetime.now() + cls.version_live_time

    @classmethod
    def _filter_not_visible_versions(cls, resource, resp_tuple):
        _, versions = resp_tuple

        from mpfs.core.user.base import User
        user = User(resource.owner_uid)
        visibility_border_dt = datetime.datetime.now()
        if user.versioning_extended_period_enabled:
            visibility_border_dt -= cls.version_visibility_period['extended']
        else:
            visibility_border_dt -= cls.version_visibility_period['common']

        i = 0
        for i, v in enumerate(versions):
            if v.record_date_created and v.record_date_created < visibility_border_dt:
                break
        else:
            return resp_tuple
        return None, versions[:i]

    @staticmethod
    def _suppress_except_block():
        if VERSIONING_SUPPRESS_HOOKS_EXCEPTION:
            error_log.exception('VERSIONING_SUPPRESS_HOOKS_EXCEPTION: %s', VERSIONING_SUPPRESS_HOOKS_EXCEPTION)
        else:
            raise

    @classmethod
    def _is_limit_exceed_per_file(cls, version_chain):
        if version_chain.is_new:
            return False
        now_dt = datetime.datetime.now()
        cur_value = version_chain.count_versions_greater_than_dt(now_dt - cls.limit_delta)
        if cur_value > cls.limit_value:
            default_log.info('Limit exceed per file. %s', version_chain.resource_id.serialize())
            return True
        return False

    @classmethod
    def _is_limit_exceed_per_user(cls, resource, versions_num=1):
        if resource.address.ext.lower() in cls.not_limited_extensions:
            return False
        if not cls.limiter_per_user.check_limit_and_increment_counter(resource.owner_uid, requests_num=versions_num):
            default_log.info('Limit exceed per user. %s', resource.resource_id.serialize())
            return True
        return False

    @classmethod
    def _put_truncate_task_if_required(cls, version_chain):
        if version_chain.is_new:
            return
        total_versions = version_chain.count_versions()
        if total_versions <= cls.truncate_activation_limit:
            return
        raw_resource_id = version_chain.resource_id.serialize()
        mpfs_queue.put(
            {'raw_resource_id': raw_resource_id , 'truncate_limit': cls.truncate_limit},
            'truncate_versions',
            deduplication_id=raw_resource_id
        )
