# -*- coding: utf-8 -*-
import time
import mpfs.engine.process
import mpfs.common.errors as errors
from mpfs.config import settings
from mpfs.core.operations.base import Operation
from mpfs.core.wake_up.session import SessionId
from mpfs.core.services.push_service import xiva_send, xiva_subscribe
from mpfs.common.util import hashed, to_json
from mpfs.common.errors import DeviceSubscriptionNotFound

WAKE_UP_PUSH_INTERVAL = settings.wake_up['push_interval']
WAKE_UP_TTL = settings.wake_up['ttl']
WAKE_UP_PUSH_RETRIES_COUNT = settings.wake_up['push_retries_count']

default_log = mpfs.engine.process.get_default_log()


class WakeUpOperation(Operation):
    type = 'wake_up'
    subtype = 'send_push'
    task_name = 'operation_service'

    @staticmethod
    def get_xiva_subscribtion_id(uid, device_id):
        xiva_response = xiva_subscribe.subscriptions_list(uid)
        normalized_device_id = device_id.replace('-', '').lower()
        subscription_id = None
        for each in xiva_response:
            if each.uuid == normalized_device_id:
                subscription_id = each.xiva_id
                break
        return subscription_id

    @classmethod
    def Create(cls, uid, odata, **kw):
        device_id = odata['device_id']
        subscription_id = cls.get_xiva_subscribtion_id(uid, device_id)
        if not subscription_id:
            raise DeviceSubscriptionNotFound('No subscription id found for device %s' % device_id)
        odata['subscription_id'] = subscription_id
        odata['wake_up_start_ts'] = time.time()
        operation = super(WakeUpOperation, cls).Create(uid, odata, **kw)
        return operation

    def __init__(self, *args, **kwargs):
        super(WakeUpOperation, self).__init__(*args, **kwargs)
        self.session_id = SessionId(self.uid, self.id).serialize()
        self.xiva_send = xiva_send
        self.xiva_subscribe = xiva_subscribe

    def get_uniq_id(self):
        return hashed("%s:%s" % (self.uid, self.data['device_id']))

    def update_start_ts(self):
        self.data['wake_up_start_ts'] = time.time()
        self.save()

    def _process(self, *args, **kwargs):
        uid = self.uid
        subscription_id = self.data['subscription_id']

        payload = to_json({'root': {'session_id': self.session_id, 'tag': 'wake_up'}})

        # only one subscription id in this operation
        subscription = [{uid: subscription_id}]
        for _ in xrange(WAKE_UP_PUSH_RETRIES_COUNT):
            try:
                push_status = self.xiva_send.batch_send(subscription, 'wake_up', payload)
            except errors.XivaError:
                pass
            else:
                if push_status[0].is_success():
                    break
        else:
            default_log.info('%s attempts of sending push notification failed.' % WAKE_UP_PUSH_RETRIES_COUNT)

        if int(time.time()) > self.data['wake_up_start_ts'] + WAKE_UP_TTL:
            self.set_completed()
        else:
            wake_up_interval = self.data.get('interval', 0)
            self.reenque(max(WAKE_UP_PUSH_INTERVAL, wake_up_interval))
