# -*- coding: utf-8 -*-
import time

from collections import defaultdict

from mpfs.core.queue import mpfs_queue
from mpfs.core.user.base import User
import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util.datetime_util import next_datetime_with_hour
from mpfs.common.errors.billing import BillingProductIsSingletone
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.billing.api import (
    service_create,
    service_delete,
)
from mpfs.core.bus import Bus
from mpfs.core.services.passport_service import passport
from mpfs.core.services.staff_service import StaffService
from mpfs.core.user.constants import YATEAM_DIR_PATH
from mpfs.core.user.dao.user import UserDAO

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

YATEAM_ENABLE_FOR_ALL = settings.yateam['enable_for_all']
YATEAM_ENABLED_YATEAM_UIDS = settings.yateam['enabled_yateam_uids']
YATEAM_REMOVE_FOLDER_DELAYED = settings.yateam['remove_folder']['delayed']
YATEAM_REMOVE_FOLDER_DELAYED_START_HOUR = settings.yateam['remove_folder']['delayed_start_hour']
YATEAM_RECREATE_FOLDER_DELAY = settings.yateam['recreate_folder']['delay']


def is_yateam_root(path):
    return path == YATEAM_DIR_PATH


def is_yateam_subtree(path):
    return is_yateam_root(path) or path.startswith(YATEAM_DIR_PATH + '/')


def resync_yateam_user(yateam_uid):
    """Проверить информацию о yateam_uid в базе и на Стаффе, сделать вывод об изменениях
    в ятимности. При необходимости отреагировать на изменения.
    """
    log.info('Resyncing yateam uid %s' % yateam_uid)
    try:
        staff_user = StaffService().get_user_info(yateam_uid)
    except errors.StaffUserNotFound:
        log.info('No user in staff with yateam uid %s' % yateam_uid)
        staff_user = None

    user_items = UserDAO().get_users_by_yateam_uid(staff_user.yateam_uid)
    users = [User(x.uid) for x in user_items]
    _resync(users, staff_user)


def resync_user(uid):
    log.info('Resyncing uid %s' % uid)

    all_staff_user_infos = StaffService().get_all_user_infos()
    staff_user_infos = [x for x in all_staff_user_infos if x.uid == uid]
    log.info('Found %s yateam_uids for uid %s' % ([x.yateam_uid for x in staff_user_infos], uid))
    if len(staff_user_infos) == 1:
        staff_user = staff_user_infos[0]
        user_items = UserDAO().get_users_by_yateam_uid(staff_user.yateam_uid)
    else:
        staff_user = None
        user_items = []

    uids = {x.uid for x in user_items} | {uid}
    users = [User(x) for x in uids]
    _resync(users, staff_user)


def _resync(users, staff_user):
    for user in users:
        if staff_user is None:
            log.info('Resetting yateam for user %s' % user.uid)
            reset_yateam(user.uid)
            continue

        if not staff_user.uid:
            log.info('Staff user has no linked uid, resetting yateam for user %s' % user.uid)
            reset_yateam(user.uid)
            continue

        if staff_user.uid != user.uid:
            log.info('Staff user has linked uid %s, resetting yateam for user %s' % (staff_user.uid, user.uid))
            reset_yateam(user.uid)
            continue

        if staff_user.is_dismissed:
            log.info('Staff user is dismissed, removing NDA folder for user %s' % user.uid)
            user.reset_yateam()
            _remove_yateam_dir(user.uid, staff_user.yateam_uid)

        userinfo = passport.userinfo(uid=user.uid)
        if not userinfo.get('is_2fa_enabled', False):
            log.info('User does not have 2FA, removing NDA folder for user %s' % user.uid)
            _remove_yateam_dir(user.uid, staff_user.yateam_uid)

    if staff_user and staff_user.uid and not staff_user.is_dismissed:
        log.info('Staff user has linked uid %s, making yateam' % staff_user.uid)
        make_yateam(staff_user.uid, staff_user.yateam_uid)


def resync_yateam_users():
    staff_users = StaffService().get_all_user_infos()
    user_items = UserDAO().fetch_yateam_users()

    grouped_by_yateam_uid = defaultdict(lambda: {'users': []})

    for staff_user in staff_users:
        yateam_uid = staff_user.yateam_uid
        grouped_by_yateam_uid[yateam_uid]['staff_user'] = staff_user

    for user_item in user_items:
        user = User(user_item.uid)
        yateam_uid = user.yateam_uid
        grouped_by_yateam_uid[yateam_uid]['users'].append(user)

    for yateam_uid, value in grouped_by_yateam_uid.iteritems():
        if YATEAM_ENABLE_FOR_ALL or yateam_uid in YATEAM_ENABLED_YATEAM_UIDS:
            staff_user = value.get('staff_user')
            users = value.get('users')
            _resync(users, staff_user)


def make_yateam(uid, yateam_uid):
    try:
        user = User(uid)
    except errors.StorageInitUser:
        log.info('User %s doesn\'t exist. Can\'t make YaTeam directory' % uid)
        return

    user.make_yateam(yateam_uid)

    # не создаём папку всем до релиза
    if YATEAM_ENABLE_FOR_ALL or yateam_uid in YATEAM_ENABLED_YATEAM_UIDS:
        if user_has_nda_rights(user):
            _create_yateam_dir(uid)

    try:
        service_create(uid, 'yandex_staff', 'bonus')
    except BillingProductIsSingletone:
        pass


def reset_yateam(uid):
    try:
        user = User(uid)
    except errors.StorageInitUser:
        return

    if user.yateam_uid:
        yateam_uid = user.yateam_uid
        user.reset_yateam()
        _remove_yateam_dir(user.uid, yateam_uid)

    try:
        service_delete(uid, pid='yandex_staff')
    except IndexError:
        pass


def _create_yateam_dir(uid):
    fs = Bus(uid=uid)
    try:
        fs.mkdir(uid, Address.Make(uid, YATEAM_DIR_PATH).id)
    except errors.ResourceExist:
        log.info('/disk/Yandex Team (NDA) already exists for user %s' % uid)
        return


def _remove_yateam_dir(uid, yateam_uid):
    if YATEAM_REMOVE_FOLDER_DELAYED:
        start_time = next_datetime_with_hour(YATEAM_REMOVE_FOLDER_DELAYED_START_HOUR)
        stime = time.mktime(start_time.timetuple())
    else:
        stime = None
    mpfs_queue.put({'uid': uid, 'yateam_uid': yateam_uid}, 'move_yateam_folder_to_chief', deduplication_id='move_yateam_folder_to_chief__' + uid, stime=stime)


def check_yateam_subtree_access(user_or_uid, path, owner_uid):
    """Проверить доступ к ресурсу внутри ятимной папки для пути ``path`` и
    пользователя ``user_or_uid``.
    """
    if is_yateam_subtree(path) and user_has_nda_rights(owner_uid):
        show_nda = mpfs.engine.process.get_show_nda()

        # Кладун не прокидывает uid и не умеет определять яндексовые подсети. Кладун всегда передаёт show_nda=kladun.
        # Хоть мы теперь и не проверяем сети, этот флаг означает, что запрос из кладуна, поэтому uid проверять не надо.
        # https://st.yandex-team.ru/CHEMODAN-36201
        if show_nda == 'kladun':
            return

        if not user_has_nda_rights(user_or_uid):
            raise errors.ResourceNotFound('Tried to access YaTeam directory without permission')


def user_has_nda_rights(user_or_uid):
    """True означает, что пользователю можно пользоваться НДА папкой. В частности:
        * пользователю надо автоматически создавать и воссоздавать папку по пути YATEAM_DIR_PATH
        * у пользователя папку по пути YATEAM_DIR_PATH надо считать НДАшной
        * у пользователя есть права на простмотр публичных файлов из НДАшных папок 
    """
    if user_or_uid is None:
        return False

    user = user_or_uid
    if isinstance(user_or_uid, basestring):
        try:
            user = User(user_or_uid)
        except errors.StorageInitUser:
            user = None

    if user is None:
        return False

    if not user.is_yateam():
        return False

    userinfo = passport.userinfo(uid=user.uid)
    if not userinfo.get('is_2fa_enabled', False):
        return False

    return True


def update_user_yateam_status_async(uid):
    mpfs_queue.put({'uid': uid}, 'update_user_yateam_status')


def recreate_yateam_dir(uid):
    mpfs_queue.put({'uid': uid}, 'restore_yateam_folder', delay=YATEAM_RECREATE_FOLDER_DELAY)
