# -*- coding: utf-8 -*-

import time

import datetime
import yaml

from mpfs.common.errors import FailoverFileIsTooOld
from mpfs.common.util import to_json
from mpfs.config import settings
from mpfs.core.zookeeper.shortcuts import get_checksum, LAST_TIME_MODIFICATION
from mpfs.engine.process import get_default_log, get_error_log


default_log = get_default_log()
error_log = get_error_log()


def get_cached_settings():
    default_log.info('Loading settings from cache file, path %s' % settings.zookeeper['cache']['file_path'])
    try:
        with open(settings.zookeeper['cache']['file_path']) as f:
            read_data = yaml.load(f)
        document_time = read_data[LAST_TIME_MODIFICATION]
        settings_str = to_json(read_data[settings.zookeeper['settings_path']])
    except Exception:
        error_log.error('Failed read data from cache file', exc_info=True)
        raise

    cur_time = int(time.time())
    if cur_time - document_time > settings.zookeeper['cache']['maximum_age']:
        document_datetime = datetime.datetime.fromtimestamp(document_time)
        datetime_timedelta = datetime.datetime.fromtimestamp(cur_time) - document_datetime
        error_log.error('Failover setting file is too old. File creation date is %s, time delta is %s. '
                        'Use old settings' % (document_datetime, datetime_timedelta))
        raise FailoverFileIsTooOld()

    default_log.info('Successfully got settings from cache file, settings checksum is %s' %
                     get_checksum(read_data[settings.zookeeper['settings_path']]))
    return settings_str
