# -*- coding: utf-8 -*-
import click


def debug(message, verbose=False):
    if verbose:
        click.echo(message)


def validate_mpfs_settings(arcadia_settings, verbose=False):
    from mpfs.core.zookeeper.shortcuts import update_settings as mpfs_update_settings
    from mpfs.config.errors import ValidationConfigError
    try:
        mpfs_update_settings(arcadia_settings, strict=True)
    except ValidationConfigError as e:
        debug(e, verbose)
        return False
    debug('Settings valid', verbose)
    return True


def upload_settings_to_zookeeper(arcadia_settings, verbose=False):
    from mpfs.common.util import to_json
    from mpfs.config import settings
    from mpfs.core.zookeeper.manager import ZookeeperClientManager

    zk_client = None
    try:
        zk_client = ZookeeperClientManager.create_client()
        zk_client.start(timeout=settings.zookeeper['connection_timeout'])

        debug('Connected to Zookeeper', verbose)
        zookeeper_value = bytes(to_json(arcadia_settings))
        debug('Try to set `%s` with value: %s' % (settings.zookeeper['settings_path'], zookeeper_value), verbose)
        zk_client.set(settings.zookeeper['settings_path'], zookeeper_value)
    except:
        raise
    finally:
        if zk_client:
            zk_client.stop()
            zk_client.close()


@click.command('update_settings', help=u"Обновить настройки в Zookeeper из Аркадии")
@click.option("-u", "--user", help=u"Имя пользователя для доступа в Аркадию", required=True)
@click.option("--verbose", help=u"Отображать весь процесс", is_flag=True)
@click.option("--validate", help=u"Провалидировать об global_settings.yaml", is_flag=True)
@click.option("--dry-run", help=u"Не обновлять в Zookeeper (полезно для валидации и проверки связи)", is_flag=True)
def update_settings(user, verbose, validate, dry_run):
    import mpfs.engine.process
    mpfs.engine.process.setup_anyone_script()

    file_path = "disk/mpfs/apps/common/conf/mpfs/zookeeper_settings.yaml"
    settings_command = [
        'svn', 'cat',
        'svn+ssh://%s@arcadia-ro.yandex.ru/arc/trunk/arcadia/%s' % (user, file_path)
    ]
    import yaml
    from subprocess import check_output

    debug('Execute command: %s' % settings_command, verbose)
    arcadia_settings_raw = check_output(settings_command, shell=False)

    arcadia_settings = yaml.safe_load(arcadia_settings_raw)
    debug('Load settings: %s' % arcadia_settings, verbose)

    if validate:
        assert validate_mpfs_settings(arcadia_settings, verbose)

    if not dry_run:
        upload_settings_to_zookeeper(arcadia_settings, verbose)

    click.echo('All done')
