# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import posixpath
import socket

from kazoo.client import KazooClient
from kazoo.handlers.gevent import SequentialGeventHandler
from kazoo.handlers.threading import SequentialThreadingHandler

from mpfs.config import settings
from mpfs.core.services.conductor_service import ConductorService


class ZookeeperClientManager(object):
    clients = {}

    @classmethod
    def set_client(cls, obj, base_path=settings.zookeeper['base_path']):
        """Установить глобальный клиент для работы с Зукипером.

        :param obj: Объект клиента, экземпляр класса `kazoo.KazooClient`.
        :param base_path: префикс chroot клиента для разделения клиентов
        :return: Переданный объект клиента.
        """
        cls.clients[base_path] = obj
        return cls.clients[base_path]

    @classmethod
    def create_client(cls, handler=None, base_path=settings.zookeeper['base_path']):
        """Создает клиента для работы с Зукипером и устанавливает префиксный путь
        для всех последующих запросов."""
        environment = settings.get_environment()
        if handler is None:
            if settings.zookeeper['async_mode']:
                handler = SequentialGeventHandler()
            else:
                handler = SequentialThreadingHandler()

        zk = KazooClient(
            hosts=settings.zookeeper['hosts'],
            handler=handler,
            auth_data=[('sasl', '%s:%s' % (settings.zookeeper['user'], settings.zookeeper['password']))],
            timeout=settings.zookeeper['connection_timeout']
        )
        assert base_path.startswith(posixpath.sep)
        if environment == 'development':
            environment = socket.gethostname()
        elif environment == 'prestable':
            environment = 'production'

        zk.chroot = posixpath.join(base_path, environment)
        return zk

    @classmethod
    def get_client(cls, base_path=settings.zookeeper['base_path']):
        return cls.clients.get(base_path)
