# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import hashlib
import json
import time
from Queue import Queue

import yaml

from mpfs.common.util import from_json
from mpfs.config import settings
from mpfs.config.base import Config
from mpfs.config.errors import ValidationConfigError
from mpfs.core.zookeeper.manager import ZookeeperClientManager
from mpfs.core.zookeeper.sync_primitives import settings_queue, settings_updated_after_fork_event, empty_queue_exception
from mpfs.engine.process import get_default_log

LAST_TIME_MODIFICATION = 'last_time_modification'


default_log = get_default_log()


def unicode_repr(self, data):
    return self.represent_str(data.encode('utf-8'))


yaml.add_representer(unicode, unicode_repr)


def get_checksum(data):
    """
    Получить чексумму версии настроек.

    Чек сумму считаем таким образом, чтобы она была одинакова для конфигов, независимо от порядка перечисления клиентов
    и ключей.
    """
    return hashlib.sha256(json.dumps(data, sort_keys=True).encode('utf8')).hexdigest()


def update_settings(value, strict=False):
    if not isinstance(value, dict):
        if strict:
            raise ValidationConfigError('Malformed Zookeeper settings. Skip')
        else:
            default_log.warn('Malformed Zookeeper settings. Skip')
            return

    for key, v in value.items():
        current_setting = getattr(settings, key, None)
        if current_setting is not None:
            if isinstance(current_setting, dict):
                sub_settings = Config(current_setting)
                try:
                    sub_settings.patch(v, allow_new=True)
                except ValidationConfigError:
                    if strict:
                        raise
                    default_log.warn('Malformed Zookeeper setting `%s`. Skip' % key)
                    continue
            else:
                sub_settings = v
        else:
            if isinstance(v, dict):
                sub_settings = Config(v)
            else:
                sub_settings = v
        setattr(settings, key, sub_settings)


def write_settings_to_file(value):
    settings_str = from_json(value)
    data_to_write = {
        settings.zookeeper['settings_path']: settings_str,
        LAST_TIME_MODIFICATION: int(time.time())
    }

    with open(settings.zookeeper['cache']['file_path'], 'w') as f:
        yaml.dump(data_to_write, f, default_flow_style=False)
        default_log.info('Settings have been written to cache file successfully, path %s' %
                         settings.zookeeper['cache']['file_path'])


def prepare_zookeeper_settings(use_cache=True):
    from mpfs.common.util import from_json
    if settings.zookeeper['isolate_zk'] or not settings.zookeeper['apply_new_settings']:
        return
    from mpfs.core.zookeeper.manager import ZookeeperClientManager
    try:
        zk_client = ZookeeperClientManager.create_client()
        zk_client.start()
        value, stats = zk_client.get(settings.zookeeper['settings_path'])
        if use_cache:
            write_settings_to_file(value)
        zk_client.stop()
        default_log.info('Settings updated from Zookeeper on startup.')
    except Exception:
        if not use_cache:
            raise
        from mpfs.core.zookeeper.cache_reader import get_cached_settings
        value = get_cached_settings()
        default_log.info('Settings updated from cache on startup.')

    update_settings(from_json(value))


def watch_callback(event, worker_id='celery'):
    """Колбэк в воркере, вызывающийся при поступлении события изменения ноды.

    Получает последние данные ноды от которой пришло событие и снова на нее подписывается,
    указывая себя как получателя. Изменяет глобальную переменную в памяти воркера, храняшую *последние полученные* (но не факт что уже примененные)
    настройки доступов в API. Данные из нее будут перемещены в основную переменную в момент между запросами,
    чтобы не было ситуации когда пол запроса работает с одними настройками,
    а другая половина запроса - с другими.
    """
    path = event.path
    default_log.info('Got watch event for path "%s" on worker %s.' % (path, worker_id))
    _get_settings_from_zk(path, worker_id=worker_id)


def _get_settings_from_zk(path, worker_id='celery'):
    """Загружает сеттинги из ZK и устанавливает watch."""
    zk = ZookeeperClientManager.get_client()
    default_log.info('Getting settings from zk on worker %s.' % worker_id)
    value, stats = zk.get(path, watch=lambda x: watch_callback(x, worker_id=worker_id))
    settings_queue.put(value)
    default_log.info('Enqueued new settings on worker %s.' % worker_id)


def _update_latest_settings(value, worker_id='celery'):
    """Обновить переменную, хранящую последние полученные настройки доступов.

    :param value: Строка полученная в результате запроса данных из ноды зукипера.
    :type value: str | unicode
    :return: Чексумма данных.
    """
    data = from_json(value)
    checksum = get_checksum(data)
    from mpfs.core.zookeeper import env
    env.set_zookeeper_settings_latest(data)
    default_log.info('Worker %s updated latest settings to version %s.' % (worker_id, checksum))
    return checksum


def push_new_settings_to_queue(block=True, worker_id='celery'):
    try:
        value = settings_queue.get(block=block, timeout=None)
    except empty_queue_exception:
        if block:
            raise
        else:
            return
    else:
        if settings.zookeeper['apply_new_settings']:
            _update_latest_settings(value, worker_id=worker_id)

        if not settings_updated_after_fork_event.is_set():
            # выставляем флаг, что сеттинги обновлены и теперь можно
            # обслуживать запросы
            settings_updated_after_fork_event.set()
            default_log.info('One time settings updated event has been set on worker %s.' % worker_id)
