# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from threading import Thread

import time

from kazoo.protocol.states import KazooState

from mpfs.config import settings
from mpfs.core.zookeeper.cache_reader import get_cached_settings
from mpfs.core.zookeeper.manager import ZookeeperClientManager
from mpfs.core.zookeeper.shortcuts import _get_settings_from_zk
from mpfs.core.zookeeper.sync_primitives import settings_queue
from mpfs.engine.process import get_default_log, get_error_log

default_log = get_default_log()
error_log = get_error_log()


class ZookeeperConnectingThread(Thread):
    """
    Тред подключения к зукиперу
    """
    def __init__(self, base_path=settings.zookeeper['base_path']):
        super(ZookeeperConnectingThread, self).__init__()
        self.setDaemon(True)
        self.base_path = base_path

    def run(self):
        need_to_start = True
        while True:
            client = ZookeeperClientManager.get_client(base_path=self.base_path)
            try:
                if need_to_start:
                    client.start_async()
                    need_to_start = False
                    time.sleep(settings.zookeeper['connection_timeout'])
                if client.connected:
                    client.ensure_path(client.chroot)
                    default_log.info('Successfully connected to Zookeeper')
                    return
            except Exception:
                error_log.error(
                    'Error happened during starting Zookeeper connection, Retrying...', exc_info=True)
                need_to_start = True
                client.stop()
            else:
                error_log.error('Failed connect to Zookeeper in time. Waiting...')
                time.sleep(settings.zookeeper['connection_timeout'])


def preload_settings_and_start_zk_init_function(worker_id='celery'):
    """Прочитать файл кэша и запустить тред с подключением к зукиперу

    Если загрузить настройки из кэша не удалось, то будем ждать подключения к зукиперу. Выставляет глобальный объект
    клиента и запускает подключение к зукиперу. Также устанавливает наблюдение за нодой настроек сразу и после каждого
    события изменения.
    """
    if settings.zookeeper['isolate_zk'] or not settings.zookeeper['apply_new_settings']:
        return

    default_log.info('Worker %s initializes connection to Zookeeper.' % worker_id)

    def state_listener(state):
        # листенер который в случае потери связи будет переподписываться
        # на изменения и получать последние изменения
        default_log.info('Worker %s changed state to %s' % (worker_id, state))
        if state == KazooState.CONNECTED:
            ZookeeperClientManager.get_client().handler.spawn(
                _get_settings_from_zk, settings.zookeeper['settings_path'], worker_id=worker_id
            )

    try:
        value = get_cached_settings()
    except Exception:
        error_log.error('Can\'t use settings from cache, waiting for Zookeeper connection...')
    else:
        settings_queue.put(value)

    client = ZookeeperClientManager.create_client()
    ZookeeperClientManager.set_client(client)
    client.add_listener(listener=state_listener)
    t = ZookeeperConnectingThread()
    t.start()
